package Utils;
require Exporter;
use PDL;
use PDL::Math;
use PDL::NiceSlice;
use PDL::Core ':Internal';

use constant PI => 3.14159265358979;
use constant EC => 1.602176462e-19;     # elementary charge (C)
use constant EM => 9.10938188e-31;      # electron mass     (kg)
use constant PM => 1.67262158e-27;      # proton mass       (kg)
use constant KB => 1.3806503e-23;       # boltzmann number  (J/K)
use constant E0 => 8.854187816e-12;     # eps0 el. perm. vid (F/m)
use constant C  => 299792458.;          # speed of light (m/s)
use constant RE => 6378137.0;           # Earth radius (m)
use constant RM => 1737400.0;           # Moon radius  (m)
use constant X  => pdl[1,0,0];     #         
use constant Y  => pdl[0,1,0];     # versori 
use constant Z  => pdl[0,0,1];     #         

@ISA       = qw(Exporter);
@EXPORT    = qw(PI EC EM PM KB E0 C RE RM X Y Z 
                timp1 timp2 mmin mmax corel rotax sign 
                mw1d maxwell locMm readIGM mylinfit covar);


#------------ transform from HH:MM:SS.mmm to miliseconds ---------------
sub timp1 { 
  my $timp = shift;
  if ($timp =~ /^\d\d$/) {
    $timp .= ':00:00.000';
  } elsif ($timp =~ /^\d\d:\d\d$/) {
    $timp .= ':00.000';
  } elsif ($timp =~ /^\d\d:\d\d:\d\d$/) {
    $timp .= '.000';
  }
  $timp =~ /(\d\d):(\d\d):(\d\d)\.(\d\d\d)/;
  my $ore=$1; my $min=$2; my $sec=$3; my $msec=$4;
  my $mstot = ($sec+60.*$min+3600.*$ore)*1000.+$msec;
  return $mstot; 
}
#-----------------------------------------------------------------------


#------------ transform from miliseconds to HH:MM:SS.mmm ---------------
sub timp2 { 
  my $mstot  = shift;
  my $st     = floor($mstot/1000);   my $ms  = floor($mstot-$st*1000);
  my $mintot = floor($st/60);        my $s   = $st-$mintot*60;
  my $ore    = floor($mintot/60);    my $min = $mintot-$ore*60;
  if (length($ore) eq 1) {$ore = '0'.$ore;}
  if (length($min) eq 1) {$min = '0'.$min;}
  if (length($s)   eq 1) {$s   = '0'.$s;}
  if (length($ms)  eq 1) {$ms  = '00'.$ms;}
  if (length($ms)  eq 2) {$ms  = '0'.$ms;}
  return "$ore:$min:$s.$ms";
}
#-----------------------------------------------------------------------


#--------------------- minimum element for arrays ----------------------
sub mmin { 
  my $minv=1.e15;
  foreach $cv (@_) {
    if ($cv < $minv) {$minv=$cv;}
  }
  return $minv;
}
#-----------------------------------------------------------------------


#---------------------- maximum element for arrays ---------------------
sub mmax { 
  my $maxv=-1.e15;
  foreach $cv (@_) {
    if ($cv > $maxv) {$maxv=$cv;}
  }
  return $maxv;
}
#-----------------------------------------------------------------------


#------------------------ compute correlations -------------------------
sub corel { 
  my ($v1,$v2)  = @_;
  my $vv1=$v1-avg($v1);
  my $vv2=$v2-avg($v2);
  return sumover($vv1*$vv2)/sqrt(sumover($vv1**2)*sumover($vv2**2));
}
#-----------------------------------------------------------------------


#------------------------ compute covariance -------------------------
sub covar {
  my ($v1,$v2)  = @_;
  my $vv1=$v1-avg($v1);
  my $vv2=$v2-avg($v2);
  return sumover($vv1*$vv2)/nelem($vv1);
}
#-----------------------------------------------------------------------


#------- compute rotation matrix around vector $v with angle $a --------
sub rotax { 
  my ($v,$a)=@_;
  my ($c,$s,$t)=(cos($a),sin($a),1-cos($a));
  my $nv=norm($v);
  my ($vx,$vy,$vz)=($nv(0)->sclr,$nv(1)->sclr,$nv(2)->sclr);
  return pdl[[$t*$vx**2+$c,      $t*$vx*$vy+$s*$vz, $t*$vx*$vz-$s*$vy],
             [$t*$vx*$vy-$s*$vz, $t*$vy**2+$c,      $t*$vy*$vz+$s*$vx],
             [$t*$vx*$vz+$s*$vy, $t*$vy*$vz-$s*$vx, $t*$vz**2+$c     ]];
}
#-----------------------------------------------------------------------


#----------------------------------- sign ------------------------------
sub sign {
  my $arg=topdl(shift);
  $arg->where(approx $arg , 0, 1e-10) .=1;
  return $arg/abs($arg);
}
#-----------------------------------------------------------------------

#--------------------------- linear fit --------------------------------
sub mylinfit {
  my ($x,$y)=@_;
  my $N=nelem($x);
  my $a=(sum($x)*sum($y)/$N-sum($x*$y))/(sum($x)**2/$N-sum($x**2));
  my $b=sum($y-$a*$x)/$N;
  
  return ($a,$b); # y=ax+b
}
#-----------------------------------------------------------------------


#--------------------- Maxwell distribution function -------------------
sub mw1d {
  my ($m,$T,$v,$u)= @_;
  return sqrt($m/(2*PI*KB*$T))*exp(-$m*($v-$u)**2/(2*KB*$T));
}
#-----------------------------------------------------------------------


#------------- ansamblu statistic supus dtb maxwell --------------------
sub maxwell {
  my ($N,$box,$T,$m,$u)=@_;
  my $dtb=ones(6,$N);
  $dtb(:2).=random(3,$N)*$box;                             # position
  $dtb(3:).=$u+sqrt(2*KB*$T/$m)*erfi(1-2*random(3,$N));    # velocity
  my $T1=$m*sumover((($dtb(3:)-$u)->transpose)**2)/$N/KB;  # temperature (check)
  return ($dtb,$T1);
}
#-----------------------------------------------------------------------


#----------------- local Maxim and minim ------------------------------
sub locMm {
  my $a=shift;
  my @d=dims($a);
  my $dim=$d[0];
  my $idxM; my $nM=0;
  my $idxm; my $nm=0;
  my $mon; my $oldmon=-1;
  for (my $k=1;$k<$dim;$k++){
    if ($a($k-1) < $a($k)) {
      $mon=1;
      if ($oldmon ne 1){
        if ($nm eq 0) {
          $idxm=$k-1;
        } elsif ($nm eq 1){
          $idxm=pdl($idxm)->dummy(0)->glue(0,pdl($k-1)->dummy(0));
        } else {
          $idxm=$idxm->glue(0,pdl($k-1)->dummy(0));
        }
        $nm++;
      }
    }elsif ($a($k-1) > $a($k)){
      $mon=-1;
      if ($oldmon ne -1){
        if ($nM eq 0) {
          $idxM=$k-1;
        } elsif ($nM eq 1){
          $idxM=pdl($idxM)->dummy(0)->glue(0,pdl($k-1)->dummy(0));
        } else {
          $idxM=$idxM->glue(0,pdl($k-1)->dummy(0));
        }
        $nM++;
      }
    } else {$mon=0;}
    $oldmon=$mon;
  }
  return ($idxM,$idxm);
}
#-----------------------------------------------------------------------


#------------------ reads data in IGM format ---------------------------
sub readIGM{
  my ($file,$startt,$endd)=@_;
  my $start = timp1($startt)/1000.; 
  my $end = timp1($endd)/1000.;
  my @t_arr=();                                 
  my @bx_arr=(); my @by_arr=(); my @bz_arr=();   
  my @x_arr=();  my @y_arr=();  my @z_arr=();   
  open (FILE,"<$file");
  while (<FILE>) {
    if (/(\S+)T(\S+)(?:Z\s+|\s+)(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
      my $date=$1;
      my $time=$2;
      my $bx=$3; my $by=$4; my $bz=$5;
      my $x=$6;  my $y=$7;  my $z=$8;
      my $sec=timp1($time)/1000.;
      next if $sec < $start;
      last if $sec > $end;
      push @t_arr, $sec;
      push @bx_arr, $bx;  push @by_arr, $by; push @bz_arr, $bz; 
      push @x_arr, $x;    push @y_arr, $y;   push @z_arr, $z;
    } 
  }
  my $t=pdl(@t_arr);
  my $B=pdl(pdl(@bx_arr),pdl(@by_arr),pdl(@bz_arr));
  my $R=pdl(pdl(@x_arr),pdl(@y_arr),pdl(@z_arr));
  return (time => $t, field => $B, position => $R);
}
#-----------------------------------------------------------------------


##--------------- same as above but faster ?------------------------
#sub readIGMvec {
#  my $file = shift;
#  my ($Bx,$By,$Bz,$X,$Y,$Z,$tdarr)=
#      rcols $file,1,2,3,4,5,6,{PERLCOLS=>[0],INCLUDE=>'/^\d/'};
#  my @tarr=();
#  foreach $date_time (@$tdarr) {
#    $date_time=~/T(.+)Z/;
#    push @tarr, timp1($1);
#  }
#  return (t=>pdl(@tarr), Bx=>$Bx, By=>$By, Bz=>$Bz, x=>$X, y=>$Y, z=>$Z);
#}
##---------------------------------------------------------------


##-------------------- reads CEF data (Double Star...)-------------------
#sub parse_cef_DS {
#  my $cef_file=shift;
#  my $read=0;
#  my %data=();
#  my @record=();
#  open (CEF,"<$cef_file");
#  @record=();
#  while (<CEF>) {
#    $read=1 if /DATA_UNTIL/;
#    next unless $read;
#    next if /^\s*$/;
#    s/\s//g;
#    s/Z//g;
#    push @record,  split(',');
#    if (/^\$$/) {
#      chomp(@record);
#      split(/\s+/,@record);
#      #print "@record\n";
#      push @{$data{'date_time'}}, $record[0];
#      push @{$data{'Bx'}},        $record[6];
#      push @{$data{'By'}},        $record[7];
#      push @{$data{'Bz'}},        $record[8];
#      @record=();
#    }
#  }
#  close CEF;
#  return %data;
#}
##-----------------------------------------------------------------------

#------------------ reads CEF data (Cluster...)-----------
sub parse_cef_C {
  my $cef_file=shift;
  my $read=0;
  my %data=();
  my @record=();
  open (CEF,"<$cef_file");
  while (<CEF>) {
    $read=1 if /DATA_UNTIL/;
    next unless $read;
    next unless /^(\S+)\s+\$/;
    @record=split(',');
    push @{$data{'date_time'}}, $record[0];
    push @{$data{'Bx'}},        $record[2];
    push @{$data{'By'}},        $record[3];
    push @{$data{'Bz'}},        $record[4];
    push @{$data{'X'}},         $record[6];
    push @{$data{'Y'}},         $record[7];
    push @{$data{'Z'}},         $record[8];
  }
  close CEF;
  return %data;
}
#-----------------------------------------------------------------------


1;

__END__

=head1 NAME

Utils - some util constants and routines

=head1 Exported constants

=item *

C<PI>=3.14159265358979 : Scalar pi

=item *

C<X>=[1,0,0], C<Y>=[0,1,0], C<Z>=[0,0,1] : 3d piddles for cartesian versors

=head1 Exported functions

=item B<timp1>('HH:MM:SS.SSS') Inverse of B<time1>.

=item B<timp2>($miliseconds) Converts miliseconds in string HH:MM:SS.SSS

=item B<mmin>(@array) Returns minimum alement in @array.

=item B<mmax>(@array) Returns maximum element in @array.

=item B<corel>($v1,$v2) Returns cross-corellation between vectors (pdls)
$v1 and $v2.

=item B<rotax>($v,$a) Returns matrix (pdl) for rotation around vector (pdl) 
$v with angle (scalar) $a.

