#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Aug  6 18:37:46 2025

Module for Cluster FGM spin axis calibration.

Exported functions
------------------

cgfnew_to_fgmcal() :
    Writes the fgmcal file corrseponding to the input cfgnew file.

write_cfgnew() :
    Saves the cfgnew calibration parameters.

offsets_DS() :
    Derives offsets using the Davis-Smith method on a single data interval.

KDE() :
    Calculates the Kernel Density Estimate (KDE) for the input magnetic offset.

get_offsets() :
    Derives offsets using the Davis-Smith method on sliding windows.

range_max_B() :
    returns maximum field in range.

range_resol() :
    returns resolution in range.

@author: dragos
"""

import numpy as np
import pandas as pd
from scipy import stats
from scipy import optimize
import os
import re
import shutil
import subprocess
# from warnings import warn
from . import config as cfg


def cgfnew_to_fgmcal(cfgnewfile, debug=False):
    """
    Writes the fgmcal file corrseponding to the input cfgnew file.
    Uses the IDL script cfgnew2fgmcalPy. The output directory is
    cfg.PATHS['tmpDir'].

    Parameters
    ----------
    cfgnewfile : string
        Name of the cfgnew file.
    debug : boolean, optional

    Returns
    -------
    fgmcalfile : string
        Name of the fgmcal file.

    """

    tmpDir=cfg.PATHS['tmpDir']
    cfgDir=cfg.PATHS['FGMroot']+'cfg/'
    cfgFile=cfgDir+'cfgnew2fgmcalPy.cfg'
    lstFile=cfgDir+'cfgnew2fgmcal.lst'
    batchFile=cfg.PATHS['FGMroot']+'idl/cfgnew2fgmcalPy.batch'

    with open(cfgFile, 'w') as f:
        f.writelines([tmpDir+ '     # inputfile_path\n',
                      tmpDir+ '     # output_path4calfiles\n',
                      '0            # ADC\n',
                      'neu          # cfg or neu\n',
                      'V03          # not used ... (derived from filename)\n'])

    if tmpDir not in cfgnewfile:
        shutil.copy(cfgnewfile,tmpDir)

    with open(lstFile, 'w') as f:
        f.write(os.path.basename(cfgnewfile)+'\n')

    command="/usr/local/bin/idl -quiet -e '@"+batchFile+"'"
    result=subprocess.run(command, shell=True, capture_output=True, text=True)
    fgmcalfile=tmpDir+result.stdout.splitlines()[-1]

    if debug:
        print('cfgnew file: '+cfgnewfile)
        print('fgmcal file: '+fgmcalfile)

    return fgmcalfile


def write_cfgnew(pars, file=None, version=4):
    """
    Saves the cfgnew calibration parameters.

    Parameters
    ----------
    pars : dictionary
        Calibration parameters from load.cfgnew_pars().
    file : string, optional
        Output file. If None, the output file name is derived from the input file name.
        The default is None.
    version : integer or string, optional
        The version number of the output file. The default is 4.

    Returns
    -------
    file : string
        Output file name

    """
    if not file:
        oldFile=pars['file']
        oldV=re.findall('_([^_]*)\.cfgnew$', oldFile)[0]
        newV=version
        if isinstance(newV,int): newV='V'+str(newV).zfill(2)
        file=(cfg.PATHS['tmpDir']
              +os.path.basename(oldFile).replace('_'+oldV,'_'+newV))
    else:
        oldFile=''

    # change if GUI keys
    if 'Angle_xyz' in pars.keys():
        pars={cfg.FGMNAMES[_]: pars[_] for _ in cfg.FGMNAMES.keys()}
        pars['file']=oldFile

    with open(file, "w") as f:
        for name in cfg.FGMNAMES.values():
            if not (pars[name].ndim and pars[name].size): continue
            if not name[:6] == 'Offset': f.write('\n')
            for d in range(pars[name].shape[0]):
                f.write((name+'=')[:10]+''.join([f"{_:16.8f}" for _ in pars[name][d]])+'\n')
    return file


def _to_min_DS(x, *args):
    """
    Objective function used by offsets_DS().
    Adapted from bepiColombo.calibration.

    Parameters
    ----------
    x : float
        Z-offset

    *args :
        data: nx3 numpy array. data to fit
        squared: boolean. If true use squared magnetic field magnitude.
        mad: boolean. If true, the function returns the Median Absolute Deviation,
            otherwise the variance of the (squared) magnetic field magnitude.

    Returns
    -------
    float
        deviation from Alfvenic fluctuations.

    """
    offset=np.array([0,0,*x])
    B, mad, squared = args
    if squared:
        Babs=np.sum((B-offset)**2, axis=1)
    else:
        Babs=np.sqrt(np.sum((B-offset)**2, axis=1))
    if mad:
        return np.nanmedian(np.abs(Babs-np.nanmedian(Babs))) # MAD
        # return stats.median_abs_deviation(Babs, nan_policy='omit') # same result and time as above
    else:
        return np.nanvar(Babs)


def offsets_DS(data, ini=0, limit=50, mad=False, squared=False,
                      strict_limits=True, debug=False):
    """
    Derives offsets using the Davis-Smith method on a single data interval.
    Adapted from bepiColombo.calibration.

    Parameters
    ----------
    data : DataFrame
        The magnetic field vector.
    ini : float, optional
        The initial guess for the Z-offset. The default is 0.
    limit : float or None, optional
        The limit to be used as bounds for minimization. The default is 50.
    mad : boolean, optional
        If True, use the Median Absolute Deviation for minimization.
        MAD puts less weight on outliers.
        The default is False.
    squared : boolean, optional
        If True, use the squared magnetic field magnitude.
        The default is False (less weight on outliers).
    strict_limits : boolean, optional
        If True, the bounds will be (-limit, limit),
        othewise (init-limit, init+limit). The default is True.

    Returns
    -------
    float
        The estimated Z-offset.

    """
    if data.shape[0] < 10: return np.nan
    data=np.array(data)
    # if np.any(np.isnan(ini)): ini=[0,0,0]
    if np.isnan(ini): ini=0
    if limit is None:
        # bounds=tuple((None,None) for _ in (0,1,2))
        bounds=(None,None)
    else:
        if strict_limits:
            # bounds=tuple((-limit, limit) for _ in (0,1,2))
            bounds=(-limit, limit)
        else:
            # bounds=tuple((ini[_]-limit, ini[_]+limit) for _ in (0,1,2))
            bounds=(ini-limit, ini+limit)
    min_out = optimize.minimize(_to_min_DS, ini,
                                args=(data, mad, squared),
                                method="Powell", bounds=[bounds])
    if debug: print(min_out)
    if min_out.success: return min_out.x
    # return np.full(3,np.nan)
    return np.nan


def KDE(offset_in, guess={'Oz': 0},
             interval=(-30, 30), resol=.5, debug=False):
    """
    Calculates the Kernel Density Estimate (KDE) for the input magnetic offset.
    Adapted from bepiColombo.calibration.

    Parameters
    ----------
    offset_in : pandas.DataFrame
        Offset vectors from the Davis-Smith method.

    guess : dictionary, optional
        The initial estimation for the offsets.
        The default is {'Oz': 0}.

    interval : tuple, optional
        The offset values (nT) interval (around the guess values)
        for which the KDE is computed. The default is (-30, 30).

    resol : float, optional
        The resolution (nT) of the output KDE. The default is 0.5

    Returns
    -------
    off_KDE : pandas.DataFrame
        Probability density function

    """

    if any([x!=y for x, y in zip(offset_in.columns, guess.keys())]):
        raise ValueError('offset and guess names do not match!')

    names=guess.keys()

    eval_points= np.arange(interval[0]+min(guess.values()),
                           interval[1]+max(guess.values()), resol)
    eval_points=np.round(eval_points,4)

    # convert the guess to the nearest grid point
    for key in names:
        guess[key] = eval_points[np.abs(eval_points-guess[key]).argmin()]

    n_eval = eval_points.size

    off_KDE = pd.DataFrame(data={_: np.full(n_eval, np.nan) for _ in names},
                            index=np.round(eval_points, 4))

    off_KDE.index.name = 'offset'

    # if not offset_in.empty:
    if offset_in.shape[0]  > 10:
        for component in names:
            if debug: print('KDE on '+component+' ... ', end='', flush=True)
            offset = np.array(offset_in[[component]]).transpose()
            kde = stats.gaussian_kde(offset[np.isfinite(offset)])
            # off_KDE[component].loc[eval_points] = kde(eval_points)
            off_KDE.loc[eval_points, component] = kde(eval_points)
            if debug: print('done')

    return off_KDE


def get_offsets(B, wlen=np.timedelta64(360,'s'),
                slen=np.timedelta64(10, 's'), ini=0, limit=20,
                resol=0.025,
                mad=False, squared=False, debug=False):
    """
    Derives offsets using the Davis-Smith method on sliding windows. The final
    offsets are determined as a Kernel Density Estimate over the offsets resulted
    from all windows.
    Adapted from bepiColombo.calibration.

    Parameters
    ----------
    B : DataFrame
        The magnetic field vector.
    wlen : numpy.timedelta64, optional
        The sliding window length. The default is np.timedelta64(360,'s').
    slen : numpy.timedelta64, optional
        The sliding step. The default is np.timedelta64(10, 's').
    ini : float, optional
        Passed to offsets_DS(). The initial guess for the Z-offset.
        The default is 0.
    limit : float or None, optional
        Passed to offsets_DS(). The limit to be used as bounds for
        minimization. The default is 20.
    res: float
        Passed to KDE(). The resolution (nT) of the output KDE. The default is 0.025.
    mad : boolean, optional
        Passed to offsets_DS(). If True, use the Median Absolute Deviation
        for minimization. MAD puts less weight on outliers. Twice slower...
        The default is False.
    squared : boolean, optional
        Passed to offsets_DS(). If True, use the squared magnetic field
        magnitude. The default is False (less weight on outliers).
    debug : boolean, optional. The default is False.

    Returns
    -------
    dict
        {'Offset': float. The Z offset for the input interval.
         'Error': float. The error associated to Offsets.
                  Median Absolute Deviation if the mad keyword is set, or
                  Standard Deviation if not.
         'KDE': dataframe. KDE probability
         'off_DataFrame': dataframe. The sliding window offsets}

    """
    x0=ini

    offsets=pd.DataFrame(columns=['Oz'])
    for t in np.arange(B.index[0]+wlen/2,B.index[-1]-wlen/2,slen):
        Bwin=B[(t-wlen/2):(t+wlen/2)]

        offsets.loc[t]=offsets_DS(Bwin, ini=x0, limit=limit,
                                         mad=mad, squared=squared)
        x0=offsets.loc[t].values

        if debug: print("\r{}".format(t), end='', flush=True)
    if debug: print('')

    if mad:
        Err=stats.median_abs_deviation(offsets, nan_policy='omit')
        # Err={'O'+'xyz'[_]: Err[_] for _ in (0,1,2)}
        Err={'Oz': Err[0]}
    else:
        Err=dict(offsets.std())
    # Err={_: round(Err[_],2) for _ in Err.keys()}
    Err['Oz']=round(Err['Oz'],2)

    # guess=dict(zip(['O'+_ for _ in 'xyz'], ini))
    guess={'Oz':ini}

    kde=KDE(offsets, guess=guess, interval=(-limit,limit), resol=resol)

    # offsets to be subtracted to get the calibrated field:
    # (Bc-=np.array(list(Offsets.values())))
    Offsets=dict(kde.idxmax(skipna=True))

    offsets.attrs['window length']=wlen
    offsets.attrs['KDE resolution']=resol
    offsets.attrs['method']='median' if mad else 'mean'

    return {'Offset': Offsets['Oz'], 'Error': Err['Oz'],
            'KDE':kde, 'off_DataFrame': offsets}

def range_max_B(rng):
    " returns maximum field in range (nT)"
    return 2**(2*(int(rng)+1))

def range_resol(rng):
    " returns resolution in range (nT)"
    return 2**(2*(int(rng)-5)-1)
