#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Aug 20 07:49:48 2025

Derive the smoothed (second level) daily spin axis offsets
from the first level offsets.

For each Solar Wind Season (SWS, about 1 March +/- 4 months):
    * derive and store the yearly offset from the maximum KDE of the daily offsets.
    * derive and store the Median Absolute Deviation (MAD) of the daily offsets.
Fit a interpolated cubic spline to the yearly offsets with weights from MAD.


@author: dragos
"""
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from scipy import stats
from scipy.interpolate import UnivariateSpline
# from scipy.interpolate import make_splrep
import sys
from pathlib import Path
Cpath=str(Path(__file__).parent.parent)+'/modules/'
if not Cpath in sys.path: sys.path.append(Cpath)
import Cluster.calibration as cal
import Cluster.config as cfg


spacecraft='1234'

offD=pd.concat([pd.read_csv(cfg.PATHS['archiveDir']+'/calfiles/cfgnew/spin_axis_offset/C'
                           +_+'_offZ_daily.csv', index_col=0, parse_dates=[0])
               for _ in spacecraft], axis=1)
offD.columns=pd.MultiIndex.from_tuples([('C'+_n, _c) for _n in spacecraft
                                       for _c in list(dict.fromkeys(offD.columns))])
offsets={}
error={}
sgm={}
for sc in spacecraft:
    # center years around 1st March (the mid SW season)
    offY=pd.DataFrame(offD['C'+sc]['Oz']).resample('YE-SEP',
                origin=pd.Timestamp('2000-09-30')).apply(lambda x:
                     np.nan if len(x['Oz'].dropna()) <= 10 else
                     cal.KDE(x, interval=(-10,10),resol=.01).idxmax(skipna=True).values[0]).dropna()

    errY=pd.DataFrame(offD['C'+sc]['Oz']).resample('YE-SEP',
                origin=pd.Timestamp('2000-09-30')).apply(lambda x:
                     stats.median_abs_deviation(x['Oz'], nan_policy='omit')).dropna()

    stdY=pd.DataFrame(offD['C'+sc]['Oz']).resample('YE-SEP',
                origin=pd.Timestamp('2000-09-30')).std().dropna()

    # change labels to 1st of March
    offY.index-=pd.DateOffset(months=6, days=29)
    errY.index-=pd.DateOffset(months=6, days=29)
    stdY.index-=pd.DateOffset(months=6, days=29)
    offsets['C'+sc]=offY
    error['C'+sc]=errY
    sgm['C'+sc]=stdY.squeeze()

offY=pd.DataFrame(offsets)
errY=pd.DataFrame(error)
errY=errY.loc[offY.index]
stdY=pd.DataFrame(sgm)
stdY=stdY.loc[offY.index]



#%% interpolate

dbeg =cfg.FIRSTDAY#  np.datetime64(offY.index[0], 'D')#
dend =cfg.LASTDAY # np.datetime64(offY.index[-1], 'D')#
days=np.arange(dbeg, dend+np.timedelta64(1,'D'), np.timedelta64(1,'D'))
xs = days.astype(float)

offYinterp={}
for sc in spacecraft:

    x=np.array(offY.index).astype('datetime64[D]').astype(float)
    y=offY['C'+sc].to_numpy()
    w=1/errY['C'+sc].to_numpy()

    # smooth the endings
    x=np.array([x[0]-60,*x, x[-1]+60])
    y=np.array([y[0],*y,y[-1]])
    w=np.array([10,*w,10])

    spl = UnivariateSpline(x, y, w=w, s=.25, ext='const')
    offYinterp['C'+sc]=pd.Series(data=spl(xs), index=days)

offYinterp=pd.DataFrame(offYinterp)



#%% plot

# merge errors in the yearly offset dataframe
offY.columns=pd.MultiIndex.from_tuples([(_, 'Oz') for _ in offY.columns])
errY.columns=pd.MultiIndex.from_tuples([(_, 'MAD') for _ in errY.columns])
stdY.columns=pd.MultiIndex.from_tuples([(_, 'sigma') for _ in stdY.columns])
offY=pd.concat([offY,errY,stdY], axis=1)
offY=offY[[('C'+_sc, _) for _sc in spacecraft for _ in ('Oz', 'MAD', 'sigma')]].dropna()

years=np.array([np.datetime64(str(_.year)+'-01-01') for _ in offY.index.astype(object)])
row=1
ax=list(range(5))
ax[0]=None
for sc in spacecraft:
    ax[row]=plt.subplot(len(spacecraft), 1, row, sharex=ax[row-1],
                        # sharey=ax[row-1],
                        ylim=(-2.2, 0.7),
                        )
    offYinterp['C'+sc].plot(label= 'spline fit', color='tab:green')
    plt.autoscale(False)

    offD['C'+sc].Oz.plot(marker='.', linestyle='', alpha=0.2, label='daily', color='tab:blue')

    # fix bug in pandas plotting (otherwise x-axis is truncated to integer years)
    offY.index+=np.timedelta64(1,'s')
    offY['C'+sc]['Oz'].plot(yerr=offY['C'+sc]['MAD'], color='tab:orange',
                      marker='o', linestyle='', alpha=0.9, label='yearly',
                      ylabel=r'$\Delta O_\text{spin}$'+' (nT)')
    offY.index-=np.timedelta64(1,'s')

    for year in zip(years[::2], years[1::2]):
        plt.axvspan(year[0], year[1] , alpha=0.05, color='black')

    plt.text(0.03, 0.1, 'C'+sc,transform=ax[row].transAxes,
             bbox=dict(facecolor='white', alpha=0.2))
    plt.legend(loc='upper right')
    row+=1

plt.subplots_adjust(hspace=0)

#%% save plot

fig=ax[1].get_figure()
page_size=(16.5, 11.7) # landscape a3 size in inches...
plot_dir=cfg.PATHS['tmpDir']+'cfgnewPlots/'
fname=plot_dir+'spinAxisOffsets.pdf'
fig.set_figwidth(page_size[0])
fig.set_figheight(page_size[1])
fig.savefig(fname)
plt.close(fig)
print('saved: '+fname)

#%% save data
for sc in spacecraft:

    file=cfg.PATHS['tmpDir']+'C'+sc+'_offZ_yearly.csv'
    with open(file, 'w') as f:
        offY['C'+sc].to_csv(f, index_label='date',# header=['Oz'],
                            date_format='%Y-%m-%d', float_format='%10.3f')
    print('offsets saved to: '+file)

    file=cfg.PATHS['tmpDir']+'C'+sc+'_offZ_yearly_interpolated.csv'
    with open(file, 'w') as f:
        offYinterp['C'+sc].to_csv(f, index_label='date', header=['Oz'],
                                  date_format='%Y-%m-%d', float_format='%10.3f')
    print('offsets saved to: '+file)
