#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Aug  6 09:16:51 2025

Compute the (first level) daily Spin Axis Offsets (SAO)
using the Davis-Smith method

For each day with Solar Wind Intervals (SWI):
    * For each SWI in the day:
        . get B calibrated with last (<V03) parameters and etalon SAO
          in the spin aligned spacecraft reference (SR)
        . get the SAO for the SWI B using sliding windows:
            - for each window determine the offset using the Davis-Smith method.
            - the offset for the SWI is gived by the maximum KDE of the DS offsets.
    * compute the daily offset as a weighted (sigma and interval length) average.

@author: dragos
"""

import numpy as np
import pandas as pd
import ceflib as cef
from scipy import stats
from warnings import warn
import os
import sys
from pathlib import Path
Cpath=str(Path(__file__).parent.parent)+'/modules/'
if not Cpath in sys.path: sys.path.append(Cpath)
import Cluster.load as load
import Cluster.calibration as cal
import Cluster.config as cfg


dbeg = cfg.FIRSTDAY#'2002-01-01'#'2001-01-01'#'2004-04-01'#'2016-01-02'#
dend = cfg.LASTDAY#'2004-05-31'#'2001-01-15'#'2018-12-31'#

sc='4'
mode='n' # normal science
rng='2'

etalonDay=cfg.ETALONDAY
outVers='V06' # vers of intermediate calfiles with constant etalon SA offset
saveIntermediate=True
debug=True


dbeg=np.datetime64(dbeg, 'D')
dend=np.datetime64(dend, 'D')

# get etalon offset
cfgnewFile,V = load.cfgnew_file_name(day=etalonDay, spacecraft=sc, debug=True,
                                         version='last', maxV=3, archive=True)
cfgnewPars=load.cfgnew_pars(cfgnewFile, GUInames=True)
Ozx=cfgnewPars['offset_sc'][0,0].copy()

# get the solar wind intervals
cef.read(cfg.PATHS['GRMBdir']+'C'+str(sc)+'_GRMB.cef')
regions=pd.DataFrame(zip(cef.var('location_label'),
                         cef.datetime64('time_tags')[:,0],
                         cef.datetime64('time_tags')[:,1]),
                     columns=['label','begin', 'end'])
SWintervals=regions[regions['label']== 'OUT/SWF'][['begin', 'end']]
SWintervals['day']=SWintervals['begin'].map(lambda x: np.datetime64(x, 'D'))
SWintervals=SWintervals[(SWintervals['day'] >= dbeg) & (SWintervals['day'] <= dend)]


days=np.sort(list(set(SWintervals['day'])))
off=pd.DataFrame(columns=['Oz', 'error', 'duration'], index=days)
off.index.names = ['date']

year=days[0].year

for day in days:#[350:360]:
    # if debug: print('--- '+str(day)+' ---')
    if not day.year == year:
        if saveIntermediate:
            print("\n\n==== saving year "+ str(year)+" ====\n\n")
            interval='to_'+str(day.year)
            with open(cfg.PATHS['tmpDir']+'C'+sc+'_offZ_daily_'+interval
                      +'_'+outVers+'.csv', 'w') as f:
                off.astype(float).dropna().to_csv(f,date_format='%Y-%m-%d',
                                                  float_format='%10.3f')
        year=day.year

    off_=[]
    err_=[]
    min_=0
    weights=[]
    SWint=SWintervals[SWintervals['day']==day]
    if SWint.empty: print('No SW on '+str(day))
    for SWi in SWint[['begin','end']].to_numpy():
        db, de = SWi

        # load B calibrated with the etalon offset
        print('--- B ---')
        cfgnewFile,V = load.cfgnew_file_name(day=day, spacecraft=sc, debug=debug,
                                         version='last', maxV=3, archive=True)
        if not cfgnewFile: continue
        cfgnewPars=load.cfgnew_pars(cfgnewFile, GUInames=True)
        cfgnewPars['offset_sc'][0,0]=Ozx
        newCfgnewFile=cal.write_cfgnew(cfgnewPars, version=outVers)
        newFgmcalFile=cal.cgfnew_to_fgmcal(cfgnewfile=newCfgnewFile, debug=debug)
        B=load.mag(dbeg=db, dend=de, spacecraft=sc, rng=2, coord='sr',
                   debug=True, fgmcalfile=newFgmcalFile)

        # cleanup
        newR7File=newFgmcalFile[:-11]+'range7.fgmcal'
        for tmpFile in (newCfgnewFile, newFgmcalFile, newR7File):
            if tmpFile[:19] == cfg.PATHS['tmpDir']:
                try: os.remove(tmpFile)
                except: pass

        if B.empty: continue

        # calibrate (Davis-Smith)
        try:
            offset=cal.get_offsets(B, limit=10, mad=False, debug=True)
        except:
            warn('\n\n###### cal.get_offsets failed for the interval ['
                 +str(db)+','+str(de)+'] ######\n\n')
            continue
        if (np.isclose(offset['Error'],0)
            or np.isnan(offset['Offset'])
            or np.isnan(offset['Error'])): continue

        # gather SW intervals values for current day
        off_+=[offset['Offset']]
        err_+=[offset['Error']]
        length=np.timedelta64(de-db,'m')/np.timedelta64(1,'m')
        min_+=length
        weights+=[np.log(length)/offset['Error']]

    if not len(off_): continue

    # final daily difference from the etalon SA offset
    off.loc[day]=[np.average(off_, weights=weights),
                  np.average(err_),
                  round(min_)]

off=off.astype(float)
off=off.dropna()

off.plot(subplots=True, marker='o', linestyle='', alpha=0.1)

#%% save daily and monthly SA offsets

interval=(np.datetime_as_string(dbeg, unit='D')+'_to_'
          +np.datetime_as_string(dend, unit='D'))

# daily offsets
file=cfg.PATHS['tmpDir']+'C'+sc+'_offZ_daily_'+interval+'_'+outVers+'.csv'
with open(file, 'w') as f:
    off.to_csv(f,date_format='%Y-%m-%d', float_format='%10.3f')

# monthly offsets
offM_=off.resample('ME',label='right').apply(lambda x:
                                             np.nan if len(x['Oz'].dropna()) <= 10 else
                                             cal.KDE(x.dropna(), interval=(-10,10),
                                                 resol=.01).idxmax(skipna=True).values[0])
errM=off.resample('ME',label='right').apply(lambda x:
                     stats.median_abs_deviation(x['Oz'], nan_policy='omit'))
offM=pd.concat([offM_,errM], axis=1)
offM.columns=['Oz','Error']

file=cfg.PATHS['tmpDir']+'C'+sc+'_offZ_monthly_'+interval+'_'+outVers+'.csv'
with open(file, 'w') as f:
    offM.dropna().to_csv(f,date_format='%Y-%m', float_format='%10.3f')





# to read the saved file
# off=pd.read_csv(cfg.PATHS['tmpDir']+'C'+sc+'_offZ_daily.csv', index_col=0, parse_dates=[0])
# offD=pd.concat([pd.read_csv(cfg.PATHS['archiveDir']+'/calfiles/cfgnew/spin_axis_offset/C'
#                            +_+'_offZ_daily.csv', index_col=0, parse_dates=[0])
#                for _ in '1234'], axis=1)
# offD.columns=pd.MultiIndex.from_tuples([('C'+_n, _c) for _n in '1234'
#                                        for _c in list(dict.fromkeys(offD.columns))])
