#!/usr/bin/perl -w
use Getopt::Long qw(:config no_ignore_case bundling);
use lib $ENV{HOME}."/perl/modules";
use PDL;
use PDL::NiceSlice;
use PDL::FFT qw/:Func/;
use Utils;
use Pod::Usage;
use FGMperl;

$yy      = substr((localtime())[5],1);
$scstr   = '1 2 3 4';
$day     = '??';
$log     = 1;
$ICL     = 0;
$inboard = '';

$width=300; # scanning window width (s)
$step=60;   # scanning step (s)
$navg=15;   # number of points used for smoothing total power before finding the min

$low1=0.01;     $high1=0.2;     # band below sc spin (Hz)
$low2=0.26;     $high2=0.45;    # band above sc spin (Hz)

$me="$0 @ARGV";

$ENV{FGMVERSION} ? ($version=$ENV{FGMVERSION}) : ($version=3);

GetOptions ('h|help|?+'      => \$help,
            'y|year=i'       => \$yy,
            'm|month=i'      => \$mm,    
            's|spacecraft:s' => \$scstr,
            'd|day:s'        => \$day,
            'c|calday:s'     => \$calday,
            'v|version:s'    => \$version,
            'I|ICL'          => \$ICL,
            'a|append'       => \$append,
            'l|log!'         => \$log);

pod2usage( -verbose => $help, -noperldoc => 1) if $help;

die 'Please use the --month option' unless $mm;

$calm=$mm-1; 
if ($calm) {$caly=$yy;} else {$caly=$yy-1; $calm=12;}  

if (length($yy)      eq 1) {$yy   ='0'.$yy;}
if (length($mm)      eq 1) {$mm   ='0'.$mm;} 
if (length($day)     eq 1) {$day  ='0'.$day;}
if (length($version) eq 1) {$version ='0'.$version}
if (length($calm)    eq 1) {$calm    ='0'.$calm;}  
if (length($caly)    eq 1) {$caly    ='0'.$caly;}  

# cal params from day 28 of last month if not in command line 
$calday = $caly.$calm.'28' unless $calday; 

$uncal_file="uncal_$yy$mm$day.txt"; $uncal_file=~s/\?//g;

@scarr=split ' ', $scstr ;

$ENV{FGMROOT}  = '/home/FGM/'                unless $ENV{FGMROOT};
$ENV{SATTPATH} = "$ENV{FGMROOT}/log/atorb/"  unless $ENV{SATTPATH};
$ENV{FGMPATH}  = "$ENV{FGMROOT}/data/dcalf/" unless $ENV{FGMPATH};
$ENV{ORBITPATH} = "$ENV{FGMROOT}/log/atorb/" unless $ENV{ORBITPATH};
$calfpath = "$ENV{FGMPATH}";
$exepath  = "$ENV{FGMROOT}/bin";

#if ($ICL) {
#  $ipath = $ENV{FGMROOT}."/data/raw/ICL/$yy"."_$mm";
#} else {
#  $ipath = $ENV{FGMROOT}."/data/raw/ESTEC/";
#}

$logfiledir = "$ENV{FGMROOT}/log/cd_log/$yy"."_$mm";
$opath      = "$ENV{FGMROOT}/cfg/";

open  LOG, ">>$ENV{FGMROOT}/log/dailycal/dailycal_$yy$mm.log";
print LOG localtime()." $ENV{USER} $me\n" if $log;
close LOG;

if ($append) {
  open UNCAL, ">>$opath/$uncal_file";
} else {
  open UNCAL, ">$opath/$uncal_file";
  print UNCAL "sc1 2   3   4      range      day   hbeg   mbeg
---------------------------------------\n";
}

foreach $sc (@scarr) {
  @logfiles=sort(glob("$logfiledir/C$sc"."_$yy$mm$day"."_B.{NS,BSNS,BS}log"));
  foreach $logfile (@logfiles){ # loop over days with logs
    next unless -e $logfile;
     $lfregex="_$yy$mm".'(\d\d)_B\.(NS|BSNS|BS)log';
     $logfile=~/$lfregex/;
     $dd=$1;$bs=$2;
        
    next if ((($bs eq 'NS') or ($bs eq 'BS')) 
             and -e "$logfiledir/C$sc"."_$yy$mm$dd"."_B.BSNSlog");           
    
    $calfile    = "C$sc"."_20$calday".'_V'.$version.'.fgmcal'; 

    die "$calfile not found, please use the calday option" unless -e "$calfpath/$calfile";
    
#    if ($ICL) {
#      if ($bs eq 'NS') {
#        $rawf = "$ipath/C$sc"."_$yy$mm$dd".'_B.NS';
#      } elsif ($bs eq 'BSNS') {
#        $rawf = "$ipath/C$sc"."_$yy$mm$dd".'_B.?S';
#      } elsif ($bs eq 'BS') {
#        next if -e "$logfiledir/C$sc"."_$yy$mm$dd"."_B.BSNSlog";
#        $rawf = "$ipath/C$sc"."_$yy$mm$dd".'_B.BS';
#      }
#    } else {
#      if ($bs eq 'NS') {
#        $rawf="$ipath/cluster$sc/?sd_$sc/$yy$mm$dd".'fn.?a'."$sc";
#      } elsif ($bs eq 'BSNS') {
#        $rawf = "$ipath/cluster$sc/?sd_$sc/$yy$mm$dd".'f?.?a'."$sc";
#      } elsif ($bs eq 'BS') {
#        next if -e "$logfiledir/C$sc"."_$yy$mm$dd"."_B.BSNSlog";
#        $rawf="$ipath/cluster$sc/?sd_$sc/$yy$mm$dd".'fb.?a'."$sc";
#      }
#    }
#        
#    @files=glob("$rawf*");@mrgfiles=();  # remove empty files 
#    foreach $mf (@files) {                                    
#      push @mrgfiles, $mf unless -z $mf;  
#    }                                                         
    
    
    
    @mrgfiles=selectraw(ICL=>$ICL,sc=>$sc,bs=>$bs,yy=>$yy,mm=>$mm,dd=>$dd,dv=>'last');
    
    
    
    
    
    #next if $#mrgfiles == -1; # No data
    
    open LOGF, "<$logfile";
   $range=0; $end=0;
   @Begin=@End=();
    while (<LOGF>) {
      if (/R\s+(\d)>FGM\s+sensor\s+range\s+(\d)/) {
        $range=$1;
        die 'something wrong with range identification' unless ($range eq $2);
        #print "$range \n";
      } 
      
      next unless $range eq 2;
      
      if (/B\s+\d{4}-\d\d-\d\dT(\d\d:\d\d:\d\d\.\d\d\d)Z/) {
        $begin=$1;
        if ($end eq 0) { # first r2 begin of the day
          $begin='00:00:00.000' if $begin =~/23:59:/;
          push @Begin, $begin;
        } else {
          $gap=timp1($begin)-timp1($end);
          if ($gap > 10000) { # 10 s
            push @Begin, $begin;
            print ">--\n";
          } else {
            pop @End;  
          }
          print "gap: $gap\n";
        }        
      } elsif (/E\s+\d{4}-\d\d-\d\dT(\d\d:\d\d:\d\d\.\d\d\d)Z/) {       
        $end=$1;
        push @End, $end;
        print "$begin  $end \n";
      }
    }
    close LOGF;
    
    die "wrong..." if $#Begin ne $#End;
    
    next if $#Begin == -1; # No r2 data

    @min_times=@min_powers=();    
    for ($n=0; $n<=$#Begin; $n++)  {  #loop over the found r2 intervals
      $length= timp1($End[$n])-timp1($Begin[$n]);
      next if $length < 1800000; # 30 min
      $btime="20$yy-$mm-$dd".'T'.$Begin[$n].'Z';   
      $etime="20$yy-$mm-$dd".'T'.$End[$n].'Z';     #

      open(FGMPIPE, "$exepath/ddsmrg @mrgfiles                |          
                     $exepath/fgmtel     2>/dev/null          |       
                     $exepath/fgmcut -b $btime -e $etime      |    
                     $exepath/fgmcal -c $calfpath/$calfile -i |    
                     $exepath/fgmhrt -s scs                   | 
                     $exepath/fgmav  -s 1                     |    
                     $exepath/fgmpos                          |         
                     $exepath/igmvec -m -t1                   |");
      ($t,$By,$Bz)=rcols *FGMPIPE,0,2,3;
      close FGMPIPE;
    
      #### wcols $t,$By,$By,$Bz,'xxx'.$n.'.txt';
 
      $B=sqrt($By**2+$Bz**2);
      
      next if nelem($B) < 100;

      ($tmin,$pmin)=min_wp($B,$t,$low1,$high1,$low2,$high2,$width,$step,$navg);
    
      $tstr=timp2($tmin*1000); 
      print "found minimum $pmin at $tstr $tmin s\n";
      push @min_times, $tmin; push @min_powers, $pmin;
          
    } # end loop intervals
    
    $best_idx=minimum_ind(pdl(@min_powers));
    $best_t=$min_times[$best_idx];
    
    timp2($best_t*1000-150000)=~/(\d\d):(\d\d):\d\d\./;
    $uncal_start=$1."   ".$2;
    @uncal_sc=(0,0,0,0); $uncal_sc[$sc-1]=1;
    $uncal_line="$uncal_sc[0]   $uncal_sc[1]   $uncal_sc[2]   $uncal_sc[3]        2        $dd   $uncal_start";
    print UNCAL "$uncal_line\n";
    
    $tstr=timp2($best_t*1000); 
    print "best time for sc$sc on day $dd: $tstr  -> $uncal_start\n\n";
    
  } # end loop day
} # end loop sc

close UNCAL;
print "r2 uncal times saved in $opath/$uncal_file\n";

sub min_wp {
  my ($BB,$tt,$low1,$high1,$low2,$high2,$width,$step,$navg)=@_;
  my $B=$BB->copy;
  my $t=$tt->copy;
  my $N=nelem($t); die "too few points" if $N < $width;
  my $nstep=floor(($N-$width)/$step+1)->sclr; # we might miss the last points
  my ($pow,$frq);
  my $dynspc=zeroes($nstep,$width/2+1);
  for (my $ns=0;$ns<=$nstep-1;$ns++) {
    my $begin=$ns*$step;
    my $end=$ns*$step+$width-1;
    ($pow,$frq)=myfftpow($t($begin:$end),$B($begin:$end),2);
    $dynspc($ns,).=$pow->transpose;
  }
  my $tdyn=$t($width/2+xvals($nstep)*$step); # the time vector
  
  my $idxfrq1=which((($frq>$low1) & ($frq<$high1)) | # keep only two frq bands
                    (($frq>$low2) & ($frq<$high2)));
  
  my $Tpow=sumover($dynspc(,$idxfrq1)->xchg(0,1))->flat;
  my $Tpowavg=smooth($Tpow,$navg,1);
  my ($minT,$maxT,$minTi,$maxTi)=minmaximum($Tpowavg($navg:-$navg-1));
  my $t0=$tdyn($minTi+$navg)->sclr;
  my $p0=$minT->sclr;
  return ($t0,$p0);
}


sub myfftpow {
  my ($tt,$yy,$navg)=@_;
  my $y=$yy->copy;
  my $t=$tt->copy;

  my $n=nelem($t);
  my $dt=$t(1)-$t(0);
  
  if ($n%2) { # odd
    $f=$t->xlinvals(-($n/2.-0.5)/$n/$dt,($n/2.-0.5)/$n/$dt)->rotate(-($n-1)/2);
  } else { # even
    $f=$t->xlinvals(-($n/2.-1.)/$n/$dt,1./2./$dt)->rotate(-($n/2 -1));
  }
        
  my ($a,$b)=mylinfit($t,$y);    # extract linear trend
  $y-=$a*$t+$b;
  my $hanning=1-cos($y->xlinvals(0,2*PI)); # Hanning window
  $y*=$hanning;
  my $norm=sqrt(sum($hanning**2)); # compensate for windowing (CHECK THIS!!)
  my $yr=$y->copy; 
  my $yi=zeroes($n);
  fft($yr,$yi); 
  my $yfft=($yr->glue(1,$yi))->transpose;
  $yfft/=$norm; 

  $f=$f(:floor($n/2)); $yfft=$yfft(,:floor($n/2));

  my $pow=sqrt($yfft(0,)**2+$yfft(1,)**2)->flat;
  my $powavg=smooth($pow,$navg,0); # Reduce variance in frequency domain
         
  return ($powavg,$f);
}

sub smooth { # (weighted) average over $navg-order neighbors
  my ($yy,$navg,$w)=@_;
  my $y=$yy->copy;
  my $n=nelem($y);
  my $weight=ones(2*$navg+1);
  if ($w==1) {$weight=1-cos(ones(2*$navg+1)->xlinvals(0,2*PI));}
  my $tmp=zeroes($n + 2*$navg);  
  for (my $p=-$navg; $p<=$navg; $p++) {
    $tmp($navg+$p:$navg+$p+$n-1)+=$y/(2*$navg+1)*$weight($navg+$p);
  }
  my $yavg=$tmp($navg:$n+$navg-1);
  return $yavg # first $navg and last $navg points are not correct!
}


#-------------------------------------------------------------------------

1;

__END__

=head1 NAME

findr2.pl - Finds optimum range 2 daily calibration intervals.

=head1 SYNOPSIS

B<findr2.pl> B<--month> I<month> [B<--year> I<year>] [B<--day> I<day>} 
[B<--spacecraft> I<spacecraft>] [B<--ICL>] [B<--calday> I<yymmdd>]
[B<--version> I<version>] [B<--(no)log>] [B<--help>]

=head1 DESCRIPTION

This script finds the optimum intervals for range 2 calibration. For each day
the range 2 intervals longer than 30 minutes and with no data gaps longer than 
10s are identified from the F<cd_log> files. Using calibration parameters known
from the previous month (or from a day given in the command line) the Fourier 
power spectrum for these intervals is computed over a sliding window with a 
width of 5 minutes. The time corresponding to the minimum of the total wave
power in the frequency bands bellow (10 to 200 mHz) and above (260 to 450 mHz)
the spin frequency is selected and saved in the F<uncal> file.

=head1 OPTIONS

=over 4

=item B<-m> I<month>, B<--month> I<month> 

The month. One or two digits. Mandatory argument.

=item B<-y> I<year>, B<--year> I<year>

The year. One or two digits. Default is the current year.

=item B<-d> I<day>, B<--day> I<days>

The day. One or two digits. If absent, all the days with non-empty F<cd_log> 
files in the month are processed.

=item B<-s> I<spacecraft>, B<--spacecraft> I<spacecraft>

The spacecraft number (1-4) or list of spacecraft e.g. B<-s> C<'1 3 4'>. 
Defaults to all spacecraft. 

=item B<-I> ,B<--ICL>

Use the Imperial College London raw data. Default is to use the ESTEC raw data.

=item B<-c> I<yymmdd>, B<--calday> I<yymmdd>

The date from wich the calibration parameters are used to compute the spectra. 
If absent, the script will use day 28 from the previous month.

=item B<-v> I<version>, B<--version> I<version>

Version of the calibration files. If this option is not given then the
environment variable FGMVERSION is used. If FGMVERSION is not set, then the
default version is 3.

=item B<-a>, B<--append>

Append to the output file deleting old content. Default is to delete first the old content.

=item B<-l>, B<--log>

Record the run to dailycal log file. Default is enabled. 
Can be disbled using the B<--nolog> option.

=item B<-h>, B<-?>, B<--help>

Prints a brief help message.

=back

=head1 ENVIRONMENT

=over 4

=item FGMROOT

Root for the FGM calibration directory structure. Default to F</home/FGM/> if
not set.

=item FGMPATH

Path to calibration files (*.fgmcal and *.cfgnew). Default to
F<$FGMROOT/data/dcalf/> if not set.

=item SATTPATH

Path to orbit parameters files. Default to F<$FGMROOT/log/atorb/> if not set.

=back

=head1 FILES

F<$FGMROOT/cfg/uncal_yymm.txt> - output file for the optimum range 2 calibration
intervals. This file is used by B<mkuncal.pl> to produce the F<uncal> files.

F<$FGMROOT/data/raw/ICL/$yy_$mm/> - Imperial input path

F<$FGMROOT/data/raw/ESTEC/cluster$sc/[n|b]sd_$sc/> - ESTEC (default) input path

F<$FGMROOT/data/uncal/$yy_$mm/> - output path

F<C$s_$yy$mm$dd_B.[B|N]S> - Imperial input files

F<$yy$mm$dd.f[n|b].?a$sc> - ESTEC (default) input files

F<$FGMROOT/log/dailycal/dailycal_$yy$mm.log> - Dailycal log file.

=head1 DEPENDENCES

This script uses the following:

 ddsmrg,
 fgmtel,
 fgmcut,
 fgmcal,
 fgmhrt,
 fgmav,
 fgmpos,
 igmvec.

=head1 AUTHOR

Dragos Constantinescu <d.constantinescu@tu-bs.de>

=cut

























