#!/usr/bin/perl -w
use Getopt::Long qw(:config no_ignore_case bundling);
use lib $ENV{HOME}."/perl/modules";
use Pod::Usage;
use FGMperl;

$yy       = substr((localtime())[5],1);
$scstr    = '1 2 3 4';
$help     = 0;
$coord    = 'gse';
$res      = 'dacq'; # data acquisition rate UNCHANGED
$tbegin   = '';
$tend     = '';
$time_fmt = 0;
$outdir   = '';
$ICL      = 0;
$archived = 1;
$info     = 1;
$fgmcalfile='';
$name='';
$fgmtelOption='';
$fgmcalOption='';

$ENV{FGMVERSION} ? ($version=$ENV{FGMVERSION}) : ($version=3);

GetOptions ('h|help|?+'       => \$help, 
            'y|year:i'        => \$yy,
            'm|month:i'       => \$mm,
            'd|days:s'        => \$dstr,
            'b|begin:s'       => \$tbegin,
            'e|end:s'         => \$tend,
            's|spacecraft:s'  => \$scstr,
            'c|coordinates:s' => \$coord,
            'r|resolution:s'  => \$res,
            'o|outdir:s'      => \$outdir,
            'n|name:s'        => \$name,
            't|timeformat:i'  => \$time_fmt,   
            'f|fgmcalfile:s'  => \$fgmcalfile,    
            'I|ICL'           => \$ICL,
            'i|info!'         => \$info,
            'a|archived!'     => \$archived,
            'T|fgmtelOption:s'  => \$fgmtelOption,
            'C|fgmcalOption:s'  => \$fgmcalOption,
            'v|version:s'     => \$version);

pod2usage( -verbose => $help, -noperldoc => 1) if $help;

die 'Please give the complete date' unless ($yy and $mm and $dstr); # y2k bug!

if ($yy eq '2000') {$yy='00'}

if (length($yy)      eq 1) {$yy      ='0'.$yy}
if (length($mm)      eq 1) {$mm      ='0'.$mm}
if (length($tbegin)  eq 1) {$tbegin  ='0'.$tbegin}
if (length($tend)    eq 1) {$tend    ='0'.$tend}
if (length($version) eq 1) {$version ='0'.$version}
$fgmtelOption=' '.$fgmtelOption.' ';
$fgmcalOption=' '.$fgmcalOption.' ';

$tb='00:00:00.000';
$te='23:59:59.999';

if ($tend) {$te=$tb;substr($te,0,length($tend))=$tend;}
substr($tb,0,length($tbegin))=$tbegin if $tbegin;
 
if (not($res =~ /\D/)) {
  $av_arg="-s $res";
  $res.='s';
} elsif ($res =~ /^1\/\d+$/) {
  $res =~ /^1\/(\d+)$/; $av_arg="-f $1";
  $res =$1.'VPS';
} elsif ($res eq 'spin') {
  $av_arg='-p 26.3671875';
} elsif ($res eq 'dacq') {
  $av_arg='';
} else {die 'Invalid resolution'}

@days_list=split ' ', $dstr;
foreach $element (@days_list) {
  if ($element=~/^\d{1,2}$/) {
    push @days, $element
  } elsif ($element=~/^(\d{1,2})-(\d{1,2})$/) {
    for ($dd=$1;$dd<=$2;$dd++) {push @days, $dd}
  } else {die "incorrect day specification!"}
}
foreach $dd (@days) {$dd='0'.$dd if (length($dd) eq 1)}

$ENV{FGMROOT}   = '/home/FGM/'                   unless $ENV{FGMROOT};
$ENV{SATTPATH}  = "$ENV{FGMROOT}/log/atorb/"     unless $ENV{SATTPATH};
$ENV{ORBITPATH} = "$ENV{FGMROOT}/log/atorb/"     unless $ENV{ORBITPATH};

if ($fgmcalfile) {
  $fgmcalfile=$ENV{PWD}.'/'.$fgmcalfile unless $fgmcalfile=~/^\//;
  $fgmcalfile=~/^(.+\/).+?$/; $ENV{FGMPATH} = $1; 
} elsif ($archived) {
  $ENV{FGMPATH} = "$ENV{FGMROOT}/archive/calfiles/fgmcal/20$yy/$mm/";
} else {
  $ENV{FGMPATH} = "$ENV{FGMROOT}/data/dcalf/"    unless $ENV{FGMPATH};
}

$outdir   = "$ENV{FGMROOT}/data/output/$coord"."pos$res" unless $outdir;    
$exepath  = "$ENV{FGMROOT}/bin";
$calfpath = "$ENV{FGMPATH}";

if ($ICL) {
  $ipath = $ENV{FGMROOT}."/data/raw/ICL/$yy"."_$mm"; $iclstring='_ICL';
} else {
  $ipath = $ENV{FGMROOT}."/data/raw/ESTEC/";         $iclstring='';
}

#`mkdirhier $outdir`; 
`mkdir -p $outdir; chmod g+s $outdir`;

# need const.fgm
`cp $ENV{FGMROOT}/data/dcalf/const.fgm $ENV{FGMPATH}` unless -e "$ENV{FGMPATH}/const.fgm";

if ($info) {$info='-i'} else {$info=''}

@scarr=split ' ', $scstr ;
foreach $sc (@scarr) {
  foreach $dd (@days) {
    $begin_time = "20$yy-$mm-$dd".'T'.$tb.'Z';                    
    $end_time   = "20$yy-$mm-$dd".'T'.$te.'Z';               
    $calfile    = "C$sc"."_20$yy$mm$dd".'_V'.$version.'.fgmcal'; 
    $r7daily    = "C$sc"."_20$yy$mm$dd".'range7.fgmcal';
    $r7default  = "C$sc".'_range7.fgmcal';     
    $outputfile = "C$sc"."_20$yy$mm$dd".'_'.substr($tb,0,5).
                                        '-'.substr($te,0,5)."_$coord".
                                        'pos'.$res.$iclstring.'.dat';  
    $outputfile = "C$sc".$name.'.dat' if $name;
    if (-e "$ENV{FGMPATH}/$r7daily") {`cp $ENV{FGMPATH}/$r7daily $ENV{FGMPATH}/$r7default`}
    
#    if ($ICL) {
#      $ifiles=`find $ipath -regex ".*C$sc.$yy$mm$dd.B\.[NB]S\\(\.gz\\)?" -a ! -size 0`;
#    } else {
#      $ifiles=`find $ipath -regex ".*$yy$mm$dd?f[nb]\..a$sc\\(\.gz\\)?" -a ! -size 0`;
#    }
    
    
    @ifiles=selectraw(ICL=>$ICL,sc=>$sc,bs=>'BSNS',yy=>$yy,mm=>$mm,dd=>$dd,dv=>'last');
    
    
    
           
    die "FGM data for $yy-$mm-$dd not found" unless @ifiles;


#    $ifgz=$ifiles;
#    $ifiles =~s/\n/ /g;
#    $ifiles =~s/\.gz//g;
        
#    foreach $ifile (split(/\s+/,$ifgz)) {`gunzip $ifile` if $ifile=~/\.gz$/;}    

    foreach $ifile (@ifiles) {`gunzip $ifile` if $ifile=~/\.gz$/;}

    $ifiles=join(' ',@ifiles);$ifiles =~s/\.gz//g;

        
    $avgcmd="$exepath/fgmav $av_arg |" ;
    $avgcmd='' if ($av_arg eq '');

#    system "$exepath/ddsmrg $ifiles                     |          
#            $exepath/ddscut -b $begin_time -e $end_time |        
#            $exepath/fgmtel                             |                
#            $exepath/fgmcal -c $calfpath/$calfile -i    |        
#            $exepath/fgmhrt -s $coord                   | $avgcmd            
#            $exepath/fgmpos                             |                  
#            $exepath/igmvec -i -o $outdir/$outputfile   
#            rm cal.log";                                         


if ($fgmcalfile) { $calfpath=''; $calfile=$fgmcalfile; }

#print STDOUT "$exepath/ddsmrg $ifiles                         |          
system "$exepath/ddsmrg $ifiles                         |          
            $exepath/fgmtel $fgmtelOption               |                
            $exepath/fgmcut -b $begin_time -e $end_time |        
            $exepath/fgmcal $fgmcalOption -c $calfpath/$calfile -i |        
            $exepath/fgmhrt -s $coord                   | $avgcmd            
            $exepath/fgmpos                             |                  
            $exepath/igmvec $info -t $time_fmt -o $outdir/$outputfile   
            rm cal.log";                                         

#    foreach $ifile (split(/\s+/,$ifgz)) {
#      if ($ifile=~/\.gz$/){substr($ifile,-3,3)=''; `gzip $ifile`;}
#    }
  }
}
#-------------------------------------------------------------------------

1;

__END__

=head1 NAME

makedata.pl - Produces calibrated FGM data using raw data and 
calibration files. Based on gsepos4cd1s_multi provided by eg. 
Compressed input files are transparently managed. 

=head1 SYNOPSIS

B<makedata.pl> [B<--year> I<year>] B<--month> I<month> B<--days> I<days_list>
[B<--begin> I<begin_time>] [B<--end> I<end_time>] [B<--fgmcalfile> I<fgmcalfile>]
[B<--spacecraft> I<spacecraft>]  [B<--coordinates> I<coordinate_system>] 
[B<--resolution> I<resolution>] [B<--outdir> I<output_directory>] 
[B<--name> I<filename>] [B<--ICL>] [B<--timeformat> I<format>]
[B<--version> I<version>] [B<--(no)archived>] [B<--fgmtelOption>] [B<--fgmcalOption>] [B<--help>]

=head1 DESCRIPTION

This script produces calibrated FGM data using raw data and 
calibration files with various resolutions and in various coordinate systems.
If daily calibration files for range 7 are found, they are used, 
otherwise, the default range 7 calibration files are used. 

=head1 OPTIONS

=over 4

=item B<-y> I<year>, B<--year> I<year>

The year. One or two digits. Default is the current year.

=item B<-m> I<month>, B<--month> I<month> 

The month. One or two digits. Mandatory argument.

=item B<-s> I<spacecraft>, B<--spacecraft> I<spacecraft>

The spacecraft number (1-4) or list of spacecraft e.g B<-s> C<'1 3 4'>. 
Default all spacecraft.

=item B<-d> I<days_list>, B<--days> I<days_list>

List of days given as a string. Ranges are permitted, 
e.g B<-d> C<'3 12 15-28 30'>. Mandatory argument.

=item B<-b> I<begin_time>, B<--begin> I<begin_time>

Specify the starting time. Full format is hh:mm:ss.mmm. If only the first part
ot the format is given, the rest is filled with zeroes. e.g. B<-b> C<2> is 
same as B<-b> C<02:00:00:000> or B<-b> C<03:45> is same as 
B<-b> C<03:45:00.000>. Default is C<00:00:00.000>.

=item B<-e> I<end_time>, B<--end> I<end_time>

Specify the end time of the interval. Same format as for  B<-begin> option.
Deafult is C<23:59:59.999>.

=item B<-c> I<coordinate_system>, B<--coordinates> I<coordinate_system>

The output coordinate system. Default is GSE. Valid values are: 

=over 8
 
=item I<sr>   for spin-reference system,

=item I<scs>  for spacecraft-sun system, 

=item I<gse>  for geocentric solar ecliptic system,

=item I<gsm>  for geocentric solar magnetospheric system,

=item I<sm>   for solar magnetic system, or

=item I<j2k>  for geocentric equatorial inertial system of epoch J2000. 

=back

=head2

=item B<-r> I<resolution>, B<--resolution> I<resolution>

The resolution of output data. It can be given as an integer I<n> or as I<1/n>.
Also I<spin> can be used to obtain spin averaged data. If not specified the data
is not interpolated and the output will have the actual acquisition rate.

=item B<-I> ,B<--ICL>

Use the Imperial College London raw data. 
Default is to use the ESTEC raw data.

=item B<-o> I<output_directory>, B<--outdir> I<output_directory>

The destination directory for the output files. Default is 
I<$FGMROOT/data/output/$coordposress>.

=item  B<-n> I<filename>, B<--name> I<filename>

The base name of the output file. Output file name will be CNfilename.dat.
Default is I<CN_YYYYMMDD_hh:mm-hh:mm_coordPosRes.dat>

=item B<-f> I<fgmcalfile>, B<--fgmcalfile> I<fgmcalfile>

Specify the fgmcal calibration file to use. It also sets the $FGMPATH 
environment variable to the directory where the calibration file resides.

=item B<-t> I<format>, B<--timeformat> I<format>

Time information format: 

=over 8

=item 0 : ISO standard time string like '2000-12-02T02:05:15.798Z',

=item 1 : (double) seconds of the day,

=item 2 : (double) hours of the day

=back

Default is the ISO time string.


=item B<-v> I<version>, B<--version> I<version>

Version of the calibration files. If this option is not given then the
environment variable FGMVERSION is used. If FGMVERSION is not set, then the
default version is 3. Has no effect if the calibration file is given using the 
B<-fgmcalfile> option.

=item B<-a>, B<--(no)archived>

Use the archived calibration files. This is the default and sets the 
$FGMPATH environment variable to archived calibration files directory,
F<$FGMROOT/archive/calfiles/fgmcal/YYYY/MM/>. If this option is negated
with B<--noarchived> then the existing $FGMPATH environment variable
is used, or falls back to the default path F<$FGMROOT/data/dcalf/> if
$FGMPATH is not already set. Has no effect if the calibration file is given 
using the B<--fgmcalfile> option.

=item B<-T>, B<--fgmtelOption>

Add extra option to fgmtel (e.g. '-a' to output all vectors, incuding "bad" ones).

=item B<-C>, B<--fgmcalOption>

Add extra option to fgmcal (e.g. '-a' to output all vectors, incuding "bad" ones).

=item B<-h>, B<-?>, B<--help>

Prints a brief help message.

=back

=head1 ENVIRONMENT

=over 4

=item FGMROOT

Root for the FGM calibration directory structure. Default to F</home/FGM/> if
not set.

=item FGMPATH

Path to the fgmcal calibration files. 
Default to F<$FGMROOT/archive/calfiles/fgmcal/YYYY/MM/> 
or to F<$FGMROOT/data/dcalf/> if B<--noarchived> option is used.
The basename of the fgmcal file is used if this it is given via the 
B<-fgmcalfile> option.

=item SATTPATH

Path to orbit parameters files. Default to F<$FGMROOT/log/atorb/> if not set.

=item ORBITPATH

Path to orbit parameters files. Default to F<$FGMROOT/log/atorb/> if not set.

=back

=head1 FILES

F<$FGMROOT/data/raw/ICL/> - Imperial input path. Raw files are searched 
recursively in this directory.

F<$FGMROOT/data/raw/ESTEC/> - ESTEC (default) input path. Raw files are 
searched recursively in this directory.

F</home/FGM/archive/calfiles/fgmcal/20$yy/$mm/> - calibration files path.

F<CN_YYYYMMDD_Vnn.fgmcal>, F<CN_YYYYMMDDrange7.fgmcal> - daily calibration 
files.

F<CN_range7.fgmcal> - default range 7 calibration files.

F<$FGMROOT/data/output/coordPosRes> - output directory unless B<-o> option 
is used.

F<CN_YYYYMMDD_hh:mm-hh:mm_coordPosRes.dat> - output file

=head1 DEPENDENCES

This script uses the following:

 ddsmrg,
 ddscut,
 fgmtel,
 fgmcal,
 fgmhrt,
 fgmav, 
 fgmpos,
 igmvec.

=head1 AUTHOR

Dragos Constantinescu <d.constantinescu@tu-bs.de>

=cut
