#!/usr/bin/perl -w
use Getopt::Long qw(:config no_ignore_case bundling);
use Pod::Usage;
use lib "/home/FGM/scripts";
use FGMperl;



$y    =  substr((localtime())[5],1);
$r    =  2;
$de   = 31;
$line =  1;
$help =  0;
$log  =  1;
$idl  =  1;
$ask  =  1;
$from = '';

$ENV{FGMVERSION} ? ($version=$ENV{FGMVERSION}) : ($version=3);

$me="$0 @ARGV";

GetOptions ('h|help|?+'      => \$help, 
            's|spacecraft=i' => \$sc,
            'y|year=i'       => \$y,
            'm|month=i'      => \$m,
            'r|range=s'      => \$r,
            'b|begin=i'      => \$db,
            'e|end=i'        => \$de,
            'd|day=i'        => \$dd,
            'f|from=s'       => \$from,
            'p|parameter=s'  => \$key,
            'l|line=i'       => \$line,
            'c|column=i'     => \$column,
            'V|Value=s'      => \$newval,
            'log!'           => \$log,
            'i|idl!'         => \$idl,
            'g|group=s'      => \$grp,
            'v|version=i'    => \$version,
            'a|ask!'         => \$ask
           );

pod2usage( -verbose => $help, -noperldoc => 1) if $help;

$ENV{FGMROOT} = '/home/FGM/'                unless $ENV{FGMROOT};
$ENV{FGMPATH} = "$ENV{FGMROOT}/data/dcalf/" unless $ENV{FGMPATH};

$db=$de=$dd if $dd;

if (length($y)       eq 1) {$y       ='0'.$y;}
if (length($m)       eq 1) {$m       ='0'.$m;}
if (length($db)      eq 1) {$db      ='0'.$db;}
if (length($de)      eq 1) {$de      ='0'.$de;}
if (length($version) eq 1) {$version ='0'.$version;}

if ($grp) {
  die 'value cannot be given for a group!' if $newval;
  
  if ($grp =~ /^r\d(\d)$/){
    @newargs=(" -p offset_r$1 -c 1 --nolog ",
              " -p offset_r$1 -c 2 --nolog ",
              " -p offset_r$1 -c 3 --nolog ");
  } elsif ($grp =~ /^r\d(\d)(\d)$/) {
    @newargs=(" -p offset_r$1 -c 1 --nolog ",
              " -p offset_r$1 -c 2 --nolog ",
              " -p offset_r$1 -c 3 --nolog ",
              " -p offset_r$2 -c 1 --nolog ",
              " -p offset_r$2 -c 2 --nolog ",
              " -p offset_r$2 -c 3 --nolog ");
  } elsif ($grp =~ /^hh(\d)$/) {
    @newargs=(" -p Matrix_r$1 -l 3 -c 2 --nolog ",
              " -p Matrix_r$1 -l 3 -c 3 --nolog ");
  } elsif ($grp eq 'angle'){
    @newargs=(' -p Angle_xyz -c 1 --nolog ',
              ' -p Angle_xyz -c 2 --nolog ',
              ' -p Angle_xyz -c 3 --nolog ');
  } elsif ($grp =~ 'sco') {
    @newargs=(" -p offset_sc -c 2 --nolog ",
              " -p offset_sc -c 3 --nolog ");
  } elsif ($grp =~ /^R(\d)$/){
    @newargs=(" -p offset_r$1 -c 1 --nolog ",
              " -p offset_r$1 -c 2 --nolog ",
              " -p offset_r$1 -c 3 --nolog ",
              " -p Matrix_r$1 -l 1 -c 1 --nolog ",
              " -p Matrix_r$1 -l 1 -c 2 --nolog ",
              " -p Matrix_r$1 -l 1 -c 3 --nolog ",
              " -p Matrix_r$1 -l 2 -c 1 --nolog ",
              " -p Matrix_r$1 -l 2 -c 2 --nolog ",
              " -p Matrix_r$1 -l 2 -c 3 --nolog ",
              " -p Matrix_r$1 -l 3 -c 1 --nolog ",
              " -p Matrix_r$1 -l 3 -c 2 --nolog ",
              " -p Matrix_r$1 -l 3 -c 3 --nolog ");
  } else {die 'invalid parameter group';}
  
  #$newme=$me; $newme=~s/-g\w*\s+\w+//;#without Getopt bundling
  $newme=$me; $newme=~s/-g\s*\w+//;
  $newme=~s/--g\w*\s+\w+//;
  
  foreach $newarg (@newargs) {system $newme.$newarg;}
  
  open  LOG, ">>$ENV{FGMROOT}/log/dailycal/dailycal_$y$m.log";
  print LOG localtime()." $ENV{USER} $me\n" if $log;
  close LOG;
  
  exit;
}

$lstfile="$ENV{FGMROOT}/cfg/cfgnew2fgmcal.lst";

chdir $ENV{FGMPATH};

die "Please specify the spacecraft (option --spacecraft 1-4)" unless $sc;
die "Please specify the month (option --month 1-12)" unless $m;
die "Please specify the first day or a single day (option --begin 1-31 or --day 1-31)" unless $db;
die "Please specify the parameter to propagate (option --parameter)" unless $key;
die "Please specify the column (option --column 1-3)" unless $column;

$column--; $line--;

$CT_key=$key;

$key=$FGM_names{$key};

die "Invalid parameter!" unless $key;

$pattern='c'.$sc.'_'.$y.$m.'??_????r'.$r.'_V'.$version.'.cfgnew';

@files=glob($pattern);

@f2change=();   $day_old='00';@multiple=();
foreach $file (@files) {
  $file=~/\d{4}(\d\d)_(\d\d)(\d\d)/;
  $day=$1; #$hour=$2; $minute=$3;
  if ($day ge $db and $day le $de){
    push @f2change, $file;
    
    if ($day eq $day_old) {
      push @multiple, 
      glob('c'.$sc.'_'.$y.$m.$day.'_????r'.$r.'_V'.$version.'.cfgnew');
    }
    $day_old=$day;
    
  }
}

if ($#multiple gt 0) {
  print "\nWARNING: multiple r$r cfgnew files per day in $ENV{FGMPATH}:\n".join("\n",@multiple)."
         The fgmcal file will correspond to only one of these.
         It is recommended to remove the unnecessary cfgnew files.\n\n";
}

$nfiles=$#f2change+1; # this does not account for inifile inside interval

$iniday=$y.$m.$db;
$iniversion=$version;
@firstfile=glob('c'.$sc.'_'.$iniday.'_????r'.$r.'_V'.$iniversion.'.cfgnew');

if (not $newval) {
  if ($from){
    $from=~/^(\d\d)(\d\d)(\d\d)(v.+)?$/ or die "from format is YYMMDD[vNN]";
    $iniday=$1.$2.$3;
    $iniversion=$4 if defined($4);
    @firstfile=glob('c'.$sc.'_'.$iniday.'_????r'.$r.'_V'.$iniversion.'.cfgnew');
  }
  die "missing cfgnew file $iniversion on $iniday" if ($#firstfile eq -1);
  die "too many cal files $iniversion  on $iniday" unless ($#firstfile eq 0);
  $inifile=$firstfile[0];
  %p=readcalfile($inifile);
  $newval=$p{$key}[$column+3*$line];
  # $nfiles--;
} else {$inifile='none';}


print <<MESSAGE;

  Spacecraft: $sc, Range: $r
  
  Initialization file: $inifile
  
  Propagating $CT_key   = $newval from $y.$m.$db to $y.$m.$de
              line      = $line+1
              column    = $column+1   
  
  Corresponcence: $CT_key -> $key
  
  Number of files to change: $nfiles
  
  Ok to proceed? [y/n]
  
MESSAGE

$answer='y';
chomp($answer= <STDIN>) if $ask;

die 'ABBORTED' unless ($answer eq 'y');

open  LOG, ">>$ENV{FGMROOT}/log/dailycal/dailycal_$y$m.log";
print LOG localtime()." $ENV{USER} $me\n" if $log;
close LOG;

open (LSTF,">$lstfile");
foreach $file (@f2change) {
  print "changing $file...\n";   
  %p=readcalfile($file);      
  $oldvalue=$p{$key}[$column+3*$line];         
  $p{$key}[$column+3*$line]=$newval;  
  print "...$key($column,$line): $oldvalue -> $newval\n";  
  open (CFILE,">$file");
    print CFILE calfilestr(\%p);  
  close CFILE;        
  print LSTF "$file\n";
}
close LSTF;

system "idl -quiet -e '@/home/FGM/idl/cfgnew2fgmcal.batch' >/dev/null" if $idl;



#-------------------------------------------------------------------------

1;

__END__


=head1 NAME

modcalf.pl - Propagate changes through FGM daily calibration files. 

=head1 SYNOPSIS

B<modcalf.pl> B<--month> I<month> [B<--year> I<year>] [B<--from> I<yymmdd[vNN]>] 
B<--begin> I<day_begin>|B<--day> I<day> [B<--end> I<day_end>] 
B<--spacecraft> I<spacecraft>
B<--parameter> I<name> 
[B<--line> I<line>] B<--column> I<column> 
B<--group> I<parameter group>
[B<--Value> I<value>] B<-r-ange> I<range> 
[B<--version> I<version>] [B<--(no)log>] [B<--(no)idl>] 
[B<--(no)ask>] [B<--help>]

=head1 DESCRIPTION

This script propagates a given parameter value through FGM F<cfgnew> daily 
calibration files. If no value (B<--Value>) or initialization day (B<--from>) 
is given, the value to propagate is read 
from the first file in sequence. By default, the changes are also reflected in
the corresponding F<fgmcal> file. Groups of more parameters corresponding to
range changes, higher harmonics, or entire ranges can also be changed.

=head1 OPTIONS

=over 4

=item B<-m> I<month>, B<--month> I<month> 

The month. One or two digits. Mandatory argument.

=item B<-y> I<year>, B<--year> I<year>

The year. One or two digits. Default is the current year.

=item B<-b> I<day_begin>, B<--begin> I<day_begin>

First day in sequence. One or two digits. Either this argument or 
(B<--day> I<day>) has to be used.

=item B<-d> I<day>, B<--day> I<day>

Only this day will be changed. Takes precedence over B<-b> and B<-e> options.

=item B<-e> I<day_end>, B<--end> I<day_end>

Last day in sequence. Default is end of month.

=item B<-s> I<spacecraft>, B<--spacecraft> I<spacecraft>

The spacecraft number (1-4). Mandatory argument.

=item B<-p> I<name>, B<--parameter> I<name> 

The name of the parameter to be modified. This name is the 
GUI parameter name and NOT the name in the calibration file. 
Valid names are:

 offset_r2,
 offset_r3,
 offset_r4,
 offset_r5,
 offset_r6,
 offset_r7,

 offset_sc,

 Matrix_r2,
 Matrix_r3,
 Matrix_r4,
 Matrix_r5,
 Matrix_r6,
 Matrix_r7,

 Matrix_sc,
 Matrix_dr,
 Angle_xyz.


=item B<-l> I<line>, B<--line> I<line>

The matrix line (1-3). Default is 1.

=item B<-c> I<column>, B<--column> I<column>

The matrix column (1-3). Mandatory argument if B<-g> option is not used.

=item B<-f> I<yymmdd>, B<--from> I<yymmdd[vNN]>

Initialization day and version. Day from which the the values to be propagated
are red. If the version part is omitted the version given by the environment 
variable FGMVERSION is used unless overwritten by the B<-version>. Deafault 
version is I<03>. Without the B<-from> option, the values are red from the 
first day in the sequence unless the B<-Value> option is given.

=item B<-V> I<value>, B<--Value> I<value>

The value to be assigned. Takes precedence over the B<-from> option. 
Default value is read from the first file if neither B<-Value> or B<-from> 
options are given.

=item B<-g> I<parameter_group>, B<--group> I<parameter_group>

Specify a group of parameters to propagate. Valid values are:

 rNM  ->  -p offset_rM -c 1
          -p offset_rM -c 2
          -p offset_rM -c 3

 rNMP ->  rNM and rMP
 
 hhN  ->  -p Matrix_rN -l 3 -c 2
          -p Matrix_rN -l 3 -c 3

 angle -> -p Angle_xyz -c 1
          -p Angle_xyz -c 2
          -p Angle_xyz -c 3
 
 sco  ->   -p offset_sc -c 2
           -p offset_sc -c 3

 RN   ->  -p offset_rN -c 1
          -p offset_rN -c 2
          -p offset_rN -c 3
          -p Matrix_rN -l 1 -c 1
          -p Matrix_rN -l 1 -c 2
          -p Matrix_rN -l 1 -c 3
          -p Matrix_rN -l 2 -c 1
          -p Matrix_rN -l 2 -c 2
          -p Matrix_rN -l 2 -c 3
          -p Matrix_rN -l 3 -c 1
          -p Matrix_rN -l 3 -c 2
          -p Matrix_rN -l 3 -c 3

N,M,P can take values from 2 to 6 and must be consecutive.

=item B<-r> I<range>, B<--range> I<range>

If given, changes are propagated only from and to the given range.
Otherwise all ranges are modified.

=item B<-v> I<version>, B<--version> I<version>

Version of the calibration files. If this option is not given then the
environment variable FGMVERSION is used. If FGMVERSION is not set, then the
default version is 3.

=item B<--log>

Record the run to dailycal log file. Default is enabled. 
Can be disbled using the B<-nolog> option.

=item B<-i>, B<--idl> 

Change the F<fgmcal> files to reflect the new F<cfgnew> files.
This option is enabled by default. 
To disable it, negate: B<--noidl>.

=item B<-a>, B<--ask> 

Ask for confirmation before proceeding to propagate the changes.
This option is enabled by default. 
To disable it, negate: B<--noask>.

=item B<-h>, B<-?>, B<--help>

Prints a brief help message.

=back

=head1 EXAMPLE

propagate sc 1 offset_sc, z value, from february 20 to 28:  

C<modcalf.pl -s 1 -m 2 -b 20 -e 28 -p offset_sc -c 3>

propagate the parameters corresponding to higher harmonic calibration
for range 4 (Matrix_r4, line 3, columns 2 and 3) for spacecraft 3 
from 21 june to the end of the month:

C<modcalf.pl -s 3 -m 6 -b 21 -g hh4>

=head1 ENVIRONMENT

=over 4

=item FGMROOT

Root for the FGM calibration directory structure. Default to F</home/FGM/> if
not set.

=item FGMPATH

Path to calibration files (*.fgmcal and *.cfgnew). Default to
F<$FGMROOT/data/dcalf/> if not set.

=back

=head1 FILES

F<$FGMPATH> - working directory

F<c$sc_$yy$mm$dd_hhmmr$rV01.cfgnew> - daily calibration files

F<$FGMROOT/log/dailycal/dailycal_$yy$mm.log> - Dailycal log file.

F<FGMROOT/cfg/cfgnew2fgmcal.lst> - list of changed files. Used by 
C<cfgnew2fgmcal.pro> to produce the corresponding F<fgmcal> files.

=head1 DEPENDENCES

This script depends on the C<cfgnew2fgmcal.pro> IDL script.

=head1 AUTHOR

Dragos Constantinescu <d.constantinescu@tu-bs.de>

=cut



