#!/usr/bin/perl -w
use Getopt::Long qw(:config no_ignore_case bundling);
use lib $ENV{HOME}."/perl/modules";
use lib "/home/FGM/scripts";
use PDL;
use PDL::Graphics::PGPLOT;
use PDL::NiceSlice;
use Term::ANSIColor;
use Utils;
use Pod::Usage;
use FGMperl;

$|=1;

$yy     = substr((localtime())[5],1);
$length = 10;
$all    = '';
$range  = '';
$dstr   = '';
$b    =  1;
$e    = 31;
$scstr  = '1 2 3 4';
$ICL    = 0;
$xwin   = 1;

$ENV{FGMVERSION} ? ($version=$ENV{FGMVERSION}) : ($version='3');

GetOptions ('h|help|?+'      => \$help, 
            'y|year=i'       => \$yy,
            'm|month=i'      => \$mm,
            'd|days:s'       => \$dstr,
            'b|begin:i'      => \$b,
            'e|end:i'        => \$e,
            'v|version=s'    => \$version,
            'L|Length=i'     => \$length,
            'A|All'          => \$all,
            's|spacecraft:s' => \$scstr,
            'I|ICL'          => \$ICL,
            'r|ranges=s'     => \$range,
            'x|xwin!'        => \$xwin);

pod2usage( -verbose => $help, -noperldoc => 1) if $help;

die 'Please use the --month option' unless ($mm);

if (length($yy)       eq 1) {$yy       ='0'.$yy;}
if (length($mm)       eq 1) {$mm       ='0'.$mm;}
if (length($version)  eq 1) {$version  ='0'.$version;}

@scarr=split ' ', $scstr ;

@days=();
if ($dstr) {
  @days_list=split ' ', $dstr;
  foreach $element (@days_list) {
    if ($element=~/^\d{1,2}$/) {
      push @days, $element
    } elsif ($element=~/^(\d{1,2})-(\d{1,2})$/) {
      for ($dd=$1;$dd<=$2;$dd++) {push @days, $dd}
    } else {die "incorrect day specification!"}
  }
} else {
  for ($d=$b;$d<=$e;$d++) {push @days, $d}
}
foreach $d (@days) {$d='0'.$d if (length($d) eq 1);}

$xwin=0 if $#days;  # more than one day
$xwin=0 if $#scarr; # more than one sc

$all=' -a ' if $all;

$ENV{FGMROOT}  = '/home/FGM/'                unless $ENV{FGMROOT};
$ENV{SATTPATH} = "$ENV{FGMROOT}/log/atorb/"  unless $ENV{SATTPATH};
$ENV{FGMPATH}  = "$ENV{FGMROOT}/data/dcalf/" unless $ENV{FGMPATH};

$exepath    = "$ENV{FGMROOT}/bin";
$logfiledir = "$ENV{FGMROOT}/log/cd_log/$yy"."_$mm"; 
$offFile    = "$ENV{FGMROOT}/data/ini/rc_off.tmp";
$opath      = "$ENV{FGMROOT}/data/cfcal/";
$pdfopath   = "$ENV{FGMROOT}/data/rcplots/20$yy/$mm";
mkdir(substr($pdfopath,0,length($pdfopath)-3)); mkdir($pdfopath);
`chmod g+s $pdfopath`;

if ($xwin) {
  $win = PDL::Graphics::PGPLOT::Window->new({Dev      => '/xserve',
                                           ny       => 3,
                                           CHARSIZE => 2.5,
                                           axis     => ['BCNSTZHX','BCNSVT']});
}

foreach $sc (@scarr) {
  foreach $dd (@days){

    $calfile           = "$ENV{FGMPATH}/C$sc"."_20$yy$mm$dd"."_V$version".".fgmcal";
    $r7daily           = "$ENV{FGMPATH}/C$sc"."_20$yy$mm$dd"."range7.fgmcal";
    #$currentOffsetFile = "$ENV{FGMROOT}/data/ini/cal$sc"."_1d.neu";
    $r7default         = "$ENV{FGMPATH}/C$sc"."_range7.fgmcal";
    @cfgnewFiles       = glob("$ENV{FGMPATH}/c$sc"."_$yy$mm$dd"."_????r2_V$version".".cfgnew");
    $cfgnewFile        = $cfgnewFiles[0];
    if ($#cfgnewFiles gt 0) {
      print "\nWARNING: multiple r2 cfgnew files in $ENV{FGMPATH}:".
      join("\n      ",@cfgnewFiles)."
      Using $cfgnewFile.\n\n";
    }
    
    if ($ICL) {
      $ipath = $ENV{FGMROOT}."/data/raw/ICL/$yy"."_$mm";
    } else {
      $ipath = $ENV{FGMROOT}."/data/raw/ESTEC/cluster$sc/?sd_$sc/";
    }
    
    next unless -e $calfile;
    
    $oldOff=0;
    
    RELOAD:

    #%p=readcalfile($currentOffsetFile);
    %p=readcalfile($cfgnewFile);
        
    if (-e $r7daily) {`cp $r7daily $r7default`}


    $data_v=selectraw(ICL=>$ICL,sc=>$sc,bs=>'BSNS',yy=>$yy,mm=>$mm,dd=>$dd,dv=>'last');

#print "\n -- $data_v --\n"; die;
    if ($ICL) {
      $ifile  = "C$sc"."_$yy$mm$dd".'_B.NS';
      $burst  = "C$sc"."_$yy$mm$dd".'_B.BS';
      $both   = "C$sc"."_$yy$mm$dd".'_B.?S';
      $bsfile = $burst;
    } else {
      $ifile  = "$yy$mm$dd".'fn.?'."$data_v$sc";
      $burst  = "$yy$mm$dd".'fb.?'."$data_v$sc";
      $both   = "$yy$mm$dd".'f?.?'."$data_v$sc";
      $bsfile = "$yy$mm$dd".'fb.1'."$data_v$sc";
    }

    if (-e "$ipath/$bsfile" or -e "$ipath/$bsfile.gz"){
      $combined_logfile="$logfiledir/C$sc"."_$yy$mm$dd"."_B.BSNSlog"; 
      if (-e $combined_logfile) {
        $logfile=$combined_logfile;
        $to_merge="$ipath/$both";
      } else {
        $logfile="$logfiledir/C$sc"."_$yy$mm$dd"."_B.BSlog"; 
        $to_merge="$ipath/$burst";
      }
    } else {
      $logfile="$logfiledir/C$sc"."_$yy$mm$dd"."_B.NSlog";  
      $to_merge="$ipath/$ifile";
    }

    @Files=glob("$to_merge*");@mrgfiles=();  # remove empty files and decompress
    foreach $mf (@Files) {                                    
      next if -z $mf;        
      $mff=$mf; $mff=~s/\.gz//; push @mrgfiles, $mff;  
      `gunzip $mf` if $mf =~/\.gz$/;                                                       
    } 
    
    unless (-r $logfile) {
      print "no logfile for day $dd, spacecraft $sc\n";
      next;
    }
    
    open LOGF, "<$logfile" or die "could not open $logfile";
    $new_range=0;   $old_range=$new_range;
    $begin='00:00'; 
    while (<LOGF>) {
      if (/R\s+(\d)>FGM\s+sensor\s+range\s+(\d)/) {
        $new_range=$1;
        die 'something wrong with range identification' unless ($new_range eq $2);
      } elsif (/B\s+\d{4}-\d\d-\d\dT(\d\d:\d\d:\d\d\.\d\d\d)Z/) {
        $begin=$1;
        if (($new_range - $old_range) eq 1) { # found valid range change
          $ms_begin=timp1($begin);
          if ($ms_begin >= 86400000-1000*$length) {
            print "$old_range -> $new_range: too late\n";
            $old_range=$new_range;
            next;
          }
          if ($ms_begin <= 1000*$length) {
            print "$old_range -> $new_range: too early\n";
            $old_range=$new_range;
            next;
          }
          $Begin = timp2($ms_begin-1000*$length);                        
          $End   = timp2($ms_begin+1000*$length);                        
          $ofile="cf_c$sc"."_20$yy$mm$dd"."T$begin"."_r$old_range$new_range".".cal";
          unlink "$opath/$ofile";                                        
          system "$exepath/ddsmrg @mrgfiles                           |  
                  $exepath/ddscut -b T$Begin -e T$End                 |  
                  $exepath/fgmtel $all                                |                                   
                  $exepath/fgmcal $all -c $calfile                    |                                   
                  $exepath/fgmvec $all -m -f -t3 -r -o $opath/$ofile 2>/dev/null";    
        }
        $old_range=$new_range;
      }
    }
    close LOGF;

    foreach $mf (@Files) {   # recompress                                  
      next if -z $mf;       
      if ($mf=~/\.gz$/){substr($mf,-3,3)=''; `gzip $mf`;}                 
    }   

    @files=glob("$opath/cf_c$sc"."_20$yy$mm$dd"."T*_r??.cal");
    $fn=0;
    if ($range) {
      foreach $file (@files){
        last if $file=~/_r$range\.cal/;
        $fn++;
      }
    }
      
    if ($#files eq -1){
      print "no range changes for day $dd, spacecraft $sc\n";
      next;
    }
  
    print ' ' x 30 ."\r" if ($xwin); 
    $next=0;$db=0;$newOff=0;$sensit=0;
    @selected=(0,0);@t=(0,0);
    while (1) {
      last unless ($xwin);
      last unless ($win->device eq '/xserve');
      if (-z $files[$fn]){
        print "No data (empty $files[$fn])\n";
        last;
      }
      ($x, $y, $ch)=showrc($files[$fn],$win,pdl(@selected));
      if ($ch =~/n| /) {$next=1} elsif ($ch eq 'p') {$next=-1} else {$next=0}
      last if $ch=~/q|D/;
      if ($ch eq 'A'){
        if ($selected[0] and  $selected[1]){
          $selected[0]=$y; $t[0]=$x;
          $selected[1]=0;
          $db=0;$newOff=0;
        } elsif (!$selected[0]){
          $selected[0]=$y; $t[0]=$x;
          $db=0;$newOff=0;
        } elsif (!$selected[1]){
          $files[$fn]=~/cf_c.+T(\d\d:\d\d:\d\d\.\d\d\d)_r\d(\d).cal/;
          $trc=timp1($1)/1000.; $range=$2; $rtochange=$range;
          $selected[1]=$y; $t[1]=$x;
          $db=$selected[1]-$selected[0];
          $db=-$db if (($t[1] gt $trc) and ($t[0] lt $trc));
          $oldOff=$p{$FGM_names{"offset_r$range"}}[0];
          $newOff=$oldOff-$db;
          $sensit=$p{$FGM_names{"Matrix_r$range"}}[0]+$db/(($selected[1]+$selected[0])/2);
          open OFF, ">$offFile"; print OFF $range,' ', $db; close OFF;
        }
      } elsif ($ch eq 'r') {
        $files[$fn]=~/_r(\d\d).cal$/;
        $range=$1;
        @selected=(0,0);$db=0;
        print " reloading...                                         \r";
        goto RELOAD;    
      } elsif ($ch eq 'o') {
        $files[$fn]=~/_r(\d\d).cal$/;
        $range=$1;
        if ($selected[0] and  $selected[1]){
          print " changing offset for range $rtochange...\n";
          `modcalf.pl -y$yy -m$mm -s$sc -d$dd -poffset_r$rtochange -c1 -V\'$newOff\' --noask 2>/dev/null`;
        } else {
          print " please first select Bx values\n";
        }       
        goto RELOAD;
      } elsif ($ch eq 'b') {
        $files[$fn]=~/_r(\d\d).cal$/;
        $range=$1;
        if ($oldOff ne 0){
          print " going back to the old offset for range $rtochange...\n";
          `modcalf.pl -y$yy -m$mm -s$sc -d$dd -poffset_r$rtochange -c1 -V\'$oldOff\' --noask 2>/dev/null`;
        } else {print "no old offset available\n";}
        goto RELOAD;
      } else {
        @selected=(0,0);$db=0;$newOff=0;
      }
      $fn=($fn+$next) % ($#files+1);
      print color 'green';
      printf "%s [%s %5.2f] ",$ch,timp2($x*1000),$y;
      print color 'reset';
      printf "dBx=%1.3fnT (off -> ",$db;
      print color 'bold';
      printf "%1.3f",$newOff;
      print color 'reset';
      print ") (sns -> ";
      print color 'bold';
      printf "%1.5f",$sensit;
      print color 'reset';
      print ")         \r";
    }

    open OFF, ">$offFile"; print OFF '0 0'; close OFF;

    #$pid = fork;
    #if ($pid) { waitpid $pid,0;
    #} else { 
      print "\nSaving plots...";
      @selected=(0,0);
      foreach $file (@files){
        next if -z $file;
        $win = PDL::Graphics::PGPLOT::Window->new({Dev=>"$file.ps/VPS",
                                                   ny=>3,
                                                   CHARSIZE => 2.5,
                                                   axis => ['BCNSTZHX','BCNSVT']});
        showrc($file,$win,pdl(@selected));
        $win->close();
        `ps2pdf $file.ps $file.pdf; rm -f $file.ps`;
        $file.='.pdf';
      }
      $sc20=$sc.'_20';
      `pdfjoin @files --outfile $pdfopath/cf_c$sc20$yy$mm$dd.pdf`;
      unlink glob("$pdfopath/cf_c$sc20$yy$mm$dd"."T*.pdf");
      print " Done ($pdfopath/"."cf_c$sc20$yy$mm$dd.pdf)\n";
    #}
  } # foreach day dd
  unless ($dstr) { # no days given in the command line
    `pdfjoin $pdfopath/cf_c$sc20$yy$mm??.pdf --outfile $pdfopath/RC_c$sc20$yy$mm.pdf`;
  }
} # foreach sc


sub showrc {
  my ($file,$win,$sel) = @_;
  my ($t,$Bx,$By,$Bz,$B) = rcols $file,0,1,2,3,4;
  $file=~/cf_c(\d)_(\d\d\d\d)(\d\d)(\d\d)T(\d\d:\d\d:\d\d\.\d\d\d)_r(\d\d).cal/;
  my ($sc,$y,$m,$d,$time,$range)=($1,$2,$3,$4,$5,$6);
  my $ms_begin=timp1($time);
    
  $t*=3600;
  my $Bp=sqrt($By**2+$Bz**2);
  if ($win->device eq '/xserve') {$smb=1} else {$smb=4}
  plotPanel($t,$B,$ms_begin,$win,pdl(0,0),{Panel => 3, 
                                  axis => ['BCNSTZHX','BCNSVT'],
                                  symbol => $smb,
                                  PlotPosition => [0.1, .99, 0.15, 0.99] });
  $win->text('B', $t(0)->sclr+0.5, max($B)-(max($B)-min($B))/8);
 
  plotPanel($t,$Bp,$ms_begin,$win,pdl(0,0),{Panel => 2, 
                                                   axis => ['BCTS','BCNSVT'],
                                  symbol => $smb, color => 7,
                                  PlotPosition => [0.1, .99, 0.075, .925] });
  $win->text('B perp', $t(0)->sclr+0.5, max($Bp)-(max($Bp)-min($Bp))/8);
  
  plotPanel($t,$Bx,$ms_begin,$win,$sel,{Panel => 1, 
                                   axis => ['BCTS','BCNSVT'],
                                   symbol => $smb, color => 1,
                                   title => "$y.$m.$d  Cluster $sc, range $range",
                                   PlotPosition => [0.1, .99, 0, 0.85]  });
  $win->text('Bx', $t(0)->sclr+0.5, max($Bx)-(max($Bx)-min($Bx))/8);
  
 # $win->panel(2);
  
  if ($win->device eq '/xserve') {
    return $win->cursor({Type => 'CrossHair'})
  } else {return [0,0,0]}
}

sub plotPanel {
  my ($t,$b,$trc,$win,$sel,$opt)= @_;
  $win->erase({Panel => $$opt{Panel}});
  $win->points($t,$b, $opt);
  $win->hold;
  $win->line(ones(2)*$trc/1000,pdl(min($b),max($b)),{Panel=>$$opt{Panel},
                                                     color => 15});
  $trc0=max($t->where($t < ($trc/1000.-.03)))*1000;
  $win->line(ones(2)*$trc0/1000,pdl(min($b),max($b)),{Panel=>$$opt{Panel},
                                                     color => 15});
  $win->line($t(pdl(0,nelem($t)-1)), 
             ones(2)*$sel(0),{Panel=>$$opt{Panel},
                              color => 15}) if ($sel(0)->sclr ne 0);
  $win->line($t(pdl(0,nelem($t)-1)), 
                ones(2)*$sel(1),{Panel=>$$opt{Panel},
                                 color => 15}) if ($sel(1)->sclr ne 0);
  $win->release;
}


#-------------------------------------------------------------------------

1;

__END__

=head1 NAME

plotrc.pl - Interactivly plots range changes, calibrates the x offsets for 
range changes and saves the range chances plots as pdf files.

=head1 SYNOPSIS

B<plotrc.pl> [B<--year> I<year>] B<--month> I<month> 
[B<--days> I<days_list> | B<--begin> I<day_begin> B<--end> I<day_end>] 
B<--spacecraft> I<spacecraft>  B<--Length> I<length> B<--All> B<--ranges> I<ranges>  [B<--ICL>]
[B<--(no)xwin>] [B<--version> I<version>] [B<--help>]

=head1 DESCRIPTION

This script plots the magnetic field components Bx, sqrt(By^2+Bz^2) and the
magnetic field magnitude, B during intervals centered on the range change times
for the given spacecraft and day. It can use ESTEC or Imperial College raw data
which may also be compressed. Using keybord shortcuts the user can browse
through the identified range changes. Bx values selected with the mouse are
printed to the standard output. When two Bx values are selected, the
corresponding range change offset is calculated and a file which can be read by
the cls.pro idl calibration routine is prepared. Use I<update -> rc offset> from
cls.pro to load the prepared values. The script can also use the calculated
offset to directly modify the calibration files. When the script exits, the
plots are saved as pdf files. In non-interactive mode it produces the range
changes plots for the spacecraft and days given in the command line.

=head1 OPTIONS

=over 4

=item B<-y> I<year>, B<--year> I<year>

The year. One or two digits. Default is the current year.

=item B<-m> I<month>, B<--month> I<month> 

The month. One or two digits. Mandatory argument.

=item B<-d> I<days_list>, B<--days> I<days_list>

List of days given as a string. Ranges are permitted,  e.g B<-d> C<'3 12 15-28
30'>. If niether B<--days>, B<--begin>, or B<--end> options are given, than the
entire month is processed. 

=item B<-b> I<day_begin>, B<--begin> I<day_begin>

First day in sequence. Default is 1.

=item B<-e> I<day_end>, B<--end> I<day_end>

Last day in sequence. Default is 31.

=item B<-s> I<spacecraft>, B<--spacecraft> I<spacecraft>

The spacecraft number (1-4) or list of spacecraft e.g B<-s> C<'1 3 4'>. 
Defaults to all spacecraft.

=item B<-L> I<length>, B<--Length> I<length>

The duration in seconds before and after the range change which should be
ploted. Default is 10s. 

=item B<-A>, B<--All>

This option is passed to I<fgmtel>. If used, all data, including data marked as
'bad' will be ploted. See the fgmtel manual page.

=item B<-r> I<ranges>, B<--ranges> I<ranges>

The range change which is plotted first, e.g. B<-r> 56. Default is the first
range change of the day.

=item B<-I> ,B<--ICL>

Use the Imperial College London raw data. 
Default is to use the ESTEC raw data.

=item B<-x>, B<-xwin>

Interactive mode. Enabled by default if only one day and one spacecraft
are are given in the command line. Can be disabled using the B<--noxwin> 
option. In non-interactive mode only the pdf files with range changes
plots are produced.

=item B<-v> I<version>, B<--version> I<version>

Version of the calibration files. If this option is not given then the
environment variable FGMVERSION is used. If FGMVERSION is not set, then the
default version is 3.

=item B<-h>, B<-?>, B<--help>

Prints a brief help message.

=back

=head1 KEYBOARD AND MOUSE BINDINGS

The following commands are available in the plot window:  

=over 4

=item B<n> or B<space>

display the next range change.

=item B<p> 

display the previous range change.

=item B<o>

Update the corresponding calibration files (.cfgnew and .fgmcal) to reflect 
the calculated x offset.

=item B<b>

Undo the changes back to the x offset before the last change of the 
calibration files.

=item B<r>

read again the calibration files and redo the plot.

=item B<q> or B<mouse middle> 

quit.

=item B<mouse left> 

If two Bx values are already selected, then clear them, otherwise, select Bx
value. 

=item B<mouse right> 

clear Bx value.

=back

=head1 ENVIRONMENT

=over 4

=item FGMROOT

Root for the FGM calibration directory structure. Default to F</home/FGM/> if
not set.

=item FGMPATH

Path to calibration files (*.fgmcal and *.cfgnew). Default to
F<$FGMROOT/data/dcalf/> if not set.

=item SATTPATH

Path to orbit parameters files. Default to F<$FGMROOT/log/atorb/> if not set.

=back

=head1 FILES

F<$FGMROOT/log/cd_log/YY_MM> - log file directory.

F<Cn_YYMMDD_B.[NS/BS/BSNS]log> - log file. Used for identifying the range changes.

F<$FGMROOT/data/raw/YY_MM> - Imperial input path.

F<$FGMROOT/data/raw/ESTEC/cluster$sc/[n|b]sd_n/> - ESTEC (default) input path.

F<Cn_YYMMDD_B.[NS/BS]> - Imperial input file. Contains the raw data.

F<YYMMDD.f[n|b].?an> - ESTEC (default) input files

F<$FGMROOT/data/cfcal/> - output path for *.cal files.

F<cf_cn_YYYYMMDDThh:mm:ss.mmm_rMN.cal> - output file. One for each range change.

F<$FGMPATH/CN_YYYYMMDD_Vn.fgmcal> - calibration file used to produce the data
 for the plots. This file is updated with B<o> and B<b> interactive commands.

F<$FGMPATH/cN_YYMMDD_hhmmr2_Vn.cfgnew> - calibration file. Calibration
parameters are read from this file.  After selecting two Bx values, the computed
offset is writen in F<rc_off.tmp>. The B<o> and B<b> interactive commands can be
used to update this calibration file and the corresponding fgmcal file.

F<$FGMROOT/data/ini/rc_off.tmp> - range and offset value for cls.pro script.

F<$FGMROOT/data/rcplots/YYYY/MM/> - output directory for pdf plots.

F<cf_cn_YYYYMMDD.pdf> - plot with all range changes for the day.

F<RC_cn_YYYYMM.pdf> - plot with all range changes for the month. Only produced
if no days are specified in the command line.

=head1 DEPENDENCES

This script uses the following:

 ddscut,
 fgmtel,
 fgmcal,
 fgmvec,
 modcalf.pl,
 ps2pdf,
 pdfjoin.

=head1 AUTHOR

Dragos Constantinescu <d.constantinescu@tu-bs.de>

=cut


