#!/usr/bin/perl -w
use Getopt::Long qw(:config no_ignore_case bundling);
use Pod::Usage;

$yy     = substr((localtime())[5],1);
$mm     = 1;
$help   = 0;
$xpdf   = 1;
$output = '';

GetOptions ('h|help|?+'   => \$help, 
            'y|year=i'    => \$yy,
            'm|month=i'   => \$mm,
            'x|xpdf!'     => \$xpdf,
            'o|output:s'  => \$output);

pod2usage( -verbose => $help, -noperldoc => 1) if $help;

die 'Please use the --month option' unless $mm;

if (length($yy)  eq 1) {$yy  ='0'.$yy;}
if (length($mm)  eq 1) {$mm  ='0'.$mm;}        

$basename='table';
$ofile=$basename.'.tex';

$ENV{FGMROOT} = '/home/FGM/' unless $ENV{FGMROOT};

$output="$ENV{FGMROOT}/tmp/table.pdf" unless $output;
$output.='.pdf' unless $output=~/\.pdf/;

%months=('01' => 'January', '02' => 'February', '03' => 'March',
         '04' => 'April',   '05' => 'May',      '06' => 'June', 
         '07' => 'July',    '08' => 'August',   '09' => 'September', 
         '10' => 'October', '11' => 'November', '12' => 'December');
$month=$months{$mm};

## Read and store eclipses times

$orbitfiledir="$ENV{FGMROOT}/log/atorb/";

@E=([],[],[],[]);
for ($sc=1;$sc<=4;$sc++){
  @$E[$sc-1]=();
  for ($j=1;$j<=31;$j++){push @{$E[$sc-1]}, '';}
  open ORBITF, "<$orbitfiledir".'stef.cl'.$sc or die 'could not open orbit file';  
  while (<ORBITF>){                                                                
    if (/20$yy-$mm-(\d\d)T(\d\d:\d\d):\d\dZ\s+\S*\s+ECL \w+ (\S)\S+/){             
      $day=$1; $hm=$2; $start_end=$3;                                              
      $E[$sc-1][$day-1].= " $start_end\\,$hm ";                                    
    }                                                                              
  }                                                                                
  close ORBITF;
}

## Read and store range changes days

$logfiledir="$ENV{FGMROOT}/log/cd_log/".$yy."_$mm";

@RC=([],[],[],[]);
for ($sc=1;$sc<=4;$sc++){
  @$RC[$sc-1]=();
  for ($j=1;$j<=31;$j++){push @{$RC[$sc-1]}, '';}
  @logfiles=glob("$logfiledir/C$sc"."_$yy$mm??_B.NSlog");
  warn "        You should first produce the log files for $month:
        listfgm.pl -m $mm
        Without log files, the range changes are not printed.
        " if $#logfiles eq -1;
  foreach $logfile (@logfiles){
    $logfile=~/_$yy$mm(\d\d)_B\.NSlog/;
    $day=$1;
    $ranges='';
    open LOGF, "<$logfile";
    @rg_arr=();
    while (<LOGF>) {
      if (/^R\s+.+FGM sensor range (\d)$/) {
        if ($#rg_arr eq -1) {push @rg_arr, $1;
        } elsif (($1-$rg_arr[-1]) ge 1) {push @rg_arr, $1;}
      }
    }
    $ranges=join('',@rg_arr); 
    $ranges=~s/2//g; $ranges=~s/^.$//;
    close LOGF;
    if ($ranges) {
      @rg_arr=split '', $ranges;
      if ($#rg_arr gt 2) {
        $RC[$sc-1][$day-1]='\\begin{multicols}{2}'.
          join(' \\\\ ',@rg_arr[0..2]).'\\newpage '.
          join(' \\\\ ',@rg_arr[2..$#rg_arr]).
          '\\end{multicols}\\vspace{-13pt}';
      } else {$RC[$sc-1][$day-1]=join(' \\\\ ',@rg_arr)};
    }
  }
}


# find empty/small log files <=> no data
@DATAE=([],[],[],[]);
for ($sc=1;$sc<=4;$sc++){
  @$DATAE[$sc-1]=();
  for ($j=1;$j<=31;$j++){
    $jj=$j; $jj='0'.$j if $j <= 9;
    $logfileNS="$logfiledir/C$sc"."_$yy$mm$jj"."_B.NSlog";
    $logfileBS="$logfiledir/C$sc"."_$yy$mm$jj"."_B.BSlog";
    if (-e $logfileNS) {$Nsize= -s $logfileNS} else {$Nsize=0}
    if (-e $logfileBS) {$Bsize= -s $logfileBS} else {$Bsize=0}
    $minlogsize=255;
    if (($Nsize > $minlogsize) or ($Bsize > $minlogsize)) {
      push @{$DATAE[$sc-1]}, '';
    } else {
      print "No data on sc $sc, day $j\n";
      push @{$DATAE[$sc-1]}, '{\Huge$\times$\vspace{-1em}}';
    }
  }
}

## Read and store uncal intervals times

$cfgdir="$ENV{FGMROOT}/cfg/";
$uncalfile=$cfgdir.'uncal_'.$yy.$mm.'.txt';

@UC=([],[],[],[]);                                 
for ($sc=1;$sc<=4;$sc++){                                                
  for ($j=1;$j<=31;$j++){push @{$UC[$sc-1]}, '';}                        
}                                                                                  
if (-e $uncalfile) {
  open UCF, "<$uncalfile" or die "could not open $uncalfile";
  while (<UCF>){                                                              
    next unless /(\d)\s+(\d)\s+(\d)\s+(\d)\s+(\d)\s+(\d\d)\s+(\d\d)\s+(\d\d)/;
    @spacecraft=($1,$2,$3,$4);    
    #$range=$5;
    $day=$6;
    $hm=$7.':'.$8;
    for ($sc=1;$sc<=4;$sc++){ 
      next unless $spacecraft[$sc-1];
      $UC[$sc-1][$day-1]=$hm;
    }
  }
  close UCF;
} else {
  warn "       $uncalfile does not exist
       without it, sc offset times are not printed.";
}

$header_doc=<<HEADER_DOC;
\\documentclass[a4paper,10pt]{article}
\\usepackage[landscape]{geometry}
\\usepackage{forloop}
\\usepackage{multicol}
\\usepackage[usenames,dvipsnames]{color}
\\setlength{\\oddsidemargin}{-15mm}	
\\setlength{\\textwidth}{270mm}	
\\setlength{\\textheight}{196mm}	
\\setlength{\\topmargin}{-30mm}
\\setlength{\\parindent}{0mm}	
\\newcounter{j}
\\begin{document}
{\\bf \\large Calibration for $month 20$yy}\\bigskip\n
HEADER_DOC

$header_tab=<<'HEADER_TAB';
\begin{tabular}{r||*{4}{*{4}{c|}|}l}
\bf Day&\multicolumn{4}{c||}{\bf Cluster 1}&
        \multicolumn{4}{c||}{\bf Cluster 2}&
        \multicolumn{4}{c||}{\bf Cluster 3}&
        \multicolumn{4}{c||}{\bf Cluster 4}&\bf Day\\
 \forloop{j}{0}{\value{j} < 16}{&\rule{11.5mm}{0mm}}\\                                
 &obs&RC&sc off&HH
 &obs&RC&sc off&HH
 &obs&RC&sc off&HH
 &obs&RC&sc off&HH\\\hline
HEADER_TAB

$page1=$header_tab;
for ($dd=1;$dd<=15;$dd++) {
  $page1.="\\bf $dd\\rule{0mm}{10mm}";
  for ($sc=1;$sc<=4;$sc++){
    $page1.='&\scriptsize{\begin{minipage}[b]{8mm}
                            \tt\color{RoyalBlue}{'.$E[$sc-1][$dd-1].'}\vspace*{-2mm}
                          \end{minipage}}&
              \small{\hspace{-3.5mm}\begin{minipage}[b]{10mm}
                                    \tt\color{Rhodamine}{'.$RC[$sc-1][$dd-1].'}\vspace*{-2pt}
                                  \end{minipage}}&
              %\rule{8mm}{0.1mm}&
              \scriptsize{\begin{minipage}[b]{8mm}
              '.$DATAE[$sc-1][$dd-1].'\color{PineGreen}{'.$UC[$sc-1][$dd-1].'}\vspace*{7.4mm}
                          \end{minipage}}&';
  }
  $page1.="&\\bf $dd\\\\\\hline\n";
}
$page1=$page1.'\end{tabular}'."\n\n";


$page2=$header_tab;
for ($dd=16;$dd<=31;$dd++) {
  $page2.="\\bf $dd\\rule{0mm}{10mm}";
  for ($sc=1;$sc<=4;$sc++){
    $page2.='&\scriptsize{\begin{minipage}[b]{8mm}
                            \tt\color{RoyalBlue}{'.$E[$sc-1][$dd-1].'}\vspace*{-2mm}
                          \end{minipage}}&
              \small{\hspace{-3.5mm}\begin{minipage}[b]{10mm}
                                    \tt\color{Rhodamine}{'.$RC[$sc-1][$dd-1].'}\vspace*{-2pt}
                                  \end{minipage}}&
              %\rule{8mm}{0.1mm}&
              \scriptsize{\begin{minipage}[b]{8mm}
              '.$DATAE[$sc-1][$dd-1].'\color{PineGreen}{'.$UC[$sc-1][$dd-1].'}\vspace*{7.4mm}
                          \end{minipage}}&';
  }
  $page2.="&\\bf $dd\\\\\\hline\n";
}
$page2=$page2.'\end{tabular}'."\n\n";

open OFILE, ">$ofile";
  print OFILE $header_doc.$page1.$page2.'\end{document}';
close OFILE;

#system "pdflatex $ofile &>/dev/null"; 
system "pdflatex $ofile >/dev/null 2>&1"; 
system "xpdf $basename.pdf" if $xpdf;
unlink $basename.'.aux', $basename.'.log', $basename.'.tex';
system "mv $basename.pdf $output";

print "output to $output\n";

#-------------------------------------------------------------------------

1;

__END__

=head1 NAME

table.pl - Produces the table to be used for daily calibration.

=head1 SYNOPSIS

B<table.pl> B<--month> I<month> [B<--year> I<year>] [B<--(no)xpdf>] 
[B<--output> I<filename>] [B<--help>]

=head1 DESCRIPTION

This script produces the file F<table.pdf> containing the table to be used 
for daily calibration. Eclipses times are read from the orbit file and range 
changes from the log files. If F<uncal_mmdd.txt> exists for the given month, 
then the uncal interval times used for spacecraft offset in range two are
also included.

=head1 OPTIONS

=over 4

=item B<-m> I<month>, B<--month> I<month> 

The month. One or two digits. Mandatory argument.

=item B<-y> I<year>, B<--year> I<year>

The year. One or two digits. Default is the current year.

=item B<-x>, B<--xpdf>

Show the result with xpdf viewer. Default is yes. To disable, negate the option 
with B<--noxpdf>

=item B<-o> I<filename>, B<--output> I<filename>

Specify the output file. If the '.pdf' extension is not given, it will be 
automatically aded. Default is $FGMROOT/tmp/table.pdf . 

=item B<-h>, B<-?>, B<--help>

Prints a brief help message.

=back

=head1 ENVIRONMENT

=over 4

=item FGMROOT

Root for the FGM calibration directory structure. Default to F</home/FGM/> if
not set.

=back

=head1 FILES

F<$FGMROOT/log/atorb/stef.cl> - orbit file

F<$FGMROOT/log/cd_log/$yy_$mm/C$sc_$yy$mm??_B.NSlog> - log files

F<$FGMROOT/cfg/uncal_yymm.txt> - uncal intervals

F<table.pdf> - output

=head1 AUTHOR

Dragos Constantinescu <d.constantinescu@tu-bs.de>

=cut


