/*
 * Copyright (C) 1995/96 IGM TU Braunschweig
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "ddshrm"  
#define VERSION "5.0 (1996-12-18)"
#define PURPOSE "removes the DDS packet headers of telemetry data files"
#define USAGE "USAGE:

 ... | ddshrm [-i <int>] | ...

 The program removes the DDS packet headers from a telemetry file by reading it
 from stdin and writing the same file without the DDS packet headers to stdout."

#define OPTIONS "OPTIONS:

  -i <int>  if specified, the program expects the DDS packet headers to have
            a data source/type ID of <int> (a value between 0 and 255).
            
  -V        print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Stephan Buchert       (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke      (joerg@igpp.ucla.edu)"


#include "libutil.h"


static char *errmsg[] =
  {
/*  0 */  "Type `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Illegal usage.\n",
/*  3 */  "ERROR in %s: Illegal value of data source/type id specified.\n",
/*  4 */  "ERROR in %s: Cannot allocate enough memory for input data.\n",
/*  5 */  "ERROR in %s while reading science data.\n",
/*  6 */  "ERROR in %s: DDS packet header id is %d, expected %d.\n",
/*  7 */  "ERROR in %s while writing science data.\n",
  };


static void read_telstream(FILE *fp, char **telstream, int length)
/*----------------------------------------------------------------------------- 
 * First the function `read_telstream' allocates new memory, if the actual    *
 * memory size is different from the data length to be read. Then the data    *
 * are read. If the memory cannot be allocated or an error occurs during      *
 * reading of the data, the program will be exited with the corresponding     *
 * exit code.                                                                 *
 -----------------------------------------------------------------------------*/
{
 static int telsize;

 if (telsize != length)
   {
    if (*telstream) 
       free(*telstream);
    telsize = length;
    if ((*telstream = malloc(telsize)) == NULL)
      {
       fprintf ( stderr, errmsg[4], PROGRAM );
       exit(4);
      }
   }
 if (fread(*telstream, 1, telsize, fp) != telsize)
   {
    fprintf ( stderr, errmsg[5], PROGRAM );
    exit(5);
   }
}


int main ( int argc, char *argv[] )
{
 ddshead_t   head;
 int         id=-1;
 static int  i=1, length;
 char        copt, *telstream = 0;

 /* evaluation of the command line parameters */
 while (i < argc)
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch(copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 'i':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               {
                i++;
                id = atoi ( argv[i] );
               }
             else if ( *(argv[i]+2) != '\0' )
                id = atoi ( argv[i]+2 );
             else
               {
                fprintf ( stderr, errmsg[2], PROGRAM );
                fprintf ( stderr, errmsg[0], argv[0] );
                exit(2);
               }
             if ( (id < 0) || (id > 255) )
               {
                fprintf ( stderr, errmsg[3], PROGRAM );
                exit(3);
               }
             break;

          case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[2], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(2);
      }
    i++;
   }
  
 /* read the header bytes until end of byte stream */
 while (read_ddshead(stdin, &head) == header_length)
   {
    if ( ( id >= 0 ) && ( (int) head.source != id ) )
      {
       fprintf ( stderr, errmsg[6], PROGRAM, (int) head.source, id );
       exit(6);
      } 

    /* science packet data length */
    length = pktlen(&head);

    /* read and write the science data block */
    read_telstream(stdin, &telstream, length);
    if (fwrite(telstream, 1, length, stdout) != length)
      {
       fprintf ( stderr, errmsg[7], PROGRAM );
       exit(7);
      }
   }
 exit(0);
}

