/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching
 * modified version 5.0 (1996-12-18)
 * timesp < - > timespec_t AlphaDEC/SUN, libutil.h 
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "ddsls"  
#define VERSION "6.0 (2000-04-15)"
#define PURPOSE "extract informations from the DDS packet headers of telemetry data"
#define USAGE "USAGE:

 ddsls [-t <int>] [-a]  [-f <file-name>] [-i <cdrom-name>] [list] | ...

 The program reads the DDS packet headers from telemetry files given in list
 (default stdin) and writes some fundamental informations in ASCII code to 
 stdout."

#define OPTIONS "OPTIONS:

  -t     print the time information in the following form 
         <int> = 0 : ISO standard time string like '1996-12-02T02:05:15.398Z'
         <int> = 1 : (int) seconds of the UNIX epoch, and (int) nanoseconds
         <int> = 2 : (float) seconds of the hour
         <int> = 3 : (float) hours of the day
         <int> = 4 : character string like 'Mon Dec 02 02:05:15 1996'

  -a     print the time information of every packet even if these are
         science or housekeeping data 

  -f     write <file-name> as input, if data are coming from stdin.
         This can be used, if the input is derived from merged files
         specified by one filename using 'wild cards'

  -i     write <cdrom-name> as input if data are coming from stdin
         this can be used, if the input is derived from merged files of one
         CD-ROM

  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@geophys.nat.tu-bs.de)"

static char *errmsg[] =                           /* error messages */
   {
/*  0 */  "Use '%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Cannot allocate enough memory for the file list.\n",
/*  3 */  "ERROR in %s: Incorrect choice of time representation.\n",
/*  4 */  "ERROR in %s: Could not open input file %s.\n",
/*  5 */  "ERROR in %s: Cannot allocate enough memory for input data.\n",
/*  6 */  "ERROR in %s while reading of science data.\n",
   };


#include "libutil.h"


static void read_telstream(FILE *fp, char **telstream, int length)
/*----------------------------------------------------------------------------- 
 * First the function 'read_telstream' allocates new memory, if the actual    *
 * memory size is different from the data length to be read. Then the data    *
 * are read. If the memory cannot be allocated or an error occurs during      *
 * reading of the data, the program will be exited with the corresponding     *
 * exit code.                                                                 *
 -----------------------------------------------------------------------------*/
{
 static int telsize;

 if (telsize != length)
   {
    if (*telstream) 
       free(*telstream);
    telsize = length;
    if ((*telstream = malloc(telsize)) == NULL)
      {
       fprintf ( stderr, errmsg[5], PROGRAM );
       exit(5);
      }
   }
 if (fread(*telstream, 1, telsize, fp) != telsize)
   {
    fprintf ( stderr, errmsg[6], PROGRAM );
    exit(6);
   }
}


static void hyphenline()
/*-----------------------------------------------------------------------------
 * The function 'hyphenline' prints a line of hyphens for clarity.            *
 * The first character is a '%' to indicate that this is a comment line.      *
 -----------------------------------------------------------------------------*/
{
 printf("%%------------------------------------------------------------------------------\n");
}


static void doublehyphenline()
/*-----------------------------------------------------------------------------
 * The function 'doublehyphenline' prints a line of doublehyphens for clarity.*
 * The first character is a '%' to indicate that this is a comment line.      *
 -----------------------------------------------------------------------------*/
{
 printf("%%==============================================================================\n");
}


static void write_type(int type, int sc)
/*-----------------------------------------------------------------------------
 * The function 'write_type' writes the type of the data.                     *
 -----------------------------------------------------------------------------*/
{
 int          s, t, u;
 static char  *msg[] =  
   {
/*  0 */          "MASTER CATALOGUE OF ALL CLUSTER DATA AVAILABLE",
/*  1 */          "LONG TERM ORBIT FILE ENTRY", 
/*  2 */          "LONG TERM EVENT FILE ENTRY",
/*  3 */          "SHORT TERM ORBIT FILE ENTRY",
/*  4 */          "SHORT TERM EVENT FILE ENTRY",
/*  5 */          "SPACECRAFT ATTITUDE AND SPIN RATE ENTRY",
/*  6 */          "TIME CALIBRATION FILE ENTRY",
/*  7 */          "COMMAND HISTORY FILE ENTRY",
/*  8 */          "HOUSEKEEPING PARAMETER DEFINITION FILE FOR",
/*  9 */          "NORMAL SCIENCE DATA FOR",
/* 10 */          "BURST SCIENCE DATA FOR",
/* 11 */          "HOUSEKEEPING DATA FOR",
/* 12 */          "HOUSEKEEPING PARAMETER DEFINITION FILE FOR"
   };
 static char *exp[] =
   {
/*  0 */          "EDI",
/*  1 */          "FGM",
/*  2 */          "CIS",
/*  3 */          "PEACE",
/*  4 */          "RAPID",
/*  5 */          "WEC",
/*  6 */          "ASPOC"
   };

 if (type <= 7)
    printf("D %d>Data type: %s\n", type, msg[type]);
 else if (type <= 14)
    printf("D %d>Data type: %s %s ON CLUSTER %d (OLD TYPE ID)\n",
           type, msg[8], exp[type - 8], sc);
 else if (type == 15)
    printf("D %d>Data type: %s CLUSTER %d PLATFORM (OLD TYPE ID)\n",
           type, msg[8], sc);
 else if (type <= 171)
   {
    u = type - 30;
    s = u / 40 + 1;
    u %= 40;
    t = u / 7;
    u %= 7;
    if (t <= 2)
       printf("D %d>Data type: %s %s ON CLUSTER %d\n",
              type, msg[9 + t], exp[u], s);
    else if ((t == 3) && (u == 0))
       printf("D %d>Data type: %s CLUSTER %d PLATFORM\n", type, msg[11], s);
    else
       printf("D -1>Data type: UNKNOWN DATA TYPE ID %d\n", type);

    if (s != sc)
       printf("X 0>Error: S/C ID (S/C = %d) DOES NOT MATCH DATA TYPE (S/C = %d)", 
              sc, s);
   }
 else if((type >= 200) && (type <= 231))
   {
    u = type - 200;
    s = u / 8 + 1;
    u %= 8;
    if (u <= 6)
       printf("D %d>Data type: %s %s ON CLUSTER %d\n",
              type, msg[12], exp[u], s);
    if (u == 7)
       printf("D %d>Data type: %s CLUSTER %d PLATFORM\n", type, msg[12], s);

    if (s != sc)
       printf("X 0>Error: S/C ID (S/C = %d) DOES NOT MATCH DATA TYPE (S/C = %d)", 
              sc, s);
   }
 else
    printf("D -1>Data type: UNKNOWN DATA TYPE ID %d\n", type);
}


static void write_gs(int gs)
/*-----------------------------------------------------------------------------
 * The function 'write_gs' writes the name of the ground station.             *
 -----------------------------------------------------------------------------*/
{
 switch(gs)
   {
    case  0: 
       printf("R 0>Ground station: NOT SPECIFIED\n");
       break;
    case  1: 
       printf("R 1>Ground station: ODENWALD\n");
       break;
    case  2: 
       printf("R 2>Ground station: REDU\n");
       break;
    case  3: 
       printf("R 3>Ground station: KOUROU\n");
       break;
    case  4: 
       printf("R 4>Ground station: PERTH\n");
       break;
    case  5: 
       printf("R 5>Ground station: MALINDI\n");
       break;
    case  6: 
       printf("R 6>Ground station: CANBERRA\n");
       break;
    case  7: 
       printf("R 7>Ground station: GOLDSTONE\n");
       break;
    case 15: 
       printf("R 15>Ground station: N/A\n");
       break;
    default: 
       printf("R -1>Ground station: UNKNOWN GROUND STATION %d\n", gs);
       break;
   }
}


static void write_stream(int stream)
/*-----------------------------------------------------------------------------
 * The function 'write_stream' writes the Virtuell Channel of the data stream.*
 -----------------------------------------------------------------------------*/
{
 switch(stream)
   {
    case 0x00: 
       printf("V %d>Data stream: %02X hex = REAL-TIME VC0\n", stream, stream);
       break;
    case 0x02: 
       printf("V %d>Data stream: %02X hex = REAL-TIME VC2\n", stream, stream);
       break;
    case 0x03: 
       printf("V %d>Data stream: %02X hex = REAL-TIME VC3\n", stream, stream);
       break;
    case 0x40: 
       printf("V %d>Data stream: %02X hex = PLAYBACK VC0\n", stream, stream);
       break;
    case 0x42: 
       printf("V %d>Data stream: %02X hex = PLAYBACK VC2\n", stream, stream);
       break;
    case 0x43: 
       printf("V %d>Data stream: %02X hex = PLAYBACK VC3\n", stream, stream);
       break;
    case 0xf0: 
       printf("V %d>Data stream: %02X hex = RECALL VC0\n", stream, stream);
       break;
    case 0xf2: 
       printf("V %d>Data stream: %02X hex = RECALL VC2\n", stream, stream);
       break;
    case 0xf3: 
       printf("V %d>Data stream: %02X hex = RECALL VC3\n", stream, stream);
       break;
    case 0x4f: 
       printf("V %d>Data stream: %02X hex = RECALL PLAYBACK VC0\n", 
              stream, stream);
       break;
    case 0xe2: 
       printf("V %d>Data stream: %02X hex = RECALL PLAYBACK VC2\n", 
              stream, stream);
       break;
    case 0xe3: 
       printf("V %d>Data stream: %02X hex = RECALL PLAYBACK VC3\n",
              stream, stream);
       break;
    case 0xff: 
       printf("V %d>Data stream: %02X hex = N/A\n", stream, stream);
       break;
    default:   
       printf("V -1>Data stream: UNKNOWN DATA STREAM %02X hex\n", stream);
       break;
   }
}


static void write_timecal(int tc)
/*-----------------------------------------------------------------------------
 * The function 'write_timecal' writes the mode of the time calibration.      *
 -----------------------------------------------------------------------------*/
{
 switch(tc)
   {
    case 0:  
       printf("C %d>Time calibration: ACTUAL TIME\n", tc);
       break;
    case 1:  
       printf("C %d>Time calibration: EXTRAPOLATED TIME\n", tc);
       break;
    case 2:  
       printf("C %d>Time calibration: CONTINGENCY TIME\n", tc);
       break;
    default: 
       printf("R -1>Time calibration: UNKNOWN TIME CALIBRATION %d\n", tc);
       break;
   }
}


static void write_t(char id, int timeform, timesp t)
/*----------------------------------------------------------------------------- 
 * The function 'write_t' writes the time according to the choosen format.    *
 -----------------------------------------------------------------------------*/
{
 char  ca[32], *cp=&ca[0];

 switch(timeform)
   {
    case 0:
       get_time_str(&t, &ca[0]);
       printf("%c %s\n", id, ca);
       break;
    case 1: 
       printf("%c %9d %9d\n", id, (int)t.tv_sec, t.tv_nsec);
       break;
    case 2: 
       printf("%c % f\n", id, (t.tv_sec % 3600) + 1.e-9 * t.tv_nsec);
       break;
    case 3: 
       printf("%c % f\n", id, ((t.tv_sec % 86400) + 1.e-9 * t.tv_nsec) / 3600.);
       break;
    case 4: 
       cp = asctime(gmtime(&t.tv_sec));
       printf("%c %s", id, cp);
       break;
   }
}


static void do_event(int sci_hk,
                     int all,
                     int *event_flag, 
                     int *write_time_flag, 
                     int timeform, 
                     timesp t)
/*----------------------------------------------------------------------------- 
 * The function 'do_event' is called, if a parameter has changed. At first it *
 * sets the indicating flag 'event_flag'. Then it sets the write_time_flag,   *
 * if it has not been set before. This flag is used to avoid multiple time    *
 * writing. After this the time is written, indicating the end of a data      *
 * block with the same parameter settings. This is only sensible for science  *
 * or housekeeping data.                                                      *
 -----------------------------------------------------------------------------*/
{
 *event_flag = 1;
 if (!(*write_time_flag))
   {
    *write_time_flag = 1;
    if (sci_hk && !all)
       write_t('E', timeform, t);
    hyphenline();
   }
}


int main ( int argc, char *argv[] )
{
 FILE               *in_file = stdin;
 ddshead_t          head;
 static int         i = 1, j, k, *file_index, timeform,
                    type_ID[2], length[2], sc_ID[2], gs_ID[2], 
                    stream[2], time_cal[2], TM_acq_mode[2],
                    all, event_flag, write_time_flag = 1, sci_hk, missing;
 static timesp  t[2], time_diff; 
 char               copt, cdromname[9], filename[128], *telstream = 0;
 static double      reset_diff, reset_period = 5.15222168;

/*----------------------------------------------------------------------------- 
 * Allocate memory for the input_file_index array.                            *
 -----------------------------------------------------------------------------*/
 if (file_index)
    free(file_index);
 if ((file_index = (int *) calloc(argc, sizeof(int))) == NULL)
   {
    fprintf ( stderr, errmsg[2], PROGRAM );
    exit(2);
   }

/*----------------------------------------------------------------------------- 
 * Set the name of the input CD-ROM and the input filename to empty.          *
 -----------------------------------------------------------------------------*/
 cdromname[0] = '\0';
 filename[0] = '\0';

/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 while ( i < argc )
   {
    if (*argv[i] == '-')                           /* Is it a program option? */
      {
       copt = *(argv[i]+1);
       switch(copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n",
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 't': 
             if (*(argv[i]+2) == '\0' && i+1 < argc)
               { 
                i++;
                timeform = atoi(argv[i]);
               }
             else
                timeform = atoi(argv[i]+2);
             if (timeform < 0 || timeform > 4)
               {
                fprintf ( stderr, errmsg[3], PROGRAM );
                exit(3);
               }
             break;

          case 'a': 
             all = 1;
             break;

          case 'f':
             if (*(argv[i]+2) == '\0' && i+1 <= argc)
               {
                i++;
                strcpy ( filename, argv[i] );
               }
             else
                strcpy ( filename, argv[i]+2 );
             break;

          case 'i':
             if (*(argv[i]+2) == '\0' && i+1 <= argc)
               {
                i++;
                strcpy ( cdromname, argv[i] );
               }
             else
                strcpy ( cdromname, argv[i]+2 );
             break;

          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);
          
          default: 
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else                                          /* Is it a input file name? */
      {
       *(file_index + j) = i;
       j++;
      }
    i++;
   }
  
 do 
   {
    if (j != 0)
/*----------------------------------------------------------------------------- 
 * Open in_file, if input file name(s) is(are) given in the command line and  *
 * print a double hyphenline and the new filename. Reset 't[0].tv_sec' and    *
 * 'write_time_flag', so that the complete header information of the first    *
 * record of the new file will be printed.                                    *
 -----------------------------------------------------------------------------*/
      {
       in_file = fopen ( argv[*(file_index + k)], "r" );
       if ( in_file == NULL )
         {
          fprintf ( stderr, errmsg[4], PROGRAM, argv[*(file_index + k)] );
          exit(4);
         }
       doublehyphenline();
       printf ("I %d>Input file %d: %s\n", 
               k + 1, k + 1, argv[*(file_index + k)]);
       t[0].tv_sec = 0;
       write_time_flag = 1;
      }
    else
/*----------------------------------------------------------------------------- 
 * Print a double hyphenline and a note that data is coming from stdin.       *
 -----------------------------------------------------------------------------*/
      {
       doublehyphenline();
       if (filename[0] != '\0')
          printf ("I 0>Input data are from file: %s\n", filename );
       else if (cdromname[0] != '\0')
          printf ("I 0>Input data are from CD-ROM: %s\n", cdromname);
       else
          printf ("I 0>Input data are coming from stdin.\n");
      }

/*----------------------------------------------------------------------------- 
 * Reading the header bytes until end of byte stream.                         *
 -----------------------------------------------------------------------------*/
    while (read_ddshead(in_file, &head) == header_length)
      { 

/*----------------------------------------------------------------------------- 
 * Determination of the type ID. Test if these are science or housekeeping    *
 * data.                                                                      * 
 -----------------------------------------------------------------------------*/
       type_ID[1] = head.source;
       if (type_ID[1] >= 30 && type_ID[1] <= 171)
          sci_hk = 1;
       else
          sci_hk = 0;

/*----------------------------------------------------------------------------- 
 * Determination of the spacecraft ID.                                        * 
 -----------------------------------------------------------------------------*/
       sc_ID[1] = scid(head.ID) + 1;
       if (sc_ID[1] != sc_ID[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          printf ("S %d>S/C number %d\n", sc_ID[1], sc_ID[1]);
         }

/*----------------------------------------------------------------------------- 
 * Writing of the type ID. This should be performed after the writing of the  *
 * S/C ID, whereas the determination of the type ID must be done before       *
 * the S/C ID can be written, because 'sci_hk' is used for 'do_event'.        *
 -----------------------------------------------------------------------------*/
       if (type_ID[1] != type_ID[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          write_type(type_ID[1], sc_ID[1]);
         }
     
/*----------------------------------------------------------------------------- 
 * Determination of the science packet data length.                           * 
 -----------------------------------------------------------------------------*/
       length[1] = pktlen(&head);
       if (length[1] != length[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          printf ("L %d>Data packet length: %d bytes\n", length[1], length[1]);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the ground station ID.                                    * 
 -----------------------------------------------------------------------------*/
       gs_ID[1] = gsid(head.ID);
       if (gs_ID[1] != gs_ID[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          write_gs(gs_ID[1]);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the data stream.                                          * 
 -----------------------------------------------------------------------------*/
       stream[1] = head.VC;
       if (stream[1] != stream[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          write_stream(stream[1]);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the time calibration.                                     * 
 -----------------------------------------------------------------------------*/
       time_cal[1] = timecal(head.TC);
       if (time_cal[1] != time_cal[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          write_timecal(time_cal[1]);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the TM acquisition mode.                                  * 
 -----------------------------------------------------------------------------*/
       TM_acq_mode[1] = TMacqmode(head.TC);
       if (TM_acq_mode[1] != TM_acq_mode[0] || t[0].tv_sec == 0)
         {
          do_event(sci_hk, all, &event_flag, &write_time_flag, timeform, t[0]);
          printf ("A %d>Telemetry acquisition mode: %d\n",
                  TM_acq_mode[1], TM_acq_mode[1]);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the packet time stamp to detect data gaps. Data gaps will *
 * only be indicated, if these are science or housekeeping data.              *
 -----------------------------------------------------------------------------*/
       t[1] = cds2unix(head.t);
       if (t[0].tv_sec != 0 && sci_hk)
         {
          time_diff = subtime(&t[1], &t[0]);
          reset_diff = time_diff.tv_sec + 1.e-9 * time_diff.tv_nsec;
          if (reset_diff >= 1.5 * reset_period)
            {
             missing = (int)(reset_diff / reset_period - 0.5);
             do_event(sci_hk, all, &event_flag, &write_time_flag,
                      timeform, t[0]);
             if ( missing == 1 )
                printf ("G %d> Data gap: %d data block missing\n", 
                        missing, missing);
             else
                printf ("G %d> Data gap: %d data blocks missing\n", 
                        missing, missing);
            }
         }
          
/*----------------------------------------------------------------------------- 
 * Write the vector time and reset the write_time_flag, if the event flag is  *
 * set and the vector has no error bits set, ie if one ore more parameters    *
 * have changed or a period of faulty vectors has finished.                   *
 * Thus, this time indicates the start of a data block with no errors and the *
 * same parameter settings.                                                   *
 -----------------------------------------------------------------------------*/
       if (event_flag)
         {
          hyphenline();
          if (sci_hk && !all)
             write_t('B', timeform, t[1]);
          write_time_flag = 0;
         }

/*----------------------------------------------------------------------------- 
 * If these are not science or housekeeping data or if the 'all'-flag has     *
 * been set, the time will be written for each data record.                   *
 -----------------------------------------------------------------------------*/
       if (!sci_hk || all)
          write_t('T', timeform, t[1]);

/*----------------------------------------------------------------------------- 
 * Set the old parameters to the new ones. Reset the event flag.              *
 -----------------------------------------------------------------------------*/
       sc_ID[0] = sc_ID[1];
       type_ID[0] = type_ID[1];
       length[0] = length[1];
       gs_ID[0] = gs_ID[1];
       stream[0] = stream[1];
       time_cal[0] = time_cal[1];
       TM_acq_mode[0] = TM_acq_mode[1];
       t[0] = t[1];
       event_flag = 0;
          
/*----------------------------------------------------------------------------- 
 * Read the science data block.                                               *
 -----------------------------------------------------------------------------*/
       read_telstream(in_file, &telstream, length[1]);

      }
/*----------------------------------------------------------------------------- 
 * Write the last time as the end time of the last data block.                *
 -----------------------------------------------------------------------------*/
    if (sci_hk && !all)
       write_t('E', timeform, t[0]);
    doublehyphenline();

/*----------------------------------------------------------------------------- 
 * Close in_file, if input file name(s) is(are) given in the command line.    *
 -----------------------------------------------------------------------------*/
    if (j != 0)
       fclose(in_file);
    k++;
   }
 while (j > k);
 exit(0);
}
