/* 
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching
 * V6.0 (2000-04-15), rewritten, new calfile format, added check_bad
 * gmtime instead of cftime in datestr(timesp *)
 * struct dirent instead of dirent_h (#include dirent.h)
 * options slightly changed '-i', instead of '-x' with default name 'cal.log'
 * V6.1 replaced exit(26) with continue for errmsg[7] and error with warning
 * V6.2 changes calname at midnight & saves calnames, separator=";"
 * V6.3 2001-06-10 - dropped calpath from calnames and suppressed warning for range 6
 * V6.4 2007-07-27 - multi-line strings discarded (not supported by gcc version > 3)
 * V6.5 2009-05-25 - processes range 6 data using the parameters in column 5 of *.fgmcal
 * V6.6 2009-11-25 - calibrates range 7 using "C#_range7.fgmcal" //200911
 * V6.7 2009-12-11 - faster (reads only once the default range7 parameters)
 * V6.8 2010-11-10 - bug in cal_fgm discarded: warning for first vector in range 7
 */

#include <dirent.h>
#include "libutil.h"

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmcal"  
#define VERSION "6.8 (2010-11-10)" //201011
#define PURPOSE "apply calibration to CLUSTER FGM data"
#define USAGE "USAGE:\n\
\n\
 ... | fgmcal [-a] [-c <calfile>] [-i <infofile>] | ...\n\
\n\
 The program reads CLUSTER FGM data (in fgmtvec_t format) from stdin,\n\
 applies the calibration, and writes them to stdout.\n\
\n\
 Unless explicitely specified, calibration information is searched in the\n\
 directory specified by the environment variable FGMPATH. The highest version\n\
 of daily calfile is chosen. If no daily calfile then ground calfile is used"

#define OPTIONS "OPTIONS:\n\
\n\
  -a     process `all' data, i.e., process data even if they are marked `BAD'.\n\
         Default is to process only valid data\n\
\n\
  -c     use <calfile> as calibration file.\n\
         Default is to search for a daily calibration file and --if not\n\
         found-- for a ground calibration file, using the standard names,\n\
         in the directory specified by the environment variable FGMPATH.\n\
\n\
  -i     store the name(s) of the used calibration file(s) in <infofile>.\n\
         If '-i', but no name given, use default name: 'cal.log'.\n\
         Default is not to store the name(s).\n\
\n\
  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS: \n\
\n\
   Edita Georgescu      (eg@mps.mpg.de) \n\
   Karl-Heinz Fornacon  (k-h.fornacon@tu-bs.de) \n\
   Stephan Buchert      (scb@stelab.nagoya-u.ac.jp) "

#include <dirent.h>
#include "libutil.h"

static char *errmsg[] =
{
/*  0 */ "Use '%s -h' for help.\n",
/*  1 */ "ERROR in %s: Illegal option %s.\n",
/*  2 */ "ERROR in %s: Illegal usage.\n",
/*  3 */ "ERROR in %s: Could not open calibration file %s.\n",
/*  4 */ "ERROR in %s: File %s is a %s, not a %s calibration file.\n",
/*  5 */ "ERROR in %s while reading table of default values from %s calfile.\n",
/*  6 */ "ERROR in %s while reading entry from %s calfile.\n",
/*  7 */ "ERROR in %s: Could not apply calibration; wrong status?\n",
/*  8 */ "ERROR in %s while writing to stdout.\n",
/*  9 */ "ERROR in %s: Input data not in FGM coordinate system \n",
};

static char *warning[] =
{
/*  0 */ "%s WARNING: Could not open file %s for storing calfile name(s).\n",
/*  1 */ "%s WARNING: Environment variable %s is not set.\n",
/*  2 */ "%s WARNING: Could not apply calibration; wrong range? \n",
};

static matrix cal_m;
static double calm7[12]={0., 0., 0., 1., 0., 0., 0., 1., 0., 0., 0., 1.};    //200911

char *get_calname(const time_t *clock, int scid, const char *calpath, int *iret)
/* Return a cal file name "C#_yyyymmdd_Vvv.fgmcal"
 * in the directory specified by the environment variable "FGMPATH".
 * Input:  clock       time, determines <yyyymmdd>
 *         scid        S/C id, determines #
 * The file name with alphabetically largest <vv> is returned.
 * If no existing file is found, vv="-1"
 * Output: iret = 10 cannot open directory
 *                11 no file exists, taking ground calibration file
 * Returns: full path name in static buffer
 ------------------------------------------------------------------------*/
{
  static char cbuf[MAXPATHLEN];
  char *cp, *posp;
  char scch;
  char vv[2];
  int found = 0;
  DIR  *dp;
  struct dirent *ep;

  *iret = 0;
  if( (dp = opendir(calpath))==0 )
    {
      fprintf(stderr, "cannot open directory %s\n", calpath);
      *iret = 10;
      return 0;
    }
  strcpy(cbuf, calpath);
  posp = cbuf + strlen(cbuf);
  scch = "1234"[scid];
  cp = datestr(clock);
  while( (ep = readdir(dp))!=0 )
    {
      if( strlen(ep->d_name) < 20 )
	continue;
      if( strncmp("C", ep->d_name, 1)!=0 )
	continue;
      if( scch!=ep->d_name[1] )
	continue;
      if( strncmp(cp, &ep->d_name[3], 8)!=0 )
	continue;
      if( strncmp(".fgmcal", &ep->d_name[15], 7)!=0 )
	continue;
      if( found )
	{
	  if( strncmp(vv, &ep->d_name[13], 2)<0 )
	    {
	      strncpy(vv, &ep->d_name[13], 2);
	      strcpy(posp, ep->d_name);
	    }
	}
      else
	{
	  strncpy(vv, &ep->d_name[13], 2);
	  strcpy(posp, ep->d_name);
	  found = 1;
	}
   }
  closedir(dp);

  if( !found )
    {
      sprintf(cbuf + strlen(cbuf), "C%c.fgmcal", scch);
      *iret = 11;
    }
  
  return cbuf;
}

/*---------------------------------------------------------------------------*/
int get_calmatrix(char *calname, matrix cal_m, int *adc, char *adctill)
/*  
 * open calfile='calname'
 * reads calibration matrix (cal_m), adc_nr and time till adc changed
 * close calfile
 * returns 0 on succes, -1 if wrong calname or read-error
 ----------------------------------------------------------------------------*/

{
  int i,ji;
  char com[45],stri[15];
  double dv0,dv1,dv2,dv3,dv4;
  FILE *calf;

if ( (calf = fopen(calname, "r"))==NULL){
   fprintf(stderr,"file %s not found\n",calname);
   return(-1);
   }
 
  fgets(com,25,calf);
  if ((ji=sscanf(com, "%d%s",&i,stri)) < 1){
      fprintf(stderr,"calibration input error\n");
      return(-1);
      }
  *adc=i;
  strcpy(adctill,stri);
  for (i = 0; i < 48; i++){
    if ((ji=fscanf(calf, "%lf%lf%lf%lf%lf%s",&dv0,&dv1,&dv2,&dv3,&dv4,com))==0){
      fprintf(stderr,"calibration input error\n");
      return(-1);
      }
    cal_m[i][0]=dv0;
    cal_m[i][1]=dv1;   
    cal_m[i][2]=dv2;
    cal_m[i][3]=dv3;   
    cal_m[i][4]=dv4;
   /* printf("%lf %lf %lf %lf %lf \n",cal_m[i][0],cal_m[i][1],cal_m[i][2],
                                cal_m[i][3],cal_m[i][4]); */
    }    
  if( calf != NULL )
    fclose(calf);
  return(0);
}

/*----------------------------------------------------------------------------*/
int cal_fgm(fgmtvec_t *fv, matrix cal_m)
/*  
 *  evaluates status & applies corresponding calibration matrix 
 *  updates status: coord.system is now FSR
 *  returns 0 on succes, -1 if incorrect status 
  ----------------------------------------------------------------------------*/
{
  double bp[3];
  int jj,indo=0,indm=0,rnge=0,adcu=0,sensr=0;
  static int sta=0,ex=0,reg=0,rng[8] = { 7, 7, 0, 1, 2, 3, 4, 5 };
 // static int sta=0,ex=0,reg=0,rng[8] = { 5, 5, 0, 1, 2, 3, 5, 4 }; changed to deal with range 6 2009/05/05 by eg

  for( jj=0;jj<3;jj++)
     bp[jj] = 0.0;
  if (sta != fv->stat){
     sta=fv->stat;
     rnge=subbits(sta,4,7);          /* 2,3,4,5,{6,7} ==> 64,256,1024,4096,{16384,65000}*/
     sensr=get_bit(sta,7);            /* OB=0, IB=1 */
     adcu=get_bit(sta,8);             /* ADC1=0, ADC2=1 */
     ex=2*adcu+sensr;
     reg=rng[rnge];
     if (reg==7) return(-1);
     }
  for (jj = 0; jj < 3; jj++)
      bp[jj] = (double)(fv->b[jj]); 

  for (jj = 0; jj < 3; jj++) {
      indo = 12*ex + jj ;
      indm = 12*ex + 3 * (jj+1);
                           
 /* apply scale-orth-rot matrix */
      fv->b[jj] = (float)(cal_m[indm][reg]* bp[0] + cal_m[indm+1][reg]* bp[1] \
                         + cal_m[indm+2][reg]* bp[2]);                        
 /* substract offset */
      fv->b[jj] -= (float) cal_m[indo][reg] ;
  //  fprintf(stderr,"%4d %lf \n",  reg, ( float) cal_m[indo][reg]);   
      }

 /* Coordinate system is now FSR: */
  fv->stat |= 1<<16;

return(0);
} 
/* ------------------------------------------------------------------------ */

int main ( int argc, char *argv[], char *envp[] )
  {
   FILE               *strfilep=NULL;  //*inf=NULL,
   int                i, fvsz, len[2]={0,0}, all=0, level=0, iret; 
   char               copt,adctill[20]="",strname[128]="",calpath[96]="",
                      calname[128]="",calnames[300]="",clna[30]="",*p_caln="\0";
   fgmtvec_t          fv;
   unsigned int       day=0,lday=0,scid=255,adc=0,lscid=255;
   timesp             t0={0,0}, tillt={0,0};
     //200911
     FILE *calf7=NULL;
     static char caln7[MAXPATHLEN]="";
     int ji=0;
     char scch;
     double dv0;

  
   fvsz = sizeof(fgmtvec_t);
   strcpy(calname, "\0");

/* evaluate command line parameters */

 i = 1;
 while ( i < argc )
   {
   copt = *(argv[i]);
   if ( copt == '-' )
      {
      copt = *(argv[i]+1);
      switch(copt)
        {
         case 'h':
#ifndef SECURE
              fprintf ( stderr, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                         PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
              fprintf ( stderr, "%s\n\n %s %s\n\n%s\n",\
                         PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
 exit(0);

         case 'i': 
              if (*(argv[i]+2) == '\0' && i+1 <= argc)
                 {
                 if ( i+1 < argc && *argv[i+1] != '-' )
                    {
                    i++;
                    strcpy ( strname, argv[i] );
                    }
                 }
                 else
                 strcpy ( strname, argv[i]+2 );
              if (strlen(strname) < 2)
                 strcpy( strname,"cal.log" );
              strfilep = fopen ( strname, "w" );
              if ( strfilep == NULL )
                 fprintf ( stderr, warning[0], PROGRAM, strname );
              break;

         case 'c':
              if (*(argv[i]+2) == '\0' && i+1 <= argc)
                 {
                 i++;
                 strcpy ( calname, argv[i] );
                 }
                 else
                 strcpy ( calname, argv[i]+2 );
              len[0] = strlen ( calname );
              strcat(calnames,calname);
              level = 3;
              break;
         case 'a':
              all = 1;
              break;
 
         case 'V':
              fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
              exit(0);

         default:
              fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
              fprintf ( stderr, errmsg[0], argv[0] );
              exit(1);
         }
     }
     else
     {
     fprintf ( stderr, errmsg[2], PROGRAM );
     fprintf ( stderr, errmsg[0], argv[0] );
     exit(2);
     }
   i++;
   }

/* evaluate environment variables */
 calpath[0] = '\0';
 for (i = 0; envp[i] != NULL; i++)
     if (!strncmp(envp[i], "FGMPATH", 7))
        {
        strcpy(calpath, getenv("FGMPATH"));
        if (strncmp("/",&calpath[strlen(calpath)-1],1) != 0)
            strcat(calpath, "/");
        break;
        }
 if (strlen(calpath) < 2)
      strcat(calpath, "./");
 len[1] = strlen ( calpath );     
 putenv("TZ=UTC0");

/* read input vectors */
 while(read_fgmtvec(stdin,&fv) == fvsz) {             //inf <-> stdin
   if ( !all && check_bad(fv.stat) )
      continue;
/* check the coordinate system: */
   if ( subbits(fv.stat, 16, 19)!=0 )
      {
      fprintf(stderr, errmsg[4], PROGRAM);
      exit(24);
      }    
/* date: */
   day = fv.tv.tv_sec / 86400;
   t0.tv_sec = day*86400;
   t0.tv_nsec = 0;
/* spacecraft ID: */
   scid = subbits(fv.stat, 30, 32);
/* check whether we have the right cal matrix */
  if ( (day != lday) || (scid != lscid) ) 
    { 
    lday=day;
    lscid=scid;
/* define calname if not explicitely specified */
    if (level != 3)
       {
        p_caln = get_calname( &fv.tv.tv_sec, scid, calpath, &iret);
        strcat(calname,p_caln);
/* set calibration level */
        len[0] = strlen(calname);
        if ((len[0] - len[1]) < 11)
           level = 1;
           else
           level = 2;
        memcpy(clna,&calname[0]+len[1],len[0] - len[1]+1);
        if (strcmp(calnames,clna) < 0){
           strcat(calnames,clna);
           clna[0]='\0';
           strcat(calnames,";");
           }
        len[0]=0;       
        }
/* input calibration parameters */
    if ( get_calmatrix(calname, cal_m, &adc, adctill) < 0 ){
       fprintf(stderr, errmsg[6], PROGRAM, calname);
       exit(26);
       }  
/* reset calname */
    if (level != 3)
       calname[0] = '\0';  

/* input range7 calibration parameters from C#_range7.fgmcal */ //200911
     strcpy(caln7,calpath);
     scch="1234"[scid];
     sprintf(caln7 + strlen(caln7), "C%c_range7.fgmcal", scch);
     if ( ( calf7= fopen(caln7, "r") ) != NULL){
       for (i=0; i<13; i++){
         if ((ji=fscanf(calf7, "%lf",&dv0))==0)
            fprintf(stderr,"calibration input error\n");
	 else   
	 calm7[i]=dv0;  
       } // for
     } // if calf7  
     else fprintf( stderr, "range 7 calfile: %s not found; range 7 not calibrated! \n", caln7 ); 
     
     if( calf7 != NULL ) 
       fclose(calf7);
   
/* fill in last (6th) column in cal_m*/  
   for (i=0; i < 13; i++) {
   	cal_m[i][5] = calm7[i];  
        cal_m[i+12][5] = calm7[i]; 
        }
    }
    
/* write ADC_nr to status info, if fv-time < adc-till-time */
  timestr2unix(adctill, t0, &tillt);
  if ( cmptime(&fv.tv,&tillt) < 0 ) 
     { 
     if ( adc > 1) 
        set_bit(&fv.stat,8);  
     }

/* apply calibration matrix */
  if ( cal_fgm(&fv, cal_m) < 0){
        fprintf(stderr, warning[2], PROGRAM);
        continue;
        }

/* add calibration level info to fgm-status */
  if (level > 0)
     fv.stat |= level<<24; 
      
/* write calibrated vector to stdout */
  if ( write_fgmtvec(stdout,&fv) != fvsz )
        {
         fprintf ( stderr, errmsg[8], PROGRAM );
         exit(8);
        }
     }    /* endwhile */    
    if ( strfilep != NULL ) {
       fprintf(strfilep, "%s\n", calnames); 
       fclose(strfilep);
       }
//   fclose(inf);
   exit(0);
  }

