/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching
 * libutil.c & new string lengths & new status - V6.1 (2000-12-15)
 * set_daystring = one minut before midnight!!,
 * calinput string length = 300                - V6.2 (2001-03-10)
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmiff"  
#define VERSION "6.2 (2001-03-10)"
#define PURPOSE "put CLUSTER FGM data into the .iff file format"
#define USAGE "USAGE:

 ... | fgmiff <filename> { -p | -s } [-o [<outfile>]]

 The program reads CLUSTER FGM data (in fgmtvec_t format) from stdin
 and writes them on an ASCII file, using the interface file format (IFF)
 for primary or summary parameter data bases.

 <filename> is the name of a file in which the name and day-of-creation
 of the used calibration files have been stored."

#define OPTIONS "OPTIONS:

  -p     produce IFF listing for prime parameter data base.
         The input data must be spin averages.

  -s     produce IFF listing for summary parameter data base.
         The input data must be one-minute averages.
 
  -o     use <outfile> as output file. If -o is used but <outfile> is
         omitted, output is written to stdout.
         Default is to generate an output file name according to the
         IFF file naming convention.

  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@geophys.nat.tu-bs.de)"

static char *errmsg[] =                           /* error messages */
  {
/*  0 */ "Use '%s -h' for help.\n",
/*  1 */ "ERROR in %s: Illegal option %s.\n",
/*  2 */ "ERROR in %s: Could not open output file %s.\n",
/*  3 */ "ERROR in %s: Illegal usage.\n",
/*  4 */ "ERROR in %s: Missing parameter -p or -s.\n",
/*  5 */ "ERROR in %s: Options -p and -s are mutually exclusive.\n",
/*  6 */ "ERROR in %s: Missing parameter <filename>.\n",
/*  7 */ "ERROR in %s: Can't generate output file name (empty input stream).\n",
/*  8 */ "ERROR in %s while reading from stdin.\n",
  };

static char *warning[] =
  {
/*  0 */ "%s WARNING: Could not evaluate command '%s'.\n",
/*  1 */ "%s WARNING: Could not open file %s for reading calfile names.\n",
/*  2 */ "%s WARNING: Could not read calfile name for day %s from file %s.\n",
  };


#define HYPHENLINE   "-------------------------------------------------------------------------------------------------------"

#define SPINRATE 4000


#include "libutil.h"

/*
 * record of time information used in '.iff'-files
 */
typedef struct
  {
   int    year;                    /* years since 1900                      */
   int    month;                   /* month of year    (1 .. 12)            */
   int    day;                     /* day of month     (1 .. 31)            */
   int    hour;                    /* hour of day      (0 .. 23)            */
   int    min;                     /* minute of hour   (0 .. 59)            */
   float  sec;                     /* sec of minute    (0.0 .. 59.99999)    */
   int    msecofday;               /* milliseconds of the day (0 .. 8.64e7) */
  } 
ifftime_t;


ifftime_t get_iff_time ( timesp t )
/*----------------------------------------------------------------------------- 
 * determines the iff-time structure from a timesp structure.
 -----------------------------------------------------------------------------*/
{
 struct tm  ut;
 ifftime_t  res;
  
 ut = *gmtime(&t.tv_sec);

 res.year = 1900 + ut.tm_year;
 res.month = 1 + ut.tm_mon;
 res.day = ut.tm_mday;
 res.hour = ut.tm_hour;
 res.min = ut.tm_min;

 res.msecofday = (t.tv_sec % 86400) * 1000 + t.tv_nsec / 1000000;
 if ( t.tv_nsec % 1000 >= 500 )
    res.msecofday++;
 res.sec = (res.msecofday % 60000) / 1000.0;
  
 return res;
}

static void set_daystring ( timesp t, char *string )
/*-----------------------------------------------------------------------------
 * sets a string 'YYYYMMDD' representing the processed day.
 -----------------------------------------------------------------------------*/
{
 timesp   ts;
 ifftime_t    it;
 int          i, min_msec=60000;

 ts = t;
 it = get_iff_time ( ts );
/* milliseconds until midnight */
 i = 86400000 - it.msecofday;
 if ( i < min_msec )
   {
/*    use next day */
    ts.tv_sec = ts.tv_sec + (i/1000) + 1;
    it = get_iff_time ( ts );
   }
 sprintf ( string, "%04d%02d%02d", it.year, it.month, it.day );
 return;
}
      

static void print_calib ( FILE *outfile, char *program, char *filename, 
                          char *daystring, int database, int scid  )
/*----------------------------------------------------------------------------- 
 * writes the "Calib_software" and "Calib_inputs" records to the '.iff'-file.
 -----------------------------------------------------------------------------*/
{
 FILE  *pfile=NULL;
 int    i;
 char   cmd[256], string[300], calversion[80], calinput[300],\
        comment[40], software[40], input[40], spacecraft[40], xxx[3];

/* format strings, defined for summary database */
 strcpy ( comment,    "%%\n%% %s Variable Attributes\n" );
 strcpy ( software,   "V %s%%CL_SP_FGM Calib_software %s\n" );
 strcpy ( input,      "V %s%%CL_SP_FGM Calib_input %s\n" );
 strcpy ( spacecraft, "V %s%%CL_SP_FGM SC_id %1d\n" );
/* if primary database, overwrite these strings */
 if ( database == 1 )
   {
    sprintf ( xxx, "%1d_P", scid );
    strncpy ( &software[7], xxx, 3 );
    strncpy ( &input[7], xxx, 3 );
   }

/* read software version from the stdout of 'fgmcal -V' */

 cmd[0] = '\0';
 i = strlen ( program );
 if ( i > 6 )
/*    put path to executable in front of command name */
    strncat ( cmd, program, i-6 );
 strcat ( cmd, "fgmcal -V | sed 's/ /_V/' | sed 's/ /_/' | sed 's/(//' | sed 's/)//' | sed 's/-//g'" );

 pfile = popen ( cmd, "r" );
 if ( pfile != NULL )
   {
    if ( fscanf (pfile,"%s",calversion) != 1 )
       strcpy ( calversion, "(unknown)" );
    pclose ( pfile );
   }
 else
   {
    fprintf ( stderr, warning[0], PROGRAM, cmd );
    strcpy ( calversion, "(unknown)" );
   }

/* read name and version number of used calfiles from storage file */

 calinput[0] = '\0';
 pfile = fopen ( filename, "r" );
 if ( pfile != NULL )
   {
    if ( fscanf ( pfile, "%s", string ) == 1 )
        strcat ( calinput, string );
    fclose ( pfile );
   }

 if ( strlen(calinput) == 0 )
   {
    strcpy ( calinput, "(unknown)" );
    fprintf ( stderr, warning[2], PROGRAM, daystring, filename );
   }
      
 fprintf ( outfile, comment,  "Epoch" );
 fprintf ( outfile, software, "Epoch", calversion );
 fprintf ( outfile, input,    "Epoch", calinput );
 if ( database == 2 )
    fprintf ( outfile, spacecraft, "Epoch", scid );

 fprintf ( outfile, comment,  "B-vector" );
 fprintf ( outfile, software, "B_xyz_gse", calversion );
 fprintf ( outfile, input,    "B_xyz_gse", calinput );
 if ( database == 2 )
    fprintf ( outfile, spacecraft, "B_xyz_gse", scid );

 fprintf ( outfile, comment,  "B-sigma (total)" );
 fprintf ( outfile, software, "B_nsigma_t", calversion );
 fprintf ( outfile, input,    "B_nsigma_t", calinput );
 if ( database == 2 )
    fprintf ( outfile, spacecraft, "B_nsigma_t", scid );

 fprintf ( outfile, comment,  "B-sigma (magnitude)" );
 fprintf ( outfile, software, "B_nsigma_b", calversion );
 fprintf ( outfile, input,    "B_nsigma_b", calinput );
 if ( database == 2 )
    fprintf ( outfile, spacecraft, "B_nsigma_b", scid );

 fprintf ( outfile, comment,  "Status" );
 fprintf ( outfile, software, "Status", calversion );
 fprintf ( outfile, input,    "Status", calinput );
 if ( database == 2 )
    fprintf ( outfile, spacecraft, "Status", scid );

 return;
}


static void print_mode ( FILE *outfile, int database, int scid, int mode,
                         timesp t, int rec_nr )
/*----------------------------------------------------------------------------- 
 * writes the instrument mode to the '.iff'-file.
 -----------------------------------------------------------------------------*/
{
 ifftime_t    it;
 char         id[2];
 static char  *mode_str[16] =
   {
/*  0 */        "",
/*  1 */        "",
/*  2 */        ">FGM_2>FGM option 2, data rate: 16 Hz",
/*  3 */        ">FGM_3>FGM option 3, data rate: 18 Hz",
/*  4 */        ">FGM_4>FGM option 4, data rate: 22 Hz",
/*  5 */        "",
/*  6 */        "",
/*  7 */        "",
/*  8 */        "",
/*  9 */        "",
/* 10 */        ">FGM_A>FGM option A, data rate: 16 Hz",
/* 11 */        ">FGM_B>FGM option B, data rate: 18 Hz",
/* 12 */        ">FGM_C>FGM option C, data rate: 22 Hz",
/* 13 */        ">FGM_D>FGM option D, data rate: 67 Hz",
/* 14 */        "",
/* 15 */        ">FGM_F>FGM option F, MSA dump mode",
   };

 it = get_iff_time(t);
 if ( database == 1 )
    sprintf ( id, "%d", scid );
 else
    sprintf ( id, "L" );

 if ( strlen(mode_str[mode]) )
    fprintf ( outfile, 
              "G Inst_mode \"C%s>FGM>%d>%d-%02d-%02dT%02d:%02d:%06.3fZ%s\"\n",
              id, rec_nr, it.year, it.month, it.day, it.hour, it.min, it.sec, 
              mode_str[mode] );
}


static int check_data ( int new_stat, int prev_stat, int sc, int nr )
/*----------------------------------------------------------------------------- 
 * returns the first status byte of a data record:
 *  = 2   if the data is ok, 
 *  = 1   if the data should be used with caution, 
 *  = 0   if it is bad data.
 -----------------------------------------------------------------------------*/
{
 int  res = 2;

/* Test if the data should be used with caution.  ------------------------ */
 if (
     (subbits(new_stat, 19, 23) > 11) ||        /* less than 40 % of data? */
     (subbits(new_stat, 24, 26) == 1)           /* ground calibration? */
     )
    res = 1;

/* Test if it is bad data.  ---------------------------------------------- */
 if (
     (1 + subbits(new_stat, 30, 32) != sc) ||  /* wrong s/c id ? */
     (subbits(new_stat, 28, 30)) ||            /* secondary sensor or MSA? */
     (subbits(new_stat, 24, 26) == 0) ||       /* no calibration? */
     (!subbits(new_stat, 23, 24)) ||           /* data not averaged ? */
     (subbits(new_stat, 16, 19) != 4) ||       /* coordinate system not GSE? */
/*     (subbits(new_stat, 13, 16)) ||            error flag set? */
     (subbits(new_stat, 9, 11)) ||             /* calibration mode on? */
/*     (subbits(new_stat, 8, 9)) ||               not the default ADC (see fgmtel)? */
/*     (subbits(new_stat, 4, 7) != subbits(prev_stat, 4, 7) && nr != 0) ||
                                                range has changed? */
     (subbits(new_stat, 0, 4) != subbits(prev_stat, 0, 4) && nr != 0) 
                                               /* telemetry option has changed? */
    )
    res = 0;

 return res;
}

 
static void print_vector ( FILE *outfile, fgmtvec_t fv, int database, int sc,
                           int *old_mode, int *new_mode, int *nr )
/*----------------------------------------------------------------------------- 
 * writes a vector to the '.iff'-file after checking for datagaps, setting the
 * status array bytes, and marking mode changes.
 -----------------------------------------------------------------------------*/
{
 static int  prev_msec, diff_msec, prev_stat, stat[4], datagap;
 ifftime_t   it;
 char        utc[13];

 if ( !diff_msec )
   {
    if ( database == 1 )
       diff_msec = 15 * SPINRATE / 10;
    else if ( database == 2 )
       diff_msec = 90000;
   }

 it = get_iff_time ( fv.tv );
 sprintf ( utc, "%02d:%02d:%06.3f", it.hour, it.min, it.sec );

 if ( ( it.msecofday - prev_msec > diff_msec ) && ( *nr != 0 ) && ( !datagap ) )
   {
    fprintf ( outfile, "%% %s\n", HYPHENLINE );
    fprintf ( outfile, "N\n" );
    fprintf ( outfile, "%% %s\n", HYPHENLINE );
    datagap = 1;
   }

 stat[0] = check_data ( fv.stat, prev_stat, sc, *nr );

 if ( ( fv.stat != prev_stat ) || *nr == 0 )
   {
    stat[1] = subbits ( fv.stat, 19, 23 );
    stat[2] = subbits ( fv.stat,  8, 16 );
    stat[3] = subbits ( fv.stat,  4,  7 );

    *new_mode = subbits ( fv.stat, 0, 4) ;

    if ( *new_mode != *old_mode )
      {
       fprintf ( outfile, "%% %s\n", HYPHENLINE );
       print_mode ( outfile, database, sc, *new_mode, fv.tv, *nr );
       fprintf ( outfile, "%% %s\n", HYPHENLINE );
      }

    *old_mode = *new_mode;
   }

 if ( stat[0] )
   {
    fprintf ( outfile,
              "R %5d %s %8d %3d %3d %3d %3d %+.4e %+.4e %+.4e %+.4e %+.4e\n",
              *nr, utc, it.msecofday, 
              stat[0], stat[1], stat[2], stat[3],
              fv.b[0], fv.b[1], fv.b[2], fv.phavar, fv.magvar );
    (*nr)++;
    datagap = 0;
   }
 else
   {
    if ( !datagap )
      {
       fprintf ( outfile, "%% %s\n", HYPHENLINE );
       fprintf ( outfile, "N\n" );
       fprintf ( outfile, "%% %s\n", HYPHENLINE );
       datagap = 1;
      }
   }

 prev_msec = it.msecofday;
 prev_stat = fv.stat;
}


int main ( int argc, char *argv[], char *envp[] )
{
 FILE      *outfile=NULL,*inp=stdin;
 fgmtvec_t  fv;
 time_t     t;
 int        i, index=0, nr=0, database=0, sc=0, old_mode, new_mode, ngot, fvsz;
 char       copt, outname[128], string[9], *cp;

/* evaluate command line parameters */

 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
#ifndef SECURE
             fprintf ( stderr, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n",\
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stderr, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);
 
          case 'o':
             if ( *(argv[i]+2) == '\0' )
               {
                if ( i+1 < argc && *argv[i+1] != '-' )
                  {
                   i++;
                   outfile = fopen ( argv[i], "w" );
                   strcpy ( outname, argv[i] );
                  }
                else
                  {
                   outfile = stdout;
                   strcpy ( outname, "stdout" );
                  }
               }
             else
               {
                outfile = fopen ( argv[i]+2, "w" );
                strcpy ( outname, argv[i]+2 );
               }
             if ( outfile == NULL )
               {
                fprintf ( stderr, errmsg[2], PROGRAM, outname );
                exit(2);
               }
             break;

          case 'p':
             if ( database != 1 )
                database += 1;
             break;

          case 's':
             if ( database != 2 )
                database += 2;
             break;

          case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);
             
          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
            }
      }
    else 
      {
       if ( index == 0 )
          index = i;
       else
         {
          fprintf ( stderr, errmsg[3], PROGRAM );
          fprintf ( stderr, errmsg[0], argv[0] );
          exit(3);
         }
      }
    i++;
   }
  
 if ( !database )
   {
    fprintf ( stderr, errmsg[4], PROGRAM );
    fprintf ( stderr, errmsg[0], argv[0] );
    exit(4);
   }
 if ( database > 2 )
   {
    fprintf ( stderr, errmsg[5], PROGRAM );
    fprintf ( stderr, errmsg[0], argv[0] );
    exit(5);
   }
 if ( index == 0 )
   {
    fprintf ( stderr, errmsg[6], PROGRAM );
    fprintf ( stderr, errmsg[0], argv[0] );
    exit(6);
   }

/*----------------------------------------------------------------------------- 
 * Read first vector, open output file (if not yet done), print headlines
 -----------------------------------------------------------------------------*/
 fvsz = sizeof ( fgmtvec_t );
 ngot = read_fgmtvec ( inp, &fv );
 if ( ngot == fvsz )
   {
    sc = 1 + subbits ( fv.stat, 30, 32 );
    set_daystring ( fv.tv, &string[0] );
    old_mode = subbits ( fv.stat, 0, 4 );

    if ( outfile == NULL )
      {
/*       open default output file */
       if ( database == 1 )
          sprintf ( outname, "C%d_PP_FGM_%s.iff", sc, string );
       else
          sprintf ( outname, "CL_SP_FGM_%s.iff", string );
       outfile = fopen ( outname, "w" );
       if (outfile == NULL)
         {
          fprintf ( stderr, errmsg[2], PROGRAM, outname );
          exit(2);
         }
      }
   }
 else if ( ngot == 0 )
   {
    if ( outfile == NULL )
      {
       fprintf ( stderr, errmsg[7], PROGRAM );
       exit(7);
      }
   }
 else
   {
    fprintf ( stderr, errmsg[8], PROGRAM );
    exit(8);
   }


/* print header
   ------------ */
 fprintf ( outfile, "%%\n" );
 fprintf ( outfile, "%% FGM Data Reduction Software\n" );
 fprintf ( outfile, "%% A Balogh; ICSTM, PI\n" );
 fprintf ( outfile, "%% K-H Glassmeier, S Buchert, R Kempen, J Warnecke; TUBS FRG\n" );
 fprintf ( outfile, "%% E Georgescu; MPE Garching \n" );

 if ( ngot )
   {
    fprintf ( outfile, "%%\n" );
    fprintf ( outfile, "%% Instrument settings: currently not used.\n" );
    fprintf ( outfile, "G Inst_settings \"Not used\"\n" );
   }
              
 t = time(0);
 cp = asctime(gmtime(&t));
 *(cp + 24) = ' ';                     /* replaces the newline character */

 fprintf ( outfile, "%%\n" );
 fprintf ( outfile, "M Data reduction started on %s(UTC)\n", cp );

 if ( ngot )
   {
    fprintf ( outfile, "%%\n" );
    fprintf ( outfile, "%% Fill in initial instrument mode\n" );
    print_mode ( outfile, database, sc, old_mode, fv.tv, 0 );
   
    fprintf ( outfile, "%%\n" );
    fprintf ( outfile, "%% Write out col numbers and field labels for clarity\n" );
    fprintf ( outfile, "%% Note these are ignored by CDF-writing software\n" );
    fprintf ( outfile, "%% %s\n", HYPHENLINE );
    fprintf ( outfile, "%%12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234\n" );
    fprintf ( outfile, "%% R_no.   UTC Time   Millisec      Status          Bx          By          Bz      B_nsigma_t  B_nsigma_b\n" );
    fprintf ( outfile, "%% %s\n", HYPHENLINE );
   }

/*----------------------------------------------------------------------------- 
 * print vectors until EOF
 -----------------------------------------------------------------------------*/
 while ( ngot == fvsz )
   {
    print_vector ( outfile, fv, database, sc, &old_mode, &new_mode, &nr ); 
    ngot = read_fgmtvec ( inp, &fv );
   }

/*----------------------------------------------------------------------------- 
 * print footlines
 -----------------------------------------------------------------------------*/
 fprintf ( outfile, "%% %s\n", HYPHENLINE );
 fprintf ( outfile, "M End of data reached\n" );
 t = time(0);
 cp = asctime ( gmtime ( &t ) );
 *(cp + 24) = ' ';                        /* replaces the newline character */
 fprintf ( outfile, "M Data reduction finished on %s(UTC)\n", cp );

 if ( nr ) 
    print_calib ( outfile, argv[0], argv[index], string, database, sc );

 fprintf ( outfile, "%%\n" );
 fprintf ( outfile, "E %d\n", nr );
 fprintf ( outfile, "%%\n" );

 exit(0);
}
