/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweigi, 2000 MPE Garching
 * Alpha-Dec for VERSION "5.0 (1996-12-18)"
 * timesp <-> timespec_t (SUN)
 * V6.0 (2000-05-25) final-release: updated MAGLAT, MAGLON in cotrans.h
 * check_auxf added, now '-p' accepts posfile WITH or WITHOUT DDS-headers
 * -o, but no name => C#_yyyymmdd_sys.igm 
 * time dependent dipole position implemented -> V6.1
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmpos"
#define VERSION "6.1 (2000-05-25)"
#define PURPOSE "add mission-id and position to CLUSTER FGM data"
#define USAGE "USAGE:

  ... | fgmpos [-p [<posfile>]] [-o [outfile]]

  The program reads CLUSTER FGM vectors (in fgmtvec_t format) from stdin
  and adds a mission identifier and the spacecraft's position, creating
  output in a common format for spacecraft magnetometer data used at
  the Institute of Geophysics and Meteorology of TU Braunschweig.

  The coordinate system used for the magnetic field vectors may be one
  of `scs', `gse', `gsm', `sm', or `j2k'.  For the spacecraft positions,
  the same coordinate system will be used, unless in case of `scs', where
  the positions will be given in `gse'."

#define OPTIONS "OPTIONS:

  -p     use <posfile> as orbit file (*ba* on the CDROM)for determing the 
         spacecraft's position. The file may, or may not contain DDS-headers. 
         Default is to use the short term orbit file `$ORBITPATH/stof.cl#', 
         (#=1,2,3,4) with DDS-headers beeing removed.
         If -p is used but <posfile> is omitted, the preprocessed long term 
         orbit file `$ORBITPATH/ltof.cl#' will be used.

  -o     use <outfile> as the output file. If '-o', but no <outfile> given,
         default name is : C#_yyyymmdd_sys.igm, where #=1,2,3,4 stands for
         satellite number and sys for the output coordinate system. 
         Default is to write to stdout.

  -V     print version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@igpp.ucla.edu)"


#include "libutil.h"
#include "cotrans.h"


static char *errmsg[] =
  {
/* 0 */  "Type `%s -h' for help.\n",
/* 1 */  "ERROR in %s: Illegal option %s.\n",
/* 2 */  "ERROR in %s: Illegal usage.\n",
/* 3 */  "ERROR in %s: Could not open output file %s.\n",
/* 4 */  "ERROR in %s while reading from stdin.\n",
/* 5 */  "ERROR in %s: Could not open orbit file %s.\n",
/* 6 */  "ERROR in %s while reading orbit file; IERR = %i.\n",
/* 7 */  "ERROR in %s while writing output file.\n",
/* 8 */  "ERROR in %s: Illegal coordinate system, code %d.\n",
/* 9 */  "ERROR in %s: Wrong spacecraft id: %d, expected %d.\n",
  };

static char *warning[] =
  {
/* 0 */ "%s WARNING: Environment variable %s is not set.\n",
  };

/*
 * FGM vector with id-word and position
 */
typedef struct fgmtrec 
  {
   int               id;       /* identification word */
   int               stat;     /* satellite status word */
   timesp            tv;       /* POSIX time */
   float             b[3];     /* magnetic field */
   float             phavar;   /* spin phase, or total variance */
   float             magvar;   /* variance in magnitude */
   float             r[3];     /* position */
  } 
fgmtrec_t;

int check_auxf(char *inp, int isat)
/**********************************************************************
 * check aux file format                                              *
 * if it contains DDS-headers then remove and output to file: tmp_orb */
{
 FILE *inf, *outf=NULL;
 unsigned char c1[2],line[600],c[16]="",outp[8]="tmp_orb";
 int len=15,first=0,i,llen;
 
 sprintf(c1,"%1d",isat);
 if ( (inf=fopen(inp,"r")) == NULL) {
    fprintf(stderr,"file %s not found\n",inp);
    return (-1);
    }  
 while ( (i=fread(c,1,len,inf)) == len ) {
   if ((c[8] < 6) && (c[8] > 0))
     {
     llen=(int)c[11]+(int)c[10]*256;
     fread(line,1,llen,inf);
     if (!first) {
        first=1;
        if ( (outf=fopen(outp,"w")) == NULL) {
           fprintf(stderr,"could not open file %s \n",outp);
           return -1;
           } 
         }   
     if ( (line[2] == *c1) || (line[1] == *c1) )
        fwrite(line,llen,1,outf);
     }
     else{
     if ( (c[1] == *c1) || (c[2] == *c1) )
        return 0;
     }  
  } //end while 
if (first) 
   fclose(outf);
if (inf != NULL)
   fclose(inf);
return 1;
} 

int main ( int argc, char *argv[], char *envp[] )
{
 FILE          *ofile = stdout;
 fgmtvec_t     fv;
 fgmtrec_t     fr;
 int           fvsz=sizeof(fgmtvec_t), frsz=sizeof(fgmtrec_t);
 int           unit, isat, idat, ipos, ierr, i, out=0;
 int           iout=0, kode=3, use_ltof=0, css=0;
 double        r[3], revnum, tmjd; 
 char          copt, oname[128]="", pname[128],*datest="";
 static char    *csstr[8]={"FS","FSR","SR","SCS","GSE","GSM","SM","GEI"};

/* set orbit file name to empty */
 pname[0] = '\0';         

/* get command line parameters */
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' ) 
      {
       copt = *(argv[i]+1);
       switch (copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

         case 'p':
	   use_ltof = 1;
            if ( *(argv[i]+2) == '\0' && i+1 < argc )
	      {
                if ( *argv[i+1] != '-' )
                  { 
                   i++;
                   strcpy ( pname, argv[i] );
                  }
                }
	       else
                 strcpy ( pname, argv[i]+2 );
             break;

          case 'o':
             out=1; 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
	       {
                 if ( *argv[i+1] != '-' )
                    { 
                    i++;
                    strcpy ( oname, argv[i] );
                    }
                }
	       else
                strcpy ( oname, argv[i]+2 );
             break;

          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:  
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[2], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(2);
      }
    i++;
   }

/* read first record to define s/c & coordinate system */
 idat = read_fgmtvec ( stdin, &fv );
 if ( idat < fvsz )
   {
    fprintf ( stderr, errmsg[4], PROGRAM );
    exit(4);
   }
 css = subbits(fv.stat,16,19);
 isat = 1 + subbits ( fv.stat, 30, 32 );   
 datest=datestr(&fv.tv.tv_sec);

/* define default outname : if '-o' and no <outf> given & open outfile*/
 if (out){ 
    if (strlen(oname) < 2){
       sprintf(oname,"C%1d_%8s_",isat,datest);
       strcat(oname,csstr[css]);
       strcat(oname,".igm");
       }
    ofile = fopen ( oname, "w" );
    if ( ofile == NULL )
       {
       fprintf ( stderr, errmsg[3], PROGRAM, oname );
       exit(3);
       }
       else
       fprintf(stderr,"OUTPUT to: %s\n",oname);  
    }

/*  check if orbit file name given */
 i = strlen ( pname );
 if ( i > 0 ) {
/* check if dds_header & remove it */
   if (check_auxf(pname, isat) > 0){
       strcpy(pname,"");
       strcat(pname,"tmp_orb");
       i=strlen(pname);
       }
   }
   else
   {

/* use default file name */
    for ( i = 0; envp[i] != NULL; i++ )
       if ( !strncmp(envp[i],"ORBITPATH",9) )
         { 
          strcpy ( pname, getenv ( "ORBITPATH" ) );
          strcat ( pname, "/" );
          break;
         }
    i = strlen ( pname );
    if ( i == 0 )
       fprintf ( stderr, warning[0], PROGRAM, "ORBITPATH" );
    if (use_ltof)
       sprintf ( &pname[i], "ltof.cl%d", isat );
    else
       sprintf ( &pname[i], "stof.cl%d", isat );
   }
 i = strlen ( pname );
/* open orbit file */
 unit = iflu_orb_ ( &isat, pname, i );
 if ( unit <= 0 )
   {
    fprintf ( stderr, errmsg[5], PROGRAM, pname );
    exit(5);
   }

 while ( idat == fvsz )
   {
/* coordinate system of magnetic field */
    i = subbits(fv.stat,16,19);
    if (i > 3)
      {
       if (i < 7) 
	  iout = i - 3;   /* GSE, GSM, or SM */
       else
	  iout = i;       /* GEI-of-J2000.0 */
       ipos = iout;
      }
    else if (i == 3)
      {
       iout = 31;         /* SCS */
       ipos = 1;
      }
    else
      {
       fprintf ( stderr, errmsg[8], PROGRAM, i );
       exit(8);
      }
    if ( subbits(fv.stat,30,32) != (isat-1) )
      {
       fprintf ( stderr, errmsg[9], PROGRAM, 1+subbits(fv.stat,30,32), isat );
       exit(9);
      }

/* calculate Modified Julian Date, referred to 2000-01-01, 0 UT */
    tmjd = mjd2000 ( fv.tv );

/* get orbit information */
    orbit_ ( &tmjd, &kode, &unit, &ierr, &isat, &r[0], &revnum );
    if (ierr != 0)
      {
       fprintf ( stderr, errmsg[6], PROGRAM, ierr );
       exit(6);
      }
    if ( ipos != 7 )
      {
       ierr = j2ktrans ( ipos+3, tmjd, &r[0], &r[1], &r[2] );
       if (ierr != 0)
	 {
	  fprintf ( stderr, errmsg[8], PROGRAM, ipos+3 );
	  exit(8);
	 }
      }
    fr.r[0] = (float) r[0];
    fr.r[1] = (float) r[1];
    fr.r[2] = (float) r[2];

/* set identification word */
    fr.id = subbits(fv.stat,23,24);      /* high-res or averaged data */
    fr.id |= ipos << 7;                  /* coor-sys of position */
    fr.id |= iout << 11;                 /* coor-sys of magnetic field */
    fr.id |= isat << 16;                 /* mission id */

    fr.stat = fv.stat;
    fr.tv = fv.tv;
    for ( i = 0; i < 3; i++ )
       fr.b[i] = fv.b[i];
    fr.phavar = fv.phavar;
    fr.magvar = fv.magvar;
    idat = fwrite ( &fr, 1, frsz, ofile );
    if (idat < frsz) 
      {
       fprintf ( stderr, errmsg[7], PROGRAM );
       exit(7);
      }

/* read next input structure */
    idat = read_fgmtvec ( stdin, &fv );
   }

/* quit */
 exit(0);
}
