/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching
 */

/*
 * 1.2    1994-12-20  - time checks included: input blocks and output vectors
 *                                            chronologically arranged?
 *
 * 1.2.1  1995-01-17  - error messages changed
 *                    - function 'mode_option_test' detects now unknown option
 *
 * 1.2.2  1995-02-01  - filter delay is now corrected after calculating the
 *                      timesp time of the  vector, and not in terms
 *                      of HFC ticks which had less accuracy
 *                      -> 'calc_spinphase' had to be changed 
 *                    - function 'det_spinfreq' and 'calc_spinphase' use now the
 *                      determined UTC of the sun reference pulses and the UTC
 *                      of the first vector and not the HFC ticks
 *                      -> spinfreq is given in Hz
 *
 * 2.0                - use the new library functions and definitions
 *
 * 2.1                - correction for non normalized filters applied 
 *
 * 3.0                - warning message for burst mode 3 added
 *
 * 4.1                - new fgmtvec_t structure for the output vectors
 *                      -> average of raw variances stored in the new word
 *                         'magvar' of the fgmtvec_t structure
 *
 * 5.0    1996-12-18  - read/generate a cal file with the DDS mask for packets
 *                    - add 10 msec time delay for analog filter
 *
 * 6.0    1999-10-05  - no calfile search, calibration & "-g, -m" option out
 *                    - all hard-coded constants: HFC=4096., ..
 *      		     put in "const.fgm", to allow easier changing 		                      	
 *		   - in calc_rawvariance test for divisor=0 ??? MSA ???      
 *		   - in det_spin_freq test for divisor=0 (if spindif=0 return 0 )             -      16/12/99
 *		   - rewritten inctime
 *
 *        2000-03-09  - removed switch (if filter)
 *		   - introduced tests for: "no data", "power-on"
 *		   - error messages counter instead of output messages
 *		   - fill_vec_stat changed to mark "BAD" packets
 *		   - extract_vectors changed to mark( discard) "BAD" vectors 
 *		   - new function get_fgmconstants, fgmfreq out
 *		   - calc_UTC (....,double fgmfrq) added
 *		   - calc_spinphase
 *       2000-03-27   - calc_rawvariance - implemented
 *       2000-04-10   - get_eclipse marks BAD bit 27 
 * 6.1   2000-05-25   - option '-e' to give explicit STEF input (*ta* on CDROM);
 *                      no preprocessing needed; new function eclfile(int scid)
 *                    - discard last vector of previous reset if tdiff < sampling
 * 6.2  2000-09-30    - new "const.fgm" file format, range dependent t_delay
 *                    - negative frequency bug fixed (det_spinfrq uses HFC-clock)
 * 6.3  2000-11-10    - phase-jump bug for range change fixed (i > 0)
 * 6.4  2000-12-01    - cal-mode in fill_vec_stat changed
 * 6.5  2000-12-15    - option -a accepts now choices: 1/2/4, to let through
 *                      "bad" data due to rng_chng/cal_mode/eclipse 
 * 6.6  2001-02-01    - test aux.status_word for `bad' TM-packet(bit 12)&remove
 * 6.7  2001-02-11    - last_option=-1 for start-packet not being removed
 *                      output MSA_packets number, test-order changed  
 * 6.8  2001-03-01    - "doy" bug corrected (eclipse OK now)
 * 6.9  2001-03-10    - put "-a" to "-a 7"
 * 7.0  2001-05-05    - TM-mode change F(MSA) to C, throw=1
 * 7.1  2001-06-30    - reset gap for i>0 & mark bad range=1
 *                    - bad > 0 instead of ==1 for error bits
 *                    - POWER ON changed from reset_count < 3 to < 5
 * 7.2  2001-07-20    - introduced cnt_thresh for POWER ON (5 if -a, 175 else)
 * 7.3  2002-02-20    - aux.count overflow distinguished from power on
 * 7.4  2003-01-16    - nominal spin-rate and tm-packet-rate defined
 *                    - discard packet if spin difference <>5% of nominal spin-rate
 *                    - discard packet if packet-time rate <>5% of nominal
 *                    - compare sunpulses to previous & detect no sunpulse info 
 *                    - correct read_eclipse from files with DDS-header (TMO)	
 * 7.5  2003-02-03    - discard packet if packet-time rate < (nominal-5%)
 *                    - conditions re-ordered, if skip. Header & aux-info saved
 *                    - packet error bug corrected bad += (errorbits(13-15) >0)
 *                    - in extract_vectors badvecnr made static 
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmtel"  
#define VERSION "7.5 (2003-02-03)"
#define PURPOSE "extract CLUSTER FGM data from the telemetry data (RDM)"
#define USAGE "USAGE:

 fgmtel [-a <int>] [-e <ecl>] [-s] [list] | ...

 The program reads CLUSTER FGM data from telemetry files given in list
 (default stdin) and writes the primary vectors to stdout using the 
 'fgmtvec_t' data structure.
 The range depending scale factor is already taken into account. Thus, the
 output vectors are given in nT. 

 If the option -s is given, the secondary vectors are extracted instead of 
 the primary vectors."

#define OPTIONS "OPTIONS:

  -a     output all vectors, if <int> missing or <int>=7.
        <int> is a 3-bit integer, each bit corresponding to a selection rule
        <int>=1 (bit 0), output marked vectors at range changes 
        <int>=2 (bit 1), output marked vectors in calibration mode
        <int>=4 (bit 2), output data during eclipses 
         DEFAULT: ouput only non-marked vectors
                  
  -e     read eclipse info from file <ecl> (name pattern: *ta* on CDROM).
         Default is '$SATTPATH/s{|l}tef.cl#', where #=1,2,3,4.
  
  -s     extract vectors of the secondary sensor
  
  -v     verbose mode
  
  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu   (eg@mpe.mpg.de)
 Stephan Buchert   (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen   (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke    (joerg@geophys.nat.tu-bs.de)"

#include "libutil.h"

#define normal_data_length    746     /* - block length of the sensor    */
#define burst_data_length_1  2198     /*   data for the different        */
#define burst_data_length_2   834     /*   telemetry modes               */
#define burst_data_length_3  3562     /*   (without auxiliary data)      */
#define primary_data_length_a    456  /* - length of the different data  */
#define secondary_data_length_a   34  /*   segments for the different    */     
#define msa_data_length_a        256  /*   FGM options                   */
#define primary_data_length_b    536
#define secondary_data_length_b  210
#define primary_data_length_c    654
#define secondary_data_length_c   92
#define primary_data_length_d   1958
#define secondary_data_length_d  240
#define msa_data_length_f       3562
#define MSA_auxiliary_length   80     /* length of MSA auxiliary data    */
#define nmtmrate  5.15222      	 /* nominal TM-packet rate	        */
#define nmspin    16384.		 /* nominal spin rate               */

static char *errmsg[] =                           /* error messages */
  {
/*  0 */  "Use '%s -h' for help.\n",
/*  1 */  "%s ERROR: unknown option %s\n",
/*  2 */  "%s ERROR: cannot allocate memory for file list\n",
/*  3 */  "%s ERROR: cannot open input file %s\n",
/*  4 */  "%s ERROR: options 's' and 'm' cannot be used simultaneously\n",
/*  5 */  "%s ERROR: input data packets not with ascending time\n",
/*  6 */  "%s ERROR while reading instrument auxiliary data\n",
/*  7 */  "%s ERROR: cannot allocate memory for input buffer\n",
/*  8 */  "%s ERROR while reading science data\n",
/*  9 */  "internal %s ERROR: unkown telemetry option\n", 
/* 10 */  "%s ERROR: cannot allocate memory for output buffer\n",
/* 11 */  "%s ERROR while writing output data\n",
/* 12 */  "%s ERROR when opening cal directory\n",
/* 13 */  "%s ERROR when reading dds mask\n",
/* 14 */  "%s ERROR in dds mask, S/C %d does not match %d\n",
/* 15 */  "%s ERROR: input packets not from the same S/C %d",
/* 16 */  "%s ERROR: cannot allocate memory for dds mask"
  };

static char *warning[] =
  {
/*  0 */  "",
/*  1 */  "%s WARNING: mismatch between data source and packet length.\n",
/*  2 */  "%s WARNING: unknown data source in dds packet header\n",
/*  3 */  "%s WARNING: unknown FGM telemetry option .\n",
/*  4 */  "%s WARNING: mismatch between FGM telemetry option and S/C telemetry mode.\n", 
/*  5 */  "%s WARNING: output vectors not time-ordered (incorrect vector rate?).\n",
/*  6 */  "%s WARNING: input contains data of burst mode 3 which cannot be processed.\n",
/*  7 */  "%s WARNING: wrong packet time(differs by > 5%% from nominal)\n",
/*  8 */  "%s WARNING: spin-rate differs by > 5%% from the nominal spin-rate\n",
  };
static char ecln[128]="";

typedef struct toptpar 
  {
   int nvec[3];               /* number of vectors            */
   int ndec[2];               /* filter decimation            */
   int bytepos[3];            /* byte position at which the 
                                 different vector types start */
  } 
toptpar_t;

static toptpar_t tpar[5] = 
 {{{ 81,  6,  41}, {13, 185}, 
   {0, primary_data_length_a, primary_data_length_a + secondary_data_length_a}},
  {{ 95, 37,   0}, {11,  29}, 
   {0, primary_data_length_b, primary_data_length_b + secondary_data_length_b}},
  {{116, 16,   0}, { 9,  67}, 
   {0, primary_data_length_c, primary_data_length_c + secondary_data_length_c}},
  {{348, 41,   0}, { 3,  26}, 
   {0, primary_data_length_d, primary_data_length_d + secondary_data_length_d}},
  {{  0,  0, 592}, { 0,   0}, {0, 0, 0}}};

static int verbose=0, all=0, unmark=0;

static int determine_data_length_and_mode(ddshead_t *head,
                                          int *length,
                                          int *mode)
/*----------------------------------------------------------------------------- 
 * The function 'determine_data_length_and_mode' determines the length of the *
 * science data block (without science auxiliary data) from the ESOC header   *
 * bytes. The used telemetry mode can be inferred from this data length.      *
 * If there is an inconsistency between the determined data length and the    *
 * data source, the function will return the corresponding warning number and *
 * 'length' will be set to the length of the complete data block, otherwise   *
 * the function will return zero.                                             *
 -----------------------------------------------------------------------------*/
{
 *length = pktlen(head) - auxiliary_length;
 if ((head->source == 31) || (head->source == 71) ||
     (head->source == 111) || (head->source == 151))
   {
    if (*length == normal_data_length)
       *mode = 0;
    else
      {
       *length += auxiliary_length;
       return(1);
      }
   }
 else if ((head->source == 38) || (head->source == 78) ||
          (head->source == 118) || (head->source == 158))
   {
    switch (*length)
      {
       case burst_data_length_1:
          *mode = 1;
          break; 
       case burst_data_length_2:
          *mode = 2;
          break; 
       case burst_data_length_3: 
          *mode = 3;
          break; 
       default:
          *length += auxiliary_length;
          return(1);
      }
   }
 else
   {
    *length += auxiliary_length;
    return(2);
   }
 return(0);
}


static void read_telstream(FILE *in_file, char **telstream, int length)
/*----------------------------------------------------------------------------- 
 * First the function 'read_telstream' allocates new memory, if the actual    *
 * memory size is different from the data length to be read. Then the data    *
 * are read. If the memory cannot be allocated or an error occurs during      *
 * reading of the data, the program will be exited with the corresponding     *
 * exit code.                                                                 *
 -----------------------------------------------------------------------------*/
{
 static int  telsize;

 if (telsize != length)
   {
    if (*telstream) 
       free(*telstream);
    telsize = length;
    if ((*telstream = malloc(telsize)) == NULL)
      {
       fprintf ( stderr, errmsg[7], PROGRAM );
       exit(7);
      }
   }
 if (fread(*telstream, 1, telsize, in_file) != telsize)
   {
    fprintf ( stderr, errmsg[8], PROGRAM );
    exit(8);
   }
}


static int mode_option_test(int mode, int opt)
/*----------------------------------------------------------------------------- 
 * The function 'mode_option_test' tests, whether the FGM telemetry option    *
 * has a valid value and whether it fits to the S/C telemetry mode.           *
 * There are four different telemetry modes (one normal and three burst       *
 * modes). Six of the possible telemetry options (2, 3, 4, a, b, and c) fit   *
 * to normal mode (mode 0) or burst mode 2 (mode 2). Option 'd' fits to burst *
 * mode 1 (mode 1) and option 'f' fits to burst mode 3 (mode 3).              *
 * If the FGM telemetry option hasn't a valid value, the function will return *
 * the warning number 3. If there is any inconsistency, the function will     *
 * return the warning number 4. If everything is ok, it will return 0.        *
 -----------------------------------------------------------------------------*/
{
 if ((opt != 2) && (opt != 3) && (opt != 4) && (opt != 0xa) && 
     (opt != 0xb) && (opt != 0xc) && (opt != 0xd) && (opt != 0xf))
    return(3);
 switch(mode)
   {
    case 0:
       if ((opt == 2)   || (opt == 3)   || (opt == 4) ||
           (opt == 0xa) || (opt == 0xb) || (opt == 0xc))
          return(0);
       else
          return(4);
    case 1: 
       if (opt == 0xd)
          return(0);
       else
          return(4);
    case 2:
       if ((opt == 2)   || (opt == 3)   || (opt == 4) ||
           (opt == 0xa) || (opt == 0xb) || (opt == 0xc))
          return(0);
       else
          return(4);
    case 3:
       if (opt == 0xf)
          return(0);
       else
          return(4);
    default: 
       return(4);       
   }
}


static toptpar_t *get_telemetry_parameters(int option)
/*----------------------------------------------------------------------------- 
 * The function 'get_telemetry_parameters' returns a pointer to a set of      *
 * telemetry parameters. This set depends on the actual FGM telemetry option. *
 -----------------------------------------------------------------------------*/
{
 switch(option)
   {
    case 2:   return(&tpar[0]);
    case 3:   return(&tpar[1]);
    case 4:   return(&tpar[2]);
    case 0xa: return(&tpar[0]);
    case 0xb: return(&tpar[1]);
    case 0xc: return(&tpar[2]);
    case 0xd: return(&tpar[3]);
    case 0xf: return(&tpar[4]);
    default:  fprintf ( stderr, errmsg[9], PROGRAM );
              exit(9);
   }
}

static int fill_vec_stat(int sc, int sensor, int status_word, int var0)
/*----------------------------------------------------------------------------- 
 * The function 'fill_vec_stat' returns the status word of an output vector   *
 * filled by informations from the ESOC header and from the status word and   *
 * the first variance word of the science auxiliary data.                     *
 -----------------------------------------------------------------------------*/
{
 int se=0,st=0,va=0;
 int sw=0,ob=0,ib=0,cal_mode=0;
 
 se=sensor;
 st=status_word;
 va=var0;
 sw |= ((sc & 3) << 30);                                 /*    spacecraft ID */
 sw |= ((se & 3) << 28);                                 /*  prim/sec sensor */
 ob = get_bit(va,15);
 ib = get_bit(va,14);
 cal_mode = subbits(status_word,6,8);
 if (!sensor && ob && ((cal_mode + get_bit(var0,10) + get_bit(var0,12)) > 0))
      set_bit(&sw, 10);                                   /*      OB cal mode */
 if (!sensor && ib && ((cal_mode + get_bit(var0, 9) + get_bit(var0,11)) > 0))
      set_bit(&sw, 9);                                   /*      IB cal mode */
 if ( (!sensor && ob) || (sensor && ib) )
      ob = 0;
 else if ( (!sensor && !ob) || (sensor && !ib) )
      ob = 1;
 sw = bitcpy(var0, 13, sw, 12);                          /*   science filter */
 sw = bitcpy(ob, 0, sw, 7);                              /* sensor:OB/IB(0/1)*/      
 sw = bitcpy2(subbits(status_word, 8, 9), sw, 11, 12);   /*       MSA filter */
 sw = bitcpy2(subbits(status_word, 0, 4), sw, 0, 4);     /* telemetry option */
 return sw;

}


static timesp calc_UTC(timesp *reset_pulse_time, 
                           int        hfcreset, 
                           int        hfcval,
                           double     hfcfrq)
/*----------------------------------------------------------------------------- 
 * The function 'calc_UTC' calculates the UTC of a HFC counter tick value     *
 * - hfcval - from the UTC of the reset pulse - reset_pulse_time - and  the   *
 * HFC counter ticks of the reset pulse - hfcreset -.                         *
 -----------------------------------------------------------------------------*/
{
 double     delta_t, delta_t_frac, delta_t_int; 
 timesp dt;

 if (hfcreset < hfcval)
    hfcreset += 0x10000;
 delta_t = (double)(hfcreset - hfcval) / hfcfrq;
 delta_t_frac = modf(delta_t, &delta_t_int);
 dt.tv_sec  = delta_t_int;
 dt.tv_nsec = 1.e9 * delta_t_frac;
 return subtime(reset_pulse_time, &dt);
}



static void correct_filter_delay(timesp *vec_timep, int dec, double fgmfrq)
/*----------------------------------------------------------------------------- 
 * The function 'correct_filter_delay' corrects the time of a vector for the  *
 * filter delay. This delay is half the filter length and can be calculated   *
 * from the filter decimation level 'dec'.                                    *
 -----------------------------------------------------------------------------*/
{
 double filter_delay, filter_delay_frac, filter_delay_int;
 timesp dt;

 filter_delay = (2.0 * dec - 0.5) / fgmfrq;
 filter_delay_frac = modf(filter_delay, &filter_delay_int);
 dt.tv_sec  = filter_delay_int;
 dt.tv_nsec = 1.e9 * filter_delay_frac;
 *vec_timep = subtime(vec_timep, &dt);
}


static double det_spinfrq(int tt_sun0, int tt_sun1, double hfcfrq)
/*----------------------------------------------------------------------------- 
 * The function 'det_spinfrq' determines the spin frequency in Hz. It uses    *
 * the timetags of two successive sun reference pulses 'tt_sun0' and 'tt_sun1'*
 -----------------------------------------------------------------------------*/
{
 if ((tt_sun1-tt_sun0) == 0 ) return 0;
 if ((tt_sun1-tt_sun0) < 0 )
    tt_sun1 += 0x10000;
 return (2.0 * M_PI * hfcfrq / (double) (tt_sun1 - tt_sun0) );
}


static double calc_spinphase(double spinfreq, 
                             timesp *t_sun0,
                             timesp *t_vec,
                             double spinph0)
/*----------------------------------------------------------------------------- 
 * The function 'calc_spinphase' calculates the spinphase of a vector from    *
 * the spin frequency - spinfreq (given in Hz) -, the time of a sun reference *
 * pulse - t_sun0 -, and the time of the vector - t_vec -.                    *
 -----------------------------------------------------------------------------*/
{
 double spinphase;
 timesp time_diff; 

 time_diff = subtime(t_vec, t_sun0);
 spinphase = spinfreq * (1.0 * time_diff.tv_sec + 1.e-9 * time_diff.tv_nsec);
 if (spinphase < 0)
    spinphase += 2 * M_PI;
 else
    spinphase = fmod((spinph0*M_PI/180.+spinphase), 2 * M_PI);
 return spinphase;
}

char *eclfile( int scid)
/* ------------------------------------------------------------------------
 * define default eclipse file name for S/C                               * 
 *----------------------------------------------------------------------- */   
 {
 FILE *inf=NULL;
 char *sstri,*eclf="",inp[128]="./";
 static char *sc[4]={"1","2","3","4"},*scn="0";
 
 scn=sc[scid];
 if ( (sstri=getenv("SATTPATH")) != NULL ) 
    strcpy(inp,sstri);
 if (strncmp("/",&inp[strlen(inp)-1],1) != 0)
    strcat(inp, "/");
 strcat(inp,"stef.cl");   
 strcat(inp,scn);   
 if ( (inf=fopen(inp,"r")) == NULL) {
    inp[strlen(inp)-8]='l';
    }  
 eclf=inp;
 return (eclf);
 }
 
int get_eclipse(char *inp, char *year, int cdoy, timesp *tb, timesp *te)
/* ------------------------------------------------------------------------
 * gets eclipse info from STEF file *inp                                  * 
 * returns 1 if eclipse found, else 0                                     *
 * [tb, te] eclipse time interval (used by extract_vectors to mark "BAD"  *
 *----------------------------------------------------------------------- */   
{
 FILE *inf= NULL;
 char stline[150]="",linee[100]="",line[150]="",ss[5]="",tstr[20]="";
 char c[16]="",pr[2]="P";
 int i,nre=0,len=15,doy=0,start=0,llen;
 timesp t0={0,0};
 
 if ( (inf=fopen(inp,"r")) == NULL) {
    fprintf(stderr,"STEF/LTEF file %s not found\n",inp);
    return 0;
    }
 ss[0]='\0';
 while ( (i=fread(c,1,len,inf) == len ) && (doy < (cdoy+1)) )
   {
   if ( (c[8] < 6) && (c[8] > 0) )
     {
     llen=(int)c[11]+(int)c[10]*256;
     fgets(line,llen+1,inf); 
     }
     else 
     {
     if (fgets(linee,100,inf) != NULL){ 
        strcpy(line,c);
        line[15]='\0';
        strcat(line,linee);
        }
     }  
   if ( ((i=strncmp("ECL",&line[46],3)) == 0) &&
                          ((i=strncmp(year,&line[12],4)) == 0) )
      {
      strncpy(ss,&line[0],3);
      doy=(int)atoi(ss);
      if ((i=strncmp("Z",&line[43],1)) == 0)
         {
         if ( (doy == (cdoy-1)) || (cdoy == doy) )   
            {
            stline[0]='\0';
            strcat(stline,line);
            strncpy(pr,&line[9],1);
            start=1;
            }
         }  // if Z
         else
         {   
         if ( ( (doy == cdoy) || (doy == (cdoy+1)) ) 
                  && (start) && ((i=strncmp(pr,&line[9],1)) == 0) )       
            {
            nre++;
            strncpy(tstr,&stline[12],19);
            timestr2unix(tstr,t0,tb);
            strncpy(tstr,&line[12],19);
            timestr2unix(tstr,t0,te);
            start=0;
            } 
        } // else if Z
     }    // if ecl
  }   // while
  fclose(inf);
return nre;
}


static double calc_rawvariance(unsigned short *variance)
/*----------------------------------------------------------------------------- 
 * The function 'calc_rawvariance' calculates an average of the variances     *
 * found in the raw data. These variances are determined over 116 vectors.    *
 * Thus, there can be eight or nine values within one telemetry reset period. *
 -----------------------------------------------------------------------------*/
{
 double     res, rawvar[9];
 int        i, num, mantissa, exponent;

 res = 0.0;
 num = 0;
 for ( i = 0; i < 9; i++ )
   {
    exponent = subbits ( variance[1+i], 0, 5 );
    exponent -= 11;  //-16+5
    mantissa = subbits ( variance[1+i], 5, 16 );
    if ( exponent != 0 || mantissa != 0 )
      {
       if (exponent>=0)  
         rawvar[i] = mantissa << exponent; 
       if (exponent<0)  
         rawvar[i] = mantissa >> (- exponent); 
       res += rawvar[i];
       num++;
      }
   }
 if( num > 0 ) res /= ( 116 * num ); else res=0;
 return res;
}


/*----------------------------------------------------------------------------- 
 * Decrement the pointer to a bit in the input telemetry stream of bytes.     *
 * If alignment boundary is reached, increment the byte stream pointer.       *
 -----------------------------------------------------------------------------*/
/*void decbp(char **telstream, int *bitptr)
{
 if (--(*bitptr) < 0)
   {
    (*telstream)++;
    *bitptr = 7;
   }
}
*/
/* Macro is faster than function: */
#define decbp(telstream, bitptr) \
 if (--(*bitptr) < 0) \
   {                  \
    (*telstream)++;   \
    *bitptr = 7;      \
   }


static int mask[] = { 0x00000000, 0x00000001, 0x00000003, 0x00000007,
		      0x0000000f, 0x0000001f, 0x0000003f, 0x0000007f,
		      0x000000ff, 0x000001ff, 0x000003ff, 0x000007ff,
		      0x00000fff, 0x00001fff, 0x00003fff, 0x00007fff,
		      0x0000ffff, 0x0001ffff, 0x0003ffff, 0x0007ffff,
		      0x000fffff, 0x001fffff, 0x003fffff, 0x007fffff,
		      0x00ffffff, 0x01ffffff, 0x03ffffff, 0x07ffffff,
		      0x0fffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff
		    };

int fgmword(char **telstream, int *bitptr)
/*----------------------------------------------------------------------------- 
 * The function 'fgmword' returns 14 bits, ie one vector component from the   *
 * FGM telemetry stream 'telstream'.                                          *
 * 'bitptr' points to the sign bit on input and to the range bit on output.   *
 -----------------------------------------------------------------------------*/
{
 int res = 0/*, i*/;
 int nbits, nget, shift, bitp1, m;

/*----------------------------------------------------------------------------- 
 * Get the sign bit and shift it to bit position 14.                          *
 * As the 14 bits raw data are in 2's complement, the right-shifting of the   *
 * sign bit from position 32 (MSB) to position 14 ensures that the value will *
 * be correct even if it is negative.                                         *
 -----------------------------------------------------------------------------*/
 res = bitcpy(**telstream, *bitptr, res, 31) >> 18;
 decbp(telstream, bitptr);
/*----------------------------------------------------------------------------- 
 * Get the remaining 13 bits.
 -----------------------------------------------------------------------------*/
/* Turbo version, get up to 8 bits at a time: */
 nbits = 13;
 nget  = *bitptr+1;
 res  |= (**telstream & mask[nget]) << (nbits - nget);
 *bitptr = 7;
 ++(*telstream);
 nbits -= nget;
 bitp1 = 8;
 while( 1 )
   {
     nget   = ( bitp1<nbits ) ? bitp1 : nbits;
     m      = (**telstream & (mask[nget] << (bitp1-nget)));
     shift  = nbits - bitp1;
     res   |= ( shift>=0 ) ? m << shift : m >> -shift;
     *bitptr -= nget-1;
     decbp(telstream, bitptr);
     nbits -= nget;
     if( nbits<=0 )
       return res;
     bitp1  = *bitptr+1;
   }

/* Loop over every single bit: */
/*
 for (i = 12; i >= 0; --i)
   {
*/
/*    res = bitcpy(**telstream, *bitptr, res, i); */
/*     res |= (1 && (**telstream & (1<<*bitptr))) << i;
     decbp(telstream, bitptr);
   } 
 return res;
*/ 
}


int get_fgmvecs(int nget, char *telstream, fgmvec_t *ifgm)
/*----------------------------------------------------------------------------- 
 * The function 'get_fgmvecs' converts the FGM telemetry stream 'telstream'   *
 * to int vectors + range.                                                    *
 * 'telstream' points to the start byte on input and to last byte on output.  *
 * 'bitptr' points to the start bit in the first byte on input and to the     *
 * next start bit on output.                                                  *
 * Return value is the number of FGM vectors.                                 *
 -----------------------------------------------------------------------------*/
{
 int nvecs = 0, bitptr = 7;

 while (nvecs < nget)
   {
    ifgm->v[0] = fgmword(&telstream, &bitptr);              /*   x component */
    ifgm->rng = bitcpy(*telstream, bitptr, 0, 2);           /* 1st range bit */
    decbp(&telstream, &bitptr);
    ifgm->v[1] = fgmword(&telstream, &bitptr);              /*   y component */
    ifgm->rng = bitcpy(*telstream, bitptr, ifgm->rng, 1);   /* 2nd range bit */
    decbp(&telstream, &bitptr);
    ifgm->v[2] = fgmword(&telstream, &bitptr);              /*   z component */
    ifgm->rng = bitcpy(*telstream, bitptr, ifgm->rng, 0);   /* 3rd range bit */
    decbp(&telstream, &bitptr);
    ifgm++;
    nvecs++;
   }
 return nvecs;
}


static void iv2fv(float *fvp, fgmvec_t *ivp)
/*----------------------------------------------------------------------------- 
 * The function 'iv2fv' converts an FGM vector in integer format as received  *
 * from the telemetry stream to float values: input = ivp, output = fvp.      *
 * It takes into account the range depending scale factor but no calibration  *
 * information.                                                               *
 -----------------------------------------------------------------------------*/
{
 int      i;
 float    factor;

 factor = 8.0 / (1 << (7 - ivp->rng) * 2);
 for (i = 0; i < 3; i++)
    fvp[i] = ivp->v[i] * factor;
}


static void extract_vectors(int sensor, 
                            timesp *reset_pulse_time, 
                            char *telstream,
                            int option,
                            int sc,
                            auxdata_t *aux, int eclflag)
/*----------------------------------------------------------------------------- 
 * The function 'extract_vectors' extracts FGM vectors from the data stream   *
 * block 'telstream' and writes them to the file 'out_file'. The number of    *
 * vectors to be extracted and the starting byte position of the different    *
 * vector types within the data block are given in 'tparp' which also         *
 * contains the data reduction rate regarding the FGM sampling frequency of   *
 * 201.75 Hz. If the output vectors are not chronologically arranged, or the  *
 * memory cannot be allocated, or an error occurs during writing of the data, *
 * the program will be exited with the corresponding exit code.               *
 -----------------------------------------------------------------------------*/
{
 static fgmvec_t    *int_vec;
 toptpar_t          *tparp;
 fgmtvec_t          float_vec;
 static fgmtvec_t   last_vec; 
 static timesp      old_vec_time = {0, 0}, tbegin = {0, 0}, tend = {0, 0};
 timesp             sun_pulse_time0, dt;
 static double      fgmfreq=201.75,hfcfrq=4096.,spinph0=0.,alpha=6.5; 
 static int         int_vec_num, nrint=0, begin=1, first=1, lastrng=-1, ldoy=0;
 static int         obdelay[2], ibdelay[2], badvecnr=0, rngchng=0, rce=0;
 double             spinfreq, delta_phase;                        
 int                status_word,hfcreset,hfcfirst,hfcsun1,hfcsun0;
 int                i, ngot, ob, ib, tdelay, doy, gap=0, irng=0;
 double             tdiff=0.0,delttd=0.0;
 int                sampling_period=0, filter=0, stat0=0 ;
 char               year[5];
 struct tm          ut;
    
 status_word = aux->status_word;
 hfcreset = aux->HFC_reset;
 hfcsun1 = aux->HFC_recent_sun; 
 hfcsun0 =  aux->HFC_previous_sun;
 if (sensor) 
    hfcfirst = aux->HFC_secondary;
    else 
    hfcfirst = aux->HFC_primary;
/*----------------------------------------------------------------------------- 
 * Get the FGM sampling frequency, HFC frequency & other "constants".         *
 *----------------------------------------------------------------------------*/
 if (first){
     if ((get_fgmconstants(sc, &hfcfrq, &fgmfreq, &spinph0, &alpha,
                          &obdelay[0], &ibdelay[0])) < 0){
        fprintf(stderr,"file read error in const.fgm\n");
        exit(4);
        }                                                
    }
/*----------------------------------------------------------------------------- 
 * Get the option depending telemetry parameters.                             *
 -----------------------------------------------------------------------------*/
 tparp = get_telemetry_parameters(option);
/*----------------------------------------------------------------------------- 
 * Determine the vector sampling period in 'nanosec'.                         *
 -----------------------------------------------------------------------------*/
 sampling_period = 1.e9 * tparp->ndec[sensor] / fgmfreq;
/*----------------------------------------------------------------------------- 
 * Fill the output status word. Mark "BAD" vectors due to cal_mode & errors   *
 -----------------------------------------------------------------------------*/
 float_vec.stat = fill_vec_stat(sc, sensor, status_word, aux->variance[0]);
/*----------------------------------------------------------------------------- 
 * Initialize the filter option for the primary vectors.                      *
 -----------------------------------------------------------------------------*/
 if (sensor == 0)
    filter = get_bit(aux->variance[0], 13);
/*----------------------------------------------------------------------------- 
 * Allocate buffer for vectors in integer format.                             *
 -----------------------------------------------------------------------------*/
 if (int_vec_num != tparp->nvec[sensor])
   {
    int_vec_num = tparp->nvec[sensor];
    if (int_vec) 
       free(int_vec);
    if ((int_vec = (fgmvec_t *) malloc(int_vec_num * sizeof(fgmvec_t))) 
        == NULL)
      {
       fprintf ( stderr, errmsg[10], PROGRAM );
       exit(10);
      }
   }
/*----------------------------------------------------------------------------- 
 * Get the integer vectors.                                                   *
 -----------------------------------------------------------------------------*/
 ngot = get_fgmvecs(tparp->nvec[sensor], telstream + tparp->bytepos[sensor],
                    int_vec);
 if (((int_vec + ngot -1)->rng == 0) || ((int_vec + ngot -1)->rng > 7))
    ngot--;
 irng= int_vec->rng / 4; 
/*----------------------------------------------------------------------------- 
 * Calculate UTC of first vector, and correct it for the filter delay.        *
 * Calculate the tdelay difference for range change correction.               *
 -----------------------------------------------------------------------------*/
 float_vec.tv = calc_UTC(reset_pulse_time, hfcreset, hfcfirst, hfcfrq);

/* Analog filter: */
 ob = get_bit(aux->variance[0], 15);
 ib = get_bit(aux->variance[0], 14);
 tdelay = 10000000;
 if ((!ob && !sensor) || (!ib && sensor)){ 
    tdelay=obdelay[irng];
    delttd=obdelay[irng]-obdelay[!irng];
    }
 if ((ob && !sensor) || (ib && sensor))
    { 
    tdelay=ibdelay[irng];
    delttd=ibdelay[irng]-ibdelay[!irng];
    }
 dt.tv_sec  = 0;
 dt.tv_nsec = tdelay;
 float_vec.tv = subtime(&float_vec.tv, &dt);
/* Digital filter: */
 if (filter)
    correct_filter_delay(&float_vec.tv, tparp->ndec[sensor], fgmfreq);
/*----------------------------------------------------------------------------- 
 * Test if the time of the first vector of the current telemetry reset period *
 * is greater than the time of the last BUFFERED vector of the preceding one  *
 -----------------------------------------------------------------------------*/
 tdiff = (double)(float_vec.tv.tv_sec-old_vec_time.tv_sec)*1000000.0+
         (double)(float_vec.tv.tv_nsec-old_vec_time.tv_nsec)/1000.0;
 if ( ( tdiff < (0.0007 * (double)sampling_period) ) || 
      ( tdiff > (0.0015 * (double)sampling_period) ) ){
    if (verbose)
       fprintf ( stderr,"time diff [musec]:%10d%10d\n",(int) tdiff,
                                       (int) (sampling_period/1000) );
    gap=1;                                  
    }
    else 
    gap=0;
 ut = *gmtime(&float_vec.tv.tv_sec);
 sprintf(year, "%d",1900+ut.tm_year);
 doy = ut.tm_yday+1;
 if (doy != ldoy) {
      first=1;
      ldoy=doy;
      }
/*----------------------------------------------------------------------------- 
 * Get the time intervals of (moon-earth) half/eclipse for the day            *
 *----------------------------------------------------------------------------*/
 if (first){
     first=0;
     if (strlen(ecln) < 2)
        strcpy(ecln,eclfile(sc));
     nrint=get_eclipse(ecln,year,doy,&tbegin,&tend);
     }
/*----------------------------------------------------------------------------- 
 * Mark "BAD" vectors due to eclipse or halfshadow                            *
 -----------------------------------------------------------------------------*/
 if (nrint > 0) {
    if ( (cmptime(&float_vec.tv, &tbegin) > 0) &&
                                  (cmptime(&float_vec.tv, &tend) < 0) )
       set_bit(&float_vec.stat,27);      
    }
 if (eclflag) set_bit(&float_vec.stat,27);      

/*----------------------------------------------------------------------------- 
 * Determine the UTC of the before-last sun reference pulse.                  *
 -----------------------------------------------------------------------------*/
 sun_pulse_time0 = calc_UTC(reset_pulse_time, hfcreset, hfcsun0, hfcfrq);
/*----------------------------------------------------------------------------- 
 * Determine the spin frequency in Hz.                                        *
 -----------------------------------------------------------------------------*/
 spinfreq = det_spinfrq( hfcsun0, hfcsun1, hfcfrq );
/*----------------------------------------------------------------------------- 
 * Calculate the spin phase of first vector.                                  *
 -----------------------------------------------------------------------------*/
 spinph0=0.0;
 float_vec.phavar = calc_spinphase( spinfreq, &sun_pulse_time0,
                                              &float_vec.tv, spinph0);
/*----------------------------------------------------------------------------- 
 * Set the value for the raw data variance.                                   *
 * One value for all vectors of the same reset period.                        *
 -----------------------------------------------------------------------------*/
 float_vec.magvar = calc_rawvariance(aux->variance);
/*-----------------------------------------------------------------------------
 * Correct for calibration mode set-on/off delay                                          *
 *----------------------------------------------------------------------------*/
   if (!begin){
   if( ( get_bit(float_vec.stat,(10-get_bit(float_vec.stat,7))) != 
       get_bit(last_vec.stat,(10-get_bit(last_vec.stat,7))) ) && 
        ((last_vec.magvar/float_vec.magvar) < 10) &&
        ((float_vec.magvar/last_vec.magvar) < 10) )
      float_vec.stat=last_vec.stat;
   }
   stat0 = float_vec.stat;
/*----------------------------------------------------------------------------- 
 * Determine the phase difference between two successive vectors.             *
 -----------------------------------------------------------------------------*/
 delta_phase = tparp->ndec[sensor] / fgmfreq * spinfreq;
/*----------------------------------------------------------------------------- 
 * Fill the range into the status word of the float vector, transform the     *
 * integer vector to the float vector, correct for the non-normalized filters *
 * (in option 3, B, and D), and write the float vector to 'out_file'.         *
 * Then increase the time and the phase for the next output vector.           * 
 -----------------------------------------------------------------------------*/
 for (i = 0; i < ngot; i++)
   {
    if (((int_vec + i)->rng > 0) && ((int_vec + i)->rng < 8))
      {
       float_vec.stat |= ((int_vec + i)->rng << 4); 
/*----------------------------------------------------------------------------- 
 * Mark "BAD" vectors due to range changes                                    *
 -----------------------------------------------------------------------------*/
       if ((lastrng >= 0) && ((int_vec + i)->rng != lastrng) 
                                              && ((!gap) || (i>0)) ){
         if ( ((int_vec + i+1)->rng != lastrng) && (i < (ngot-1))){
          badvecnr = 3;
          if ( ((lastrng == 3) && ((int_vec + i)->rng == 4)) ||
               ((lastrng == 4) && ((int_vec + i)->rng == 3)) ) 
             {
              if (i>0) rngchng = 1; 
              if (option == 13)        
                 badvecnr = 6;
              else 
                 badvecnr = 4;
             }
           }
           else
           badvecnr = 1;  
         }
       if (badvecnr > 0){
          badvecnr--;
          set_bit(&float_vec.stat,26);
          }

       if ( (!badvecnr || badvecnr > 1) && ((int_vec + i+1)->rng > 0) )     
          lastrng = (int_vec + i)->rng;   

       iv2fv(float_vec.b, int_vec + i);
/*-----------------------------------------------------------------------------
 *  Mark as BAD vectors with range 1 or 7                                          *
 *----------------------------------------------------------------------------*/
   if (((int_vec + i)->rng < 2) || ((int_vec + i)->rng > 5))
      set_bit(&float_vec.stat,26);     
/*----------------------------------------------------------------------------- 
 * Output "BAD" vectors only if fgmtel option = "-a"                          *
 * Output last BUFFERED vector from the preceding reset if tdiff OK           *
 -----------------------------------------------------------------------------*/
      if ( (!begin  && (tdiff > (0.0007 * (double)sampling_period)) )
                    && ( all || !check_bad(last_vec.stat) ) ) 
           if ( write_fgmtvec(stdout, &last_vec) != sizeof(fgmtvec_t) )
             {
             fprintf ( stderr, errmsg[11], PROGRAM );
             exit(11);
             }
/*----------------------------------------------------------------------------- 
 * Correct time stamp and phase if significant range change (3<->4) occured   *
 -----------------------------------------------------------------------------*/    
      if (rngchng){
        inctime(&float_vec.tv, delttd);
        float_vec.phavar =  calc_spinphase( spinfreq, &sun_pulse_time0,
                                              &float_vec.tv, spinph0);
        rngchng = 0;
        }
      rce=mark_vector(unmark,&float_vec);
      last_vec=float_vec;
      begin=0;
      float_vec.stat = stat0; 
      inctime(&float_vec.tv, sampling_period);
      if ( tdiff < 0.0007 * (double)sampling_period )
         tdiff = tdiff + sampling_period; 
      float_vec.phavar =  calc_spinphase( spinfreq, &sun_pulse_time0,
                                     &float_vec.tv, spinph0);
//;      float_vec.phavar += delta_phase;
//;      float_vec.phavar = fmod(float_vec.phavar, 2 * M_PI);
     }
   }// for

/*----------------------------------------------------------------------------- 
 * Store last_vec time as 'old_vec_time' to be used for the time test of the  *
 * next reset period.                                                         * 
 -----------------------------------------------------------------------------*/
    old_vec_time = last_vec.tv;
}

//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
int main ( int argc, char *argv[] )
{
 FILE         *in_file = stdin;
 ddshead_t    head;
 auxdata_t    aux;
 int          i = 1, j = 0, k = 0, *file_index = 0, throw = 0,
              sensor = 0, sc = 0, mode, option, length, last_option=-1,
              warn_nr = 0, errmsg_cnt = 0, cnt_thresh = 175, last_cnt=-1;
 timesp       reset_pulse_time, old_reset_pulse_time = {0, 0};
 char         time_str[25], copt, *telstream = 0;
 int          prev_sc  = -1, bad_packets_cnt = 0, bad=0, iret, msa_cnt=0;
 int          dtsun  = 0, eclflag = 0, oldsun0 = 0, oldsun1 = 0, dtt = 0;
 float        dete=0.0;
// float vari0;

 clock_t start_time, end_time;
 start_time = clock();
 putenv("TZ=UTC0");

/*----------------------------------------------------------------------------- 
 * Allocate memory for the input_file_index array.                            *
 -----------------------------------------------------------------------------*/
 if (file_index)
    free(file_index);
 if ((file_index = (int *) calloc(argc, sizeof(int))) == NULL)
   {
    fprintf ( stderr, errmsg[2], PROGRAM );
    exit(2);
   }

/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 while ( i < argc )
   {
    if (*argv[i] == '-')                           /* Is it a program option? */
      {
       copt = *(argv[i]+1);
       switch(copt)
         {
         case 'h':
#ifndef SECURE
            fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n",
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
            fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
            exit(0);

         case 'a':
	   all = 1;
            if ( *(argv[i]+2) == '\0' && i+1 < argc )
               {
                i++;
                unmark = atoi ( argv[i] );
                if ( unmark <= 0 )
                i--;
               }
               else
                unmark = atoi ( argv[i]+2 );
            if (!unmark) 
               unmark = 7; 
            cnt_thresh = 5;
         break;

         case 'e':
            if ( *(argv[i]+2) == '\0' && i+1 <= argc )
               {
               if (  *argv[i+1] != '-' )
                 {
                 i++;
                 strcpy (ecln, argv[i]);
                 }
               }
               else
               strcpy (ecln, (argv[i]+2));
	   break;

         case 's': 
            sensor = 1;
            break;

         case 'v':
	   verbose = 1;
	   break;

         case 'V': 
            fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
            exit(0);
          
         default: 
            fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
            fprintf ( stderr, errmsg[0], argv[0] );
            exit(1);
         }
      }
    else                                      /* explicit input file name? */
      {
       *(file_index + j) = i;
       j++;
      }
    i++;
   }

 iret = 0;

 do 
   {
    if (j != 0)
/*----------------------------------------------------------------------------- 
 * Open in_file, if input file name(s) is(are) given in the command line.     *
 -----------------------------------------------------------------------------*/
      {
       in_file = fopen ( argv[*(file_index + k)], "r" );
       if (in_file == NULL)
         {
          fprintf ( stderr, errmsg[3], PROGRAM, argv[*(file_index + k)] );
          exit(3);
         }
      }
      
/*----------------------------------------------------------------------------- 
 * Reading the header bytes until end of byte stream.                         *
 -----------------------------------------------------------------------------*/
    while (read_ddshead(in_file, &head) == header_length)
      {

/*----------------------------------------------------------------------------- 
 * Determination of the spacecraft ID.                                        * 
 -----------------------------------------------------------------------------*/
       sc = scid(head.ID);

/*----------------------------------------------------------------------------- 
 * Test if the new packet is the same S/C.
 -----------------------------------------------------------------------------*/
      if( prev_sc>=0 && sc!=prev_sc )
         {
          fprintf ( stderr, errmsg[15], PROGRAM, prev_sc );
	 exit(15);
	 }
/*----------------------------------------------------------------------------- 
 * Determination of the packet time stamp.
 -----------------------------------------------------------------------------*/
       reset_pulse_time = cds2unix(head.t);

/*----------------------------------------------------------------------------- 
 * Test if the new packet is chronologically arranged 
 -----------------------------------------------------------------------------*/
       if (cmptime(&reset_pulse_time, &old_reset_pulse_time) <= 0)
         {
          fprintf ( stderr, errmsg[5], PROGRAM );
          get_time_str(&old_reset_pulse_time, time_str);
          fprintf ( stderr, " Last valid telemetry reset at: %s\n", time_str );
          get_time_str(&reset_pulse_time, time_str);
          fprintf ( stderr, " Following telemetry reset at:  %s\n", time_str );
          exit(5);
         }

/*----------------------------------------------------------------------------- 
 * Determination of the science packet data length and the S/C telemetry mode.* 
 -----------------------------------------------------------------------------*/
       warn_nr = determine_data_length_and_mode(&head, &length, &mode);

/*----------------------------------------------------------------------------- 
 * Reading the FGM science auxiliary data block.                              *
 -----------------------------------------------------------------------------*/
       if (!warn_nr)
         {
          if (read_auxdata(in_file, &aux) != auxiliary_length)
            {
             fprintf ( stderr, errmsg[6], PROGRAM );
             exit(6);
            }
         }

/*----------------------------------------------------------------------------- 
 * Read one data block from the input data stream. Store it in 'telstream'.   *
 -----------------------------------------------------------------------------*/
       read_telstream(in_file, &telstream, length);
       if (mode == 3) 
         {
         last_option=15;
         throw=1;
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         msa_cnt++;
         continue; 
         }  


//*----------------------------------------------------------------------------
//* FGM OFF?
//*----------------------------------------------------------------------------
      if ((aux.counts + aux.HFC_reset) == 0) {
         last_cnt=-1;    
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         bad_packets_cnt++;
         continue; 
         }  

/*----------------------------------------------------------------------------- 
 * Get the FGM telemetry option and test if it fits to S/C telemetry mode.    *
 -----------------------------------------------------------------------------*/
       option = subbits((int)aux.status_word, 0, 4);
       warn_nr = mode_option_test(mode, option);
/*----------------------------------------------------------------------------- 
 * Print message if option doesn't fit to TM-header                                             *
 -----------------------------------------------------------------------------*/
      if (warn_nr >0)
         {
         fprintf(stderr, warning[warn_nr], PROGRAM);
         if (verbose)
	 {
	   get_time_str(&reset_pulse_time, time_str);
	   fprintf(stderr, warning[warn_nr], PROGRAM );
	   fprintf(stderr,
	   " Data of reset period at %s could not be processed.\n",time_str);
	 }
	warn_nr = 0;
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         errmsg_cnt++; 
	continue;
         }

//*----------------------------------------------------------------------------
//* POWER-ON?
//*----------------------------------------------------------------------------
      if ( (last_cnt < 0) && (option == 12) && 
                              ((aux.counts < cnt_thresh) || (aux.counts == 28)))
         {
//         vari0=calc_rawvariance(aux.variance);
//         fprintf(stderr,"%5i%10.2f\n",aux.counts,vari0);
         last_option=option;
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         bad_packets_cnt++;
         continue; 
         }  

//*----------------------------------------------------------------------------
//* TM-Mode change?
//*----------------------------------------------------------------------------
      if ((throw > 0) || ((last_option != option) && (last_option >= 0))) {
         throw=0;
         last_option=option;
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         bad_packets_cnt++;
         continue; 
         }  

//*----------------------------------------------------------------------------
//* BAD data flag set?
//*----------------------------------------------------------------------------
     bad += (subbits((int)aux.status_word, 12, 16) > 0);
     if ((bad == 3) || (subbits((int)aux.status_word,12,16)==0)) bad=0;
     if (bad > 0) {
         prev_sc = sc;
         old_reset_pulse_time = reset_pulse_time;
         last_cnt=aux.counts;
         bad_packets_cnt++;
         continue; 
         }  
     
//*---------------------------------------------------------------------------- 
//* Warn, if time diff. of succ. TMpackets < nominal packet-rate - 5%       
//* ---------------------------------------------------------------------------
      get_time_str(&reset_pulse_time, time_str);
      dete=( reset_pulse_time.tv_sec - old_reset_pulse_time.tv_sec )+1.e-9*
           ( reset_pulse_time.tv_nsec - old_reset_pulse_time.tv_nsec);
      if (aux.counts<last_cnt) dtt=(int)aux.counts-(int)last_cnt+65536; 
   	              else dtt=(int)aux.counts-(int)last_cnt;
      if ((last_cnt < 0) || (dtt > 1000)) dete=0.0; else dete=dete-dtt*nmtmrate;
      if ((fabs(dete) > 0.05*nmtmrate) && (dete<0)) warn_nr=7;
      
//*---------------------------------------------------------------------------- 
//* Warn, if time difference of successive sun-pulses > spin frequ. +/- 5% *
//*----------------------------------------------------------------------------
      if ((aux.HFC_recent_sun-aux.HFC_previous_sun) < 0 )
	dtsun=aux.HFC_recent_sun-aux.HFC_previous_sun+0x10000;
	else
	dtsun=aux.HFC_recent_sun-aux.HFC_previous_sun;	
//*---------------------------------------------------------------------------- 
//* SET eclipse-flag, if sunpulse times didn't change from the previous packet 
//*---------------------------------------------------------------------------- 
      eclflag=(aux.HFC_recent_sun == oldsun1) && (aux.HFC_previous_sun == oldsun0);
      if ((dtsun >(int)(1.05*nmspin)) || (dtsun <(int)(0.95*nmspin))) warn_nr=8;
	
//*---------------------------------------------------------------------------- 
//* Print message if bad TM-packet-time, or spinrate and skip TM-packet                         
//*----------------------------------------------------------------------------
      if (warn_nr > 5)
         {
         fprintf(stderr, warning[warn_nr], PROGRAM);
         if (verbose)
	  {
//            fprintf(stderr,"%6i %6i %6i %7.1f %7.1f   %s\n",warn_nr,aux.counts,
//            last_cnt, dete*100./nmtmrate, fabs(dtsun-nmspin)/409.6/nmspin, time_str);
	   get_time_str(&reset_pulse_time, time_str);
	   fprintf(stderr, warning[warn_nr], PROGRAM );
	   fprintf(stderr,
	   " Data of reset period at %s could not be processed.\n",time_str);
	  }
	warn_nr = 0;
         prev_sc = sc;
         last_cnt=aux.counts;
         old_reset_pulse_time = reset_pulse_time;
         last_option=option;
         bad_packets_cnt++;
	continue;
         }
         
/*----------------------------------------------------------------------------- 
 * Save S/C id, day no and reset pulse time and sunpulses.                                  *
 -----------------------------------------------------------------------------*/
       prev_sc = sc;
       old_reset_pulse_time = reset_pulse_time;
       last_option = option;
       last_cnt = aux.counts;
       oldsun0 = aux.HFC_previous_sun;
       oldsun1 = aux.HFC_recent_sun;

/*----------------------------------------------------------------------------- 
 * Extract the vectors from the data stream block 'telstream'.                *
 -----------------------------------------------------------------------------*/
       extract_vectors( sensor, &reset_pulse_time, &telstream[0],
                        option, sc, &aux, eclflag );

   } //endwhile
/*----------------------------------------------------------------------------- 
 * Close in_file, if input file name(s) is(are) given in the command line.    *
 -----------------------------------------------------------------------------*/
    if (j != 0)
       fclose(in_file);
    k++;
  } while (j > k);

 if( verbose )
   {
     end_time = clock();
     fprintf(stderr, "fgmtel run time: %g sec\n",
	     (end_time - start_time)/((double) CLOCKS_PER_SEC));
   }
if (bad_packets_cnt > 0)
  fprintf (stderr," %10i empty or bad packets discarded \n",bad_packets_cnt );
if (errmsg_cnt > 0) {
  fprintf (stderr," %10i bad packets discarded due to ",errmsg_cnt );
  fprintf (stderr, warning[3], PROGRAM );
  }
if (msa_cnt > 0) 
  fprintf (stderr," %10i MSA packets discarded",msa_cnt );
exit(0);
}
