/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "iadp"
#define VERSION "5.0 (1996-12-18)"
#define PURPOSE "process CLUSTER FGM data interactively"
#define USAGE "USAGE:

 iadp

 The input to the program are normal or burst mode science data files
 containing the raw FGM data, like those distributed on the raw data
 medium (RDM)."

#define OPTIONS "OPTIONS:

  -V   print version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Stephan Buchert       (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke      (joerg@geophys.nat.tu-bs.de)"

#define MENUEHELP "
===============================================================================
 (a)scii ......... produce an ASCII listing of FGM vectors
 (c)oorsys ....... set the output coordinate system
 (d)ds ........... extract infos from the DDS header
 (e)xit .......... quit program
 (f)ile .......... read new input data file
 (h)elp .......... display this list of commands
 (i)nfo .......... produce an ASCII info listing for FGM vectors
 (l)ong .......... long binary output containing vectors with position
 for(m)at ........ set the output format for ASCII time strings
 (n)ext .......... select the next time intervall
 (r)ate .......... set the output data rate
 (s)hort ......... short binary output containing only vectors
 (t)ime .......... select new time intervall
 pre(v)ious ...... select the previous time intervall
"

#define MAX_FILENAMELENGTH 120
#define MAX_PATHLENGTH 96

#define PROMPT "IADP>"
#define HYPHENLINE "-------------------------------------------------------------------------------"
#define DOUBLELINE "==============================================================================="

#define RCFN ".iadprc"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "libutil.h"


static char *errmsg[] =
   {
/*  0 */  "Type '%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Could not open temporary file for storing the data.\n",
   };

static char *warning[] =
  {
/*  0 */  "",
/*  1 */  "%s WARNING: Valid input data not yet read.\n",
/*  2 */  "%s WARNING: Start of time intervall not yet selected.\n",
/*  3 */  "%s WARNING: End of time intervall not yet selected.\n",
/*  4 */  "%s WARNING: Output coordinate system not yet selected.\n",
/*  5 */  "%s WARNING: Output vector rate not yet selected.\n",
/*  6 */  "%s WARNING: Could not open pipe to determine period of input data.\n",
/*  7 */  "%s WARNING: No valid time period of input data.\n",
/*  8 */  "%s WARNING: Could not evaluate time specification.\n",
/*  9 */  "%s WARNING: Invalid time intervall specification.\n",
/* 10 */  "%s WARNING: End of actual time intervall already greater than end of input data.\n",
/* 11 */  "%s WARNING: Begin of actual time intervall already lower than begin of input data.\n",
/* 12 */  "%s WARNING: No valid data within selected time intervall.\n",
/* 13 */  "%s WARNING: Error during executing of command:\n%s.\n",
/* 14 */  "%s WARNING: Could not find input file.\n"
  };


static int decide ( char  *question, int def )
/*----------------------------------------------------------------------------- 
 * The function 'decide' prints 'question' to stderr and waits for an answer. *
 * The default value for the answer is defined by 'def' and can be adopted    *
 * by an empty input.                                                         *
 -----------------------------------------------------------------------------*/
{
 char   str[10];

 fprintf ( stderr, "%s\n", HYPHENLINE );
 if ( def )
    fprintf ( stderr, "%s (y/n)? [yes] ", question );
 else
    fprintf ( stderr, "%s (y/n)? [no] ", question );
 gets ( str );
 if ( str != "" )
   {
    if ( str[0] == 'y' || str[0] == 'Y' || str[0] == 'j' || str[0] == 'J' )
       return(1);
    else if ( str[0] == 'n' || str[0] == 'N' )
       return(0);
    else
       return(def);
   }
 else
    return(def);
}


static void remove_LF ( char *str )
/*----------------------------------------------------------------------------- 
 * The function 'remove_LF' removes a 'LF' character at the end of the        *
 * input string 'str'.                                                        *
 -----------------------------------------------------------------------------*/
{
 int   i;

 i = strlen ( str );
 if ( (int)str[i-1] == 10 ) 
    str[i-1] = '\0';
}


static void get_string ( char *newstr, char *oldstr, int keep_length )
/*----------------------------------------------------------------------------- 
 * The function 'get_string' reads a string from stdin controlled by          *
 * oldstring. If the new string is empty, 'oldstring' will be taken. For each *
 * space within 'newstring' the corresponding character of 'oldstring' is     *
 * taken. If keep_length is not zero, the length of the oldstring will not be *
 * changed even if the new string is shorter, ie only character replacements  *
 * are possible.                                                              *
 -----------------------------------------------------------------------------*/
{
 int   i, j, k;

 remove_LF ( oldstr );
 i = strlen ( oldstr );
 newstr[0] = '\0';
 fprintf ( stderr, "Current: %s\n", oldstr );
 fprintf ( stderr, "Changes: " );
 gets ( newstr );
 remove_LF ( newstr );
 j = strlen ( newstr );
 if ( !j )
    strcpy ( newstr, oldstr );
 else
   {
    if ( !keep_length )
      {
       i = ( ( i <= j ) ? i : j );
       j = 0;
       while ( j < i && newstr[j] == ' ' )
         {
          newstr[j] = oldstr[j];
          j++;
         }
      }
    else
      {
       for ( k = 0; k < i; k++ )
          if ( ( k < j && newstr[k] == ' ' ) || k >= j )
             newstr[k] = oldstr[k];
       newstr[i] = '\0';
      }
   }
}


static void evaluate_environment ( char  **envp, 
                                   char   *homepath,
                                   char   *logpath, 
                                   char   *datpath,
                                   char   *tmppath )
/*----------------------------------------------------------------------------- 
 * The function 'evaluate_environement' initializes some input and output     *
 * pathes by looking at the environment variables.                            *
 -----------------------------------------------------------------------------*/
{
 char   calpath[MAX_PATHLENGTH], currentpath[MAX_PATHLENGTH], 
        sattpath[MAX_PATHLENGTH], orbitpath[MAX_PATHLENGTH],
       *tmpdefault = "/tmp";
 int    i;

 homepath[0] = '\0';
 logpath[0] = '\0';
 datpath[0] = '\0';
 tmppath[0] = '\0';
 calpath[0] = '\0';
 currentpath[0] = '\0';
 sattpath[0] = '\0';
 orbitpath[0] = '\0';

 for ( i = 0; envp[i] != NULL; i++ )
   {
    if ( !strncmp ( envp[i], "HOME", 4 ) )
      {
       strncpy ( homepath, getenv ("HOME"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "FGMPATH", 7 ) )
      {
       strncpy ( calpath, getenv ("FGMPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "SATTPATH", 8 ) )
      {
       strncpy ( sattpath, getenv ("SATTPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "ORBITPATH", 9 ) )
      {
       strncpy ( orbitpath, getenv ("ORBITPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "FGMLOGPATH", 10 ) )
      {
       strncpy ( logpath, getenv ("FGMLOGPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "FGMDATPATH", 10 ) )
      {
       strncpy ( datpath, getenv ("FGMDATPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "FGMTMPPATH", 10 ) )
      {
       strncpy ( tmppath, getenv ("FGMTMPPATH"), MAX_PATHLENGTH - 2 );
       continue;
      }
    if ( !strncmp ( envp[i], "PWD", 3 ) )
      {
       strncpy ( currentpath, getenv ("PWD"), MAX_PATHLENGTH - 2 );
       continue;
      }
   }

 fprintf ( stderr, "\n" );
 fprintf ( stderr, "%s\n", DOUBLELINE );
 fprintf ( stderr, "The following pathes are set by the environment:\n" );
 fprintf ( stderr, "%s\n", HYPHENLINE );
 if ( homepath[0] != '\0' )
    fprintf ( stderr, "Path for program resource file:       HOME = %s\n", 
              homepath );
 if ( calpath[0] != '\0' )
    fprintf ( stderr, "Path for calibration files:        FGMPATH = %s\n", 
              calpath );
 if ( sattpath[0] != '\0' )
    fprintf ( stderr, "Path for S/C attitude files:      SATTPATH = %s\n", 
              sattpath );
 if ( orbitpath[0] != '\0' )
    fprintf ( stderr, "Path for S/C orbit files:        ORBITPATH = %s\n", 
              orbitpath );
 if ( logpath[0] != '\0' )
    fprintf ( stderr, "Path for output log files:      FGMLOGPATH = %s\n", 
              logpath );
 if ( datpath[0] != '\0' )
    fprintf ( stderr, "Path for output data files:     FGMDATPATH = %s\n", 
              datpath );
 if ( tmppath[0] != '\0' )
    fprintf ( stderr, "Path for temporary files:       FGMTMPPATH = %s\n", 
              tmppath );
 if ( homepath[0] == '\0' || calpath[0] == '\0' || sattpath[0] == '\0' ||
      orbitpath[0] == '\0' || logpath[0] == '\0' || datpath[0] == '\0' ) 
   {
    fprintf ( stderr, "%s\n", HYPHENLINE );
    fprintf ( stderr, 
              "The following pathes are set to the current directory:\n" );
    fprintf ( stderr, "%s\n", HYPHENLINE );
    if ( homepath[0] == '\0' )
      {
       strcpy ( homepath, currentpath );
       fprintf ( stderr, "Path for program resource file:       HOME = %s\n", 
                 homepath );
      }
    if ( calpath[0] == '\0' )
      {
       strcpy ( calpath, currentpath );
       fprintf ( stderr, "Path for calibration files:        FGMPATH = %s\n", 
                 calpath );
      }
    if ( sattpath[0] == '\0' )
      {
       strcpy ( sattpath, currentpath );
       fprintf ( stderr, "Path for S/C attitude files:      SATTPATH = %s\n", 
                 sattpath );
      }
    if ( orbitpath[0] == '\0' )
      {
       strcpy ( orbitpath, currentpath );
       fprintf ( stderr, "Path for S/C orbit files:        ORBITPATH = %s\n", 
                 orbitpath );
      }
    if ( logpath[0] == '\0' )
      {
       strcpy ( logpath, currentpath );
       fprintf ( stderr, "Path for output log files:      FGMLOGPATH = %s\n", 
                 logpath );
      }
    if ( datpath[0] == '\0' )
      {
       strcpy ( datpath, currentpath );
       fprintf ( stderr, "Path for output data files:     FGMDATPATH = %s\n", 
                 datpath );
      }
   }
 if ( tmppath[0] == '\0' )
   {
    fprintf ( stderr, "%s\n", HYPHENLINE );
    fprintf ( stderr, 
              "The following pathes are set to their default values:\n" );
    fprintf ( stderr, "%s\n", HYPHENLINE );
    if ( tmppath[0] == '\0' )
      {
       strcpy ( tmppath, tmpdefault );
       fprintf ( stderr, "Path for temporary files:       FGMTMPPATH = %s\n", 
                 tmppath );
      }
   }
 strcat ( homepath, "/" );
 strcat ( logpath, "/" );
 strcat ( datpath, "/" );
}


static void read_rcfile ( char  *homepath, 
                          char  *filename, 
                          char  *selected_beginstr,
                          char  *selected_endstr,
                          char  *coorsys,
                          char  *rate,
                          char  *timeformat )
/*----------------------------------------------------------------------------- 
 * The function 'read_rcfile' tries to open the program resource file to read *
 * the previous settings for various variables.                               *
 -----------------------------------------------------------------------------*/
{
 FILE  *rcfile;
 char   rcfilename[MAX_FILENAMELENGTH], 
        read_str[MAX_FILENAMELENGTH + 8],
       *tmpstr;

 filename[0] = '\0';
 selected_beginstr[0] = '\0';
 selected_endstr[0] = '\0';
 coorsys[0] = '\0';
 rate[0] = '\0';
 timeformat[0] = '\0';
 strcpy ( rcfilename, homepath );
 strcat ( rcfilename, RCFN );

 fprintf ( stderr, "%s\n", DOUBLELINE );
 fprintf ( stderr, "Try to read program-resource-file %s\n", rcfilename );

 rcfile = fopen ( rcfilename, "r" );
 if ( rcfile != NULL )
   {
    while ( fgets ( read_str, MAX_FILENAMELENGTH + 8, rcfile ) != NULL )
      {
       if ( strlen ( read_str ) >= 8 )
          {
           if ( !strncmp ( read_str, "DATFILE=", 8 ) )
             {
              strcpy ( filename, &read_str[8] );
              remove_LF ( filename );
             }
           else if ( !strncmp ( read_str, "BEGTIME=", 8 ) )
             {
              strcpy ( selected_beginstr, &read_str[8] );
              remove_LF ( selected_beginstr );
             }
           else if ( !strncmp ( read_str, "ENDTIME=", 8 ) )
             {
              strcpy ( selected_endstr, &read_str[8] );
              remove_LF ( selected_endstr );
             }
           else if ( !strncmp ( read_str, "COORSYS=", 8 ) )
             {
              strcpy ( coorsys, &read_str[8] );
              remove_LF ( coorsys );
             }
           else if ( !strncmp ( read_str, "DATRATE=", 8 ) )
             {
              strcpy ( rate, &read_str[8] );
              remove_LF ( read_str );
             }
           else if ( !strncmp ( read_str, "TIMEFMT=", 8 ) )
             {
              strcpy ( timeformat, &read_str[8] );
              remove_LF ( timeformat );
             }
          }
      }
    fclose ( rcfile );
   }

 tmpstr = strpbrk ( timeformat, "01234" );
 if ( tmpstr == NULL )
    strcpy ( timeformat, "0" );
 else
   {
    timeformat[0] = tmpstr[0];
    timeformat[1] = '\0';
   }

 fprintf ( stderr, "%s\n", HYPHENLINE );
 fprintf ( stderr, "The following parameters have been set:\n" );

 fprintf ( stderr, "Input data file:              " );
 if ( strlen ( filename ) )
    fprintf ( stderr, "%s\n", filename );
 else
    fprintf ( stderr, "not specified\n" );

 fprintf ( stderr, "Start time of data intervall: " );
 if ( strlen ( selected_beginstr ) )
    fprintf ( stderr, "%s\n", selected_beginstr );
 else
    fprintf ( stderr, "not specified\n" );

 fprintf ( stderr, "End time of data intervall:   " );
 if ( strlen ( selected_endstr ) )
    fprintf ( stderr, "%s\n", selected_endstr );
 else
    fprintf ( stderr, "not specified\n" );

 if ( !strlen ( coorsys ) )
    fprintf ( stderr, "Output coordinate system:     not specified\n" );

 if ( !strlen ( rate ) )
    fprintf ( stderr, "Output vector rate:           not specified\n" );
}


static void write_rcfile ( char  *homepath, 
                           char  *filename, 
                           char  *selected_beginstr,
                           char  *selected_endstr,
                           char  *coorsys,
                           char  *rate,
                           char  *timeformat )
/*----------------------------------------------------------------------------- 
 * The function 'write_rcfile' tries to open the program resource file to     *
 * write the actual settings for various variables.                           *
 -----------------------------------------------------------------------------*/
{
 FILE  *rcfile;
 char   rcfilename[MAX_FILENAMELENGTH];

 strcpy ( rcfilename, homepath );
 strcat ( rcfilename, RCFN );

 rcfile = fopen ( rcfilename, "w" );
 if ( rcfile != NULL )
   {
    fprintf ( rcfile, "DATFILE=%s\n", filename );
    fprintf ( rcfile, "BEGTIME=%s\n", selected_beginstr );
    fprintf ( rcfile, "ENDTIME=%s\n", selected_endstr );
    fprintf ( rcfile, "COORSYS=%s\n", coorsys );
    fprintf ( rcfile, "DATRATE=%s\n", rate );
    fprintf ( rcfile, "TIMEFMT=%s\n", timeformat );
    fclose ( rcfile );
   }
}


static int set_intervall ( char  *selected_tmpname, 
                           char  *filename,
                           char  *selected_beginstr,
                           char  *selected_endstr,
                           int   *gotdata )
/*----------------------------------------------------------------------------- 
 * Sets the selected intervall by filling 'selected_tmpname' with the data    *
 * of the specified intervall.                                                *
 -----------------------------------------------------------------------------*/
{
 FILE  *pfile;
 char   cmd[MAX_PATHLENGTH + L_tmpnam + MAX_FILENAMELENGTH + 180], 
        timecmd[60], c;
 int    size, i;

 *gotdata = 0;
 fprintf ( stderr, "%s\n", HYPHENLINE );
 fprintf ( stderr, "Selected time intervall\n" );
 fprintf ( stderr, "Begin: %s\n", selected_beginstr );
 fprintf ( stderr, "End:   %s\n", selected_endstr );
 sprintf ( timecmd, " -b %s -e %s ", selected_beginstr, selected_endstr );
 strcpy ( cmd, "ddsmrg " );
 strcat ( cmd, filename );
 strcat ( cmd, " | ddscut" );
 strcat ( cmd, timecmd );
 strcat ( cmd, " | fgmtel" );
 strcat ( cmd, " 2> /dev/null " );
 strcat ( cmd, " | fgmcut " );
 strcat ( cmd, timecmd );
 strcat ( cmd, " > " );
 strcat ( cmd, selected_tmpname );
 fprintf ( stderr, "%s\n", HYPHENLINE );
 fprintf ( stderr, "Storing selected intervall of data.\n" );
 fprintf ( stderr, "Please wait a moment.\n" );
 
 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
    
 strcpy ( cmd, "ls -s " );
 strcat ( cmd, selected_tmpname );
 pfile = popen ( cmd, "r" );
 if ( pfile == NULL )
    return(6);
 fscanf ( pfile, "%d", &size );
 while ( fscanf ( pfile, "%c", &c ) == 1 )
    ;
 if ( size == 0 )
   {
    i = pclose ( pfile );
    return(12);
   }
 i = pclose ( pfile );
 *gotdata = 1;
 return(0);
}


static int select_intervall ( char  *selected_tmpname, 
                              char  *filename,
                              char  *total_beginstr,
                              char  *total_endstr,
                              char  *selected_beginstr,
                              char  *selected_endstr,
                              int   *gotdata )
/*----------------------------------------------------------------------------- 
 * Asks for a time intervall specification and puts the data of that          *
 * intervall in the temporary file 'selected_tmpname' by using the function   *
 * 'set_intervall'.                                                           *
 -----------------------------------------------------------------------------*/
{
 char   tmpstr[25];
 int    i;

 if ( selected_beginstr[0] == '\0' )
    strcpy ( selected_beginstr, total_beginstr );
 if ( selected_endstr[0] == '\0' )
    strcpy ( selected_endstr, total_endstr );

 fprintf ( stderr, "%s\n", DOUBLELINE );
 fprintf ( stderr, "Begin of input data period: %s\n", total_beginstr );
 fprintf ( stderr, "End of input data period:   %s\n", total_endstr );
 fprintf ( stderr, "%s\n", HYPHENLINE );

 fprintf ( stderr, "Select time intervall to be processed\n" );
 fprintf ( stderr, "Start time\n" );
 get_string ( tmpstr, selected_beginstr, 1 );
 strcpy ( selected_beginstr, tmpstr );
 fprintf ( stderr, "End time\n" );
 get_string ( tmpstr, selected_endstr, 1 );
 strcpy ( selected_endstr, tmpstr );
 i = set_intervall ( selected_tmpname,
                     filename,
                     selected_beginstr,
                     selected_endstr,
                     gotdata );
 return(i);
}


static int get_data ( char  *selected_tmpname,
                      char  *filename,
                      char  *filenametrunk,
                      char  *homepath,
                      char  *logpath,
                      char  *total_beginstr,
                      char  *total_endstr,
                      char  *selected_beginstr,
                      char  *selected_endstr,
                      int   *gotdata )
/*----------------------------------------------------------------------------- 
 * The function 'get_data' reads the data from the file 'filename' which may  *
 * contain wild cards sent them directly to a pipe to determine the time of   *
 * the first and the last vector which will be stored in 'total_beginstr' and *
 * 'total_endstr' respectively. The error messages can be written to a        *
 * selected file. After this a time intervall is selected and the data will   *
 * be stored in the temporary file 'selected_tmpname'. The time of the first  *
 * and the last vector of the selected intervall will be stored in            *
 * 'selected_beginstr' and 'selected_endstr'.                                 *
 -----------------------------------------------------------------------------*/
{
 FILE       *pfile;
 char        cmd[2 * MAX_FILENAMELENGTH + 25],
             readstr[MAX_FILENAMELENGTH + 4], filestr[MAX_FILENAMELENGTH + 4];
 int         i, length, charpos = 0;
 fgmtvec_t   vec, firstvec, lastvec;

/*----------------------------------------------------------------------------- 
 * Set the home directory for a first character '~'.                          *
 -----------------------------------------------------------------------------*/
 if ( filename[0] == '~' )
   {
    strcpy ( filestr, filename );
    strcpy ( filename, homepath );
    if ( filestr[1] == '/' )
       strcat ( filename, &filestr[2] );
    else
       strcat ( filename, &filestr[1] );
   }
/*----------------------------------------------------------------------------- 
 * Copy the input filename 'filename' to 'filenametrunk'. Start after the     *
 * last directory character ('/') and stop if a 'wild card' ('*' or '?')      *
 * occurs.                                                                    *
 -----------------------------------------------------------------------------*/
 length = strlen ( filename );
 for ( i = 0; i < length; i++ )
    if ( filename[i] == '/' )
       charpos = i + 1;
 i = 0;
 while ( charpos < length && 
         filename[charpos] != '*' && 
         filename[charpos] != '?' )
   {
    filenametrunk[i] = filename[charpos];
    i++;
    charpos++;
   }
 filenametrunk[i] = '\0';

 fprintf ( stderr, "%s\n", DOUBLELINE );
 fprintf ( stderr, "The following files have been selected:\n" );
 fprintf ( stderr, "%s\n", HYPHENLINE );
 strcpy ( cmd, "ls -d -F -1 " );
 strcat ( cmd, filename );

 i = system ( cmd ); 

 if ( i )
    return(14);
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
    
/*----------------------------------------------------------------------------- 
 * Read the input data from the file, transform them into the fgmtvec_t       *
 * structure using the program 'fgmtel', and store the output in the          *
 * temporary data storage file. Write the error messages to the logfile.      *
 -----------------------------------------------------------------------------*/
 strcpy ( cmd, "ddsmrg " );
 strcat ( cmd, filename );
 strcat ( cmd, " | fgmtel" );
 if ( decide ( "Shall the error messages be written to a file", 1 ) )
   {
    strcpy ( filestr, logpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".log" );
    fprintf ( stderr, "Output file for error messages\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " 2> " );
    strcat ( cmd, filestr );
   }

/*----------------------------------------------------------------------------- 
 * Determine the time of the first and the last input vector.                 *
 -----------------------------------------------------------------------------*/
 total_beginstr[0] = '\0';
 total_endstr[0] = '\0';
 fprintf ( stderr, "%s\n", HYPHENLINE );
 fprintf ( stderr, "Determination of start and end time of input data.\n" );
 fprintf ( stderr, "Please wait a moment.\n" );
 pfile = popen ( cmd, "r" );
 if ( pfile == NULL )
    return(6);
 if ( read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
   {
    firstvec = vec;
    get_time_str ( &vec.tv, total_beginstr );
    lastvec = vec;
   }
 else
   {
    i = pclose ( pfile );
    return(7);
   }
 while ( read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
    lastvec = vec;
 get_time_str ( &lastvec.tv, total_endstr );
 i = pclose ( pfile );
 i = select_intervall ( selected_tmpname,
                        filename,
                        total_beginstr, 
                        total_endstr,
                        selected_beginstr, 
                        selected_endstr,
                        gotdata );
 return(i);
}


static int select_files ( char  *selected_tmpname,
                          char  *filename,
                          char  *filenametrunk,
                          char  *homepath,
                          char  *logpath,
                          char  *total_beginstr,
                          char  *total_endstr,
                          char  *selected_beginstr,
                          char  *selected_endstr,
                          int   *gotdata )
/*----------------------------------------------------------------------------- 
 * Reads the name of the input file which might contain 'wild cards'          *
 * interactively from stdin and reads the data using the function 'get_data'. *
 -----------------------------------------------------------------------------*/
{
 char   new_filename[MAX_FILENAMELENGTH];
 int    i;

 fprintf ( stderr, "Select new FGM input raw data file\n" );
 get_string ( new_filename, filename, 0 );
 strcpy ( filename, new_filename );
 i = get_data ( selected_tmpname,
                filename,
                filenametrunk,
                homepath,
                logpath,
                total_beginstr,
                total_endstr,
                selected_beginstr,
                selected_endstr,
                gotdata );
 return(i);
}


static int test_settings ( int    gotdata, 
                           char  *selected_beginstr, 
                           char  *selected_endstr, 
                           char  *coorsys, 
                           char  *rate )
/*----------------------------------------------------------------------------- 
 * Test whether all neccessary parameters have been set.                      *
 -----------------------------------------------------------------------------*/
{
 if ( !gotdata )
    return(1);
 if ( selected_beginstr[0] == '\0' )
    return(2);
 if ( selected_endstr[0] == '\0' )
    return(3);
 if ( coorsys[0] == '\0' )
    return(4);
 if ( rate[0] == '\0' )
    return(5);
 return(0);
}


static void set_coorsys ( char *coorsys, char *coorsyscmd )
/*----------------------------------------------------------------------------- 
 * Set the output coordinate system.                                          *
 -----------------------------------------------------------------------------*/
{
 int    i;
 char   c[10];

 for ( i = 0; i < strlen ( coorsys ); i++ )
    c[i] = tolower ( coorsys[i] );
 switch ( c[0] )
   {
    case 'g':
       if ( c[2] == 'e' )
         {
          fprintf ( stderr, "Output coordinate system is GSE\n" );
          strcpy ( coorsys, "gse" );
          strcpy ( coorsyscmd, " -s gse " );
         }
       else if ( c[2] == 'm' )
         {
          fprintf ( stderr, "Output coordinate system is GSM\n" );
          strcpy ( coorsys, "gsm" );
          strcpy ( coorsyscmd, " -s gsm " );
         }
       else
         {
          fprintf ( stderr, "By default Output coordinate system is GSE\n" );
          strcpy ( coorsys, "gse" );
          strcpy ( coorsyscmd, " -s gse " );
         }
       break;

    case 'b':
       fprintf ( stderr, "Output coordinate system is Body-Build\n" );
       strcpy ( coorsys, "bb" );
       strcpy ( coorsyscmd, " -s bb " );
       break;

    case 's':
       if ( c[1] == 'r' )
         {
          fprintf ( stderr, "Output coordinate system is Spin Reference\n" );
          strcpy ( coorsys, "sr" );
          strcpy ( coorsyscmd, " -s sr " );
         }
       else if ( c[1] == 'c' )
         {
          fprintf ( stderr, "Output coordinate system is Spacecraft-Sun\n" );
          strcpy ( coorsys, "scs" );
          strcpy ( coorsyscmd, " -s scs " );
         }
       else if ( c[1] == 'm' )
         {
          fprintf ( stderr, "Output coordinate system is Solar Magnetic\n" );
          strcpy ( coorsys, "sm" );
          strcpy ( coorsyscmd, " -s sm " );
         }
       else
         {
          fprintf ( stderr, "By default Output coordinate system is GSE\n" );
          strcpy ( coorsys, "gse" );
          strcpy ( coorsyscmd, " -s gse " );
         }
       break;

    default:
       fprintf ( stderr, "By default Output coordinate system is GSE\n" );
       strcpy ( coorsys, "gse" );
       strcpy ( coorsyscmd, " -s gse " );
   }
}


static void select_coorsys ( char *coorsys, char *coorsyscmd )
/*----------------------------------------------------------------------------- 
 * Select the output coordinate system.                                       *
 -----------------------------------------------------------------------------*/
{
 char   tmpstr[10];

 fprintf ( stderr, "Select output coordinate system\n" );
 fprintf ( stderr, "sr  = Spin Reference system\n" );
 fprintf ( stderr, "scs = Spacecraft-Sun system\n" );
 fprintf ( stderr, "gse = Geocentric Solar Ecliptic system\n" );
 fprintf ( stderr, "gsm = Geocentric Solar Magnetospheric system\n" );
 fprintf ( stderr, "sm  = Solar Magnetic system\n" );
 fprintf ( stderr, "bb  = Body-Build system\n" );
 get_string ( tmpstr, coorsys, 0 );
 strcpy ( coorsys, tmpstr );
 fprintf ( stderr, "%s\n", HYPHENLINE );
 set_coorsys ( coorsys, coorsyscmd );
}


static void set_rate ( char *rate, char *ratecmd )
/*----------------------------------------------------------------------------- 
 * Set the output data rate.                                                  *
 -----------------------------------------------------------------------------*/
{
 char    c;
 int     i;
 float   f;

 c = tolower ( rate[0] );
 switch ( c )
   {
    case 'h':
       strcpy ( rate, "h" );
       ratecmd[0] = '\0';
       fprintf ( stderr, "Data will not be averaged (high resolution)\n" );
       break;

    case 's':
       if ( sscanf ( &rate[1], "%d", &i ) != 1 )
          i = 1;
       sprintf ( rate, "s%d", i );
       sprintf ( ratecmd, " -s %d ", i ); 
       if ( i == 1 )
          fprintf ( stderr, "Data will be averaged over %d second\n" , i );
       else
          fprintf ( stderr, "Data will be averaged over %d seconds\n" , i );
       break;

    case 'm':
       if ( sscanf ( &rate[1], "%d", &i ) != 1 )
          i = 1;
       sprintf ( rate, "m%d", i );
       sprintf ( ratecmd, " -m %d ", i );
       if ( i == 1 )
          fprintf ( stderr, "Data will be averaged over %d minute\n" , i );
       else
          fprintf ( stderr, "Data will be averaged over %d minutes\n" , i );
       break;

    case 'p':
       if ( sscanf ( &rate[1], "%f", &f ) != 1 )
          f = 0.0;
       sprintf ( rate, "p%7.3f", f );
       sprintf ( ratecmd, " -p %f ", f );
       fprintf ( stderr, 
                 "Data will be spin averaged with phase offset %7.3f deg\n" ,
                f );
       break;

    default:
       strcpy ( rate, "s1" );
       strcpy ( ratecmd, " -s 1 " );
       fprintf ( stderr, 
                 "By default data will be averaged over 1 second\n" );
   }
}


static void select_rate ( char *rate, char *ratecmd )
/*----------------------------------------------------------------------------- 
 * Select the output data rate.                                               *
 -----------------------------------------------------------------------------*/
{
 char   tmpstr[20];

 fprintf ( stderr, "Select output data rate\n" );
 fprintf ( stderr, "h        = high resolution (no average)\n" );
 fprintf ( stderr, "s<int>   = data averaged over <int> seconds\n" );
 fprintf ( stderr, "m<int>   = data averaged over <int> minutes\n" );
 fprintf ( stderr, "p<float> = data averaged over one spin (phase offset = <float> deg)\n" );
 get_string ( tmpstr, rate, 0 );
 strcpy ( rate, tmpstr );
 fprintf ( stderr, "%s\n", HYPHENLINE );
 set_rate ( rate, ratecmd );
}


static void select_timeformat ( char *timeformat )
/*----------------------------------------------------------------------------- 
 * Select the output format for ASCII time strings.                           *
 -----------------------------------------------------------------------------*/
{
 char   newstr[3], *tmpstr;

 fprintf ( stderr, "Select output format for ASCII time strings\n" );
 fprintf ( stderr, 
           "0 = ISO standard time string like '1996-12-02T02:05:15.398Z\n" );
 fprintf ( stderr, 
           "1 = (int) seconds of the UNIX epoch, and (int) nanoseconds\n" );
 fprintf ( stderr, "2 = (float) seconds of the hour\n" );
 fprintf ( stderr, "3 = (float) hours of the day\n" );
 fprintf ( stderr, "4 = character string like 'Mon Dec 02 02:05:15 1996'\n" );
 get_string ( newstr, timeformat, 1 );
 tmpstr = strpbrk ( newstr, "01234" );
 if ( tmpstr == NULL )
   {
    fprintf ( stderr, "By default the output format will be the ISO standard time string.\n" );
    strcpy ( timeformat, "0" );
   }
 else
   {
    timeformat[0] = tmpstr[0];
    timeformat[1] = '\0';
   }
}


static void make_ascii ( char  *selected_tmpname, 
                         char  *logpath,
                         char  *filenametrunk, 
                         char  *coorsyscmd,
                         char  *ratecmd,
                         char  *timeformat )
/*----------------------------------------------------------------------------- 
 * Produce an ASCII output of the vector stream.                              *
 -----------------------------------------------------------------------------*/
{
 char   cmd[MAX_PATHLENGTH + L_tmpnam + MAX_FILENAMELENGTH + 100],
        readstr[MAX_FILENAMELENGTH + 4], filestr[MAX_FILENAMELENGTH + 4];
 int    i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 strcat ( cmd, " | fgmcal " );
 strcat ( cmd, " | fgmhrt " );
 strcat ( cmd, coorsyscmd );
 if ( ratecmd[0] != '\0' )
   {
    strcat ( cmd, " | fgmav " );
    strcat ( cmd, ratecmd );
   }
 strcat ( cmd, " | fgmvec -r -p -t" );
 strcat ( cmd, timeformat );

 if ( decide ( "Shall the output be written to a file", 1 ) )
   {
    strcpy ( filestr, logpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".asc" );
    fprintf ( stderr, "Output ascii file name\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " > " );
    strcat ( cmd, filestr );
   }
 else
    fprintf ( stderr, "%s\n", DOUBLELINE );

 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
}


static void extract_dds ( char  *logpath,
                          char  *filename,
                          char  *filenametrunk,
                          char  *selected_beginstr,
                          char  *selected_endstr,
                          char  *timeformat )
/*----------------------------------------------------------------------------- 
 * Extracts infos from the DDS header and writes them to a file or to stdout. *
 -----------------------------------------------------------------------------*/
{
 char   cmd[3 * MAX_FILENAMELENGTH + 110],
        timecmd[60],
        readstr[MAX_FILENAMELENGTH + 4],
        filestr[MAX_FILENAMELENGTH + 4];
 int    i;

 sprintf ( timecmd, " -b %s -e %s ", selected_beginstr, selected_endstr );
 strcpy ( cmd, "ddsmrg " );
 strcat ( cmd, filename );
 strcat ( cmd, " | ddscut " );
 strcat ( cmd, timecmd );
 strcat ( cmd, " | ddsls -t" );
 strcat ( cmd, timeformat );
 strcat ( cmd, " -f '" );
 strcat ( cmd, filename );
 strcat ( cmd, "'" );
 if ( decide ( "Shall the output be written to a file", 1 ) )
   {
    strcpy ( filestr, logpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".dds" );
    fprintf ( stderr, "Output file name for DDS header info\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " > " );
    strcat ( cmd, filestr );
   }
 else
    fprintf ( stderr, "%s\n", DOUBLELINE );

 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
}


static void info_vec ( char  *selected_tmpname, 
                       char  *logpath,
                       char  *filenametrunk,
                       char  *timeformat )
/*----------------------------------------------------------------------------- 
 * Produce an ASCII info output for the FGM vectors.                          *
 -----------------------------------------------------------------------------*/
{
 char   cmd[MAX_PATHLENGTH + L_tmpnam + MAX_FILENAMELENGTH + 30],
        readstr[MAX_FILENAMELENGTH + 4], filestr[MAX_FILENAMELENGTH + 4];
 int    i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 strcat ( cmd, " | fgmls -t" );
 strcat ( cmd, timeformat );
 if ( decide ( "Shall the output be written to a file", 1 ) )
   {
    strcpy ( filestr, logpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".inf" );
    fprintf ( stderr, "Output ascii file name\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " > " );
    strcat ( cmd, filestr );
   }
 else
    fprintf ( stderr, "%s\n", DOUBLELINE );

 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
}


static void long_bin ( char  *selected_tmpname, 
                       char  *datpath,
                       char  *filenametrunk, 
                       char  *coorsyscmd,
                       char  *ratecmd )
/*----------------------------------------------------------------------------- 
 * Produce a long binary output containing vectors with position.             *
 -----------------------------------------------------------------------------*/
{
 char   cmd[MAX_PATHLENGTH + L_tmpnam + MAX_FILENAMELENGTH + 80],
        readstr[MAX_FILENAMELENGTH + 4], filestr[MAX_FILENAMELENGTH + 4];
 int    i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 strcat ( cmd, " | fgmcal " );
 strcat ( cmd, " | fgmhrt " );
 strcat ( cmd, coorsyscmd );
 if ( ratecmd[0] != '\0' )
   {
    strcat ( cmd, " | fgmav " );
    strcat ( cmd, ratecmd );
   }
 strcat ( cmd, " | fgmpos " );

 if ( decide ( "Shall the output be written to a file", 1 ) )
   {
    strcpy ( filestr, datpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".lbv" );
    fprintf ( stderr, "Output binary file name\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " > " );
    strcat ( cmd, filestr );
   }
 else
    fprintf ( stderr, "%s\n", DOUBLELINE );

 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
}


static int select_next_intervall ( char  *selected_tmpname,
                                   char  *filename,
                                   char  *total_endstr,
                                   char  *selected_beginstr,
                                   char  *selected_endstr,
                                   int   *gotdata )
/*----------------------------------------------------------------------------- 
 * Set the next time intervall by adding the time difference between          *
 * 'selected_beginstr' and 'selected_endstr'. Don't set the new intervall,    *
 * if 'selected_endstr' is already greater than 'total_endstr'.               *
 -----------------------------------------------------------------------------*/
{
 FILE        *pfile;
 fgmtvec_t    vec, lastvec;
 timesp       timediff, total_end, selected_begin, selected_end;
 char         cmd[MAX_PATHLENGTH + L_tmpnam + 4];
 int          i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 pfile = popen ( cmd, "r" );
 if ( pfile == NULL )
    return(6);
 if ( !read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
   {
    i = pclose ( pfile );
    return(7);
   }
 else
   {
    lastvec = vec;
    if ( !timestr2unix ( selected_beginstr, lastvec.tv, &selected_begin ) )
      {
       i = pclose ( pfile );
       return(8);
      }
   }
 while ( read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
    lastvec = vec;
 i = pclose ( pfile );
 if ( !timestr2unix ( selected_endstr, lastvec.tv, &selected_end ) ||
      !timestr2unix ( total_endstr, lastvec.tv, &total_end ))
    return(8);
 if ( cmptime ( &selected_end, &selected_begin ) <= 0 )
    return(9);
 if ( cmptime ( &selected_end, &total_end ) >= 0 )
    return(10);

 timediff = subtime ( &selected_end, &selected_begin );
 selected_begin = selected_end;
 selected_end.tv_sec += timediff.tv_sec;
 inctime ( &selected_end, timediff.tv_nsec );
 get_time_str ( &selected_begin, selected_beginstr );
 get_time_str ( &selected_end, selected_endstr );

 i = set_intervall ( selected_tmpname,
                     filename,
                     selected_beginstr, 
                     selected_endstr,
                     gotdata );
 return(i);
}


static void short_bin ( char  *selected_tmpname, 
                        char  *datpath,
                        char  *filenametrunk, 
                        char  *coorsyscmd,
                        char  *ratecmd )
/*----------------------------------------------------------------------------- 
 * Produce a short binary output containing only vectors without position.    *
 -----------------------------------------------------------------------------*/
{
 char   cmd[MAX_PATHLENGTH + L_tmpnam + MAX_FILENAMELENGTH + 70],
        readstr[MAX_FILENAMELENGTH + 4], filestr[MAX_FILENAMELENGTH + 4];
 int    i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 strcat ( cmd, " | fgmcal " );
 strcat ( cmd, " | fgmhrt " );
 strcat ( cmd, coorsyscmd );
 if ( ratecmd[0] != '\0' )
   {
    strcat ( cmd, " | fgmav " );
    strcat ( cmd, ratecmd );
   }
 if ( decide ( "Shall the output be written to a file", 1 ) )
   {
    strcpy ( filestr, datpath );
    strcat ( filestr, filenametrunk );
    strcat ( filestr, ".sbv" );
    fprintf ( stderr, "Output binary file name\n" );
    get_string ( readstr, filestr, 0 );
    strcpy ( filestr, readstr );
    strcat ( cmd, " > " );
    strcat ( cmd, filestr );
   }
 else
    fprintf ( stderr, "%s\n", DOUBLELINE );

 i = system ( cmd ); 
/*
 if ( i )
    fprintf ( stderr, warning[13], PROGRAM, cmd );
*/
}


static int select_previous_intervall ( char  *selected_tmpname,
                                       char  *filename,
                                       char  *total_beginstr,
                                       char  *selected_beginstr,
                                       char  *selected_endstr,
                                       int   *gotdata )
/*----------------------------------------------------------------------------- 
 * Set the previous time intervall by subtracting the time difference between *
 * 'selected_beginstr' and 'selected_endstr'. Don't set the new intervall,    *
 * if 'selected_beginstr' is already lower than 'total_beginstr'.             *
 -----------------------------------------------------------------------------*/
{
 FILE        *pfile;
 fgmtvec_t    vec, lastvec;
 timesp      timediff, total_begin, selected_begin, selected_end;
 char         cmd[MAX_PATHLENGTH + L_tmpnam + 4];
 int          i;

 strcpy ( cmd, "cat " );
 strcat ( cmd, selected_tmpname );
 pfile = popen ( cmd, "r" );
 if ( pfile == NULL )
    return(6);
 if ( !read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
   {
    i = pclose ( pfile );
    return(7);
   }
 else
   {
    lastvec = vec;
    if ( !timestr2unix ( selected_beginstr, lastvec.tv, &selected_begin ) ||
         !timestr2unix ( total_beginstr, lastvec.tv, &total_begin ) )
      {
       i = pclose ( pfile );
       return(8);
      }
   }
 while ( read_fgmtvec ( pfile, &vec) == sizeof ( fgmtvec_t ) )
    lastvec = vec;
 i = pclose ( pfile );
 if ( !timestr2unix ( selected_endstr, lastvec.tv, &selected_end ) )
    return(8);
 if ( cmptime ( &selected_end, &selected_begin ) <= 0 )
    return(9);
 if ( cmptime ( &total_begin, &selected_begin ) >= 0 )
    return(11);

 timediff = subtime ( &selected_end, &selected_begin );
 selected_end = selected_begin;
 selected_begin.tv_sec -= timediff.tv_sec;
 if ( selected_begin.tv_nsec < timediff.tv_nsec )
   {
    selected_begin.tv_nsec += NANOSEC;
    selected_begin.tv_sec--;
   }
 selected_begin.tv_nsec -= timediff.tv_nsec;
 get_time_str ( &selected_begin, selected_beginstr );
 get_time_str ( &selected_end, selected_endstr );

 i = set_intervall ( selected_tmpname,
                     filename,
                     selected_beginstr, 
                     selected_endstr,
                     gotdata );
 return(i);
}


int main ( int argc, char *argv[], char *envp[] )
{
 FILE  *selected_tmpfile = NULL;
 char   copt,
        homepath[MAX_PATHLENGTH],
        logpath[MAX_PATHLENGTH],
        datpath[MAX_PATHLENGTH],
        tmppath[MAX_PATHLENGTH],
        filename[MAX_FILENAMELENGTH], 
        filenametrunk[MAX_FILENAMELENGTH],
       *selected_tmpname,
        rate[10], ratecmd[15],
        coorsys[10], coorsyscmd[15],
        timeformat[3],
        total_beginstr[25], total_endstr[25],
        selected_beginstr[25], selected_endstr[25],
        menuecmd[10];
 int    i = 1, warnnr = 0, gotdata = 0, ciao = 0;


/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 while ( i < argc )
   {
    if ( *argv[i] == '-' )
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
             fprintf ( stderr, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
             exit(0);

          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], argv[0], argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] ); 
             exit(1);
         }
      }
    i++;
   }
 
 printf ( "\nIADP -- Version: %s ----------------------------- ReiKe IGM TU BS\n", VERSION );

/*----------------------------------------------------------------------------- 
 * Evaluation of the environment variables.                                   *
 -----------------------------------------------------------------------------*/
 evaluate_environment ( envp, homepath, logpath, datpath, tmppath );

/*----------------------------------------------------------------------------- 
 * Create a temporary file for storing the selected intervall of input data.  *
 -----------------------------------------------------------------------------*/
 selected_tmpname = tempnam ( tmppath, NULL );
 selected_tmpfile = fopen ( selected_tmpname, "w" );
 if ( selected_tmpfile == NULL )
   {
    fprintf ( stderr, errmsg[2], PROGRAM );
    remove ( selected_tmpname );
    exit(2);
   }
 fclose ( selected_tmpfile );

/*----------------------------------------------------------------------------- 
 * Read the old settings from the '.iadprc'-file.                             *
 -----------------------------------------------------------------------------*/
 read_rcfile ( homepath, filename,
               selected_beginstr, selected_endstr,
               coorsys, rate, timeformat );

/*----------------------------------------------------------------------------- 
 * Set the coordinate system and the appropriate command, if a string for the *
 * coordinate system has been read from the '.fgmdprc'-file.                  *
 -----------------------------------------------------------------------------*/
 coorsyscmd[0] = '\0';
 if ( strlen ( coorsys ) )
    set_coorsys ( coorsys, coorsyscmd );
 
/*----------------------------------------------------------------------------- 
 * Set the output data rate and the appropriate command, if a string for the  *
 * data rate has been read from the '.fgmdprc'-file.                          *
 -----------------------------------------------------------------------------*/
 ratecmd[0] = '\0';
 if ( strlen ( rate ) )
    set_rate ( rate, ratecmd );

/*----------------------------------------------------------------------------- 
 * If a filename has been read from the '.fgmdprc'-file, read the input data  *
 * to determine start and end times and to get possible error messages.       *
 -----------------------------------------------------------------------------*/
 filenametrunk[0] = '\0';
 if ( strlen ( filename ) )
    if ( decide ( "Shall the selected input data file be used", 1 ) )
      {
       warnnr = get_data ( selected_tmpname,
                           filename, 
                           filenametrunk,
                           homepath,
                           logpath,
                           total_beginstr,
                           total_endstr,
                           selected_beginstr,
                           selected_endstr,
                           &gotdata );
       if ( warnnr )
         {
          fprintf ( stderr, warning[warnnr], PROGRAM );
          if ( warnnr != 12 )
            {
             fprintf ( stderr, "Therefore, no valid data could be read.\n" );
             fprintf ( stderr, "Please select new input data file.\n" );
            }
         }
      }

/*----------------------------------------------------------------------------- 
 * Start the main menue.                                                      *
 -----------------------------------------------------------------------------*/
 do
   {
    fprintf ( stderr, "%s\n", DOUBLELINE );
    fprintf ( stderr, "Select action (type 'h' for help)\n" );
    fprintf ( stderr, "%s ", PROMPT );
    gets ( menuecmd );
    copt = tolower ( menuecmd[0] );
    switch ( copt )
      {
       case 'a':
          warnnr = test_settings ( gotdata, 
                                   selected_beginstr, selected_endstr,
                                   coorsys, rate );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          else
             make_ascii ( selected_tmpname,
                          logpath, filenametrunk,
                          coorsyscmd, ratecmd, timeformat );
          break;

       case 'c':
          select_coorsys ( coorsys, coorsyscmd );
          break;

       case 'd':
          warnnr = test_settings ( gotdata, 
                                   selected_beginstr, selected_endstr,
                                   coorsys, rate );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          else
             extract_dds ( logpath, 
                           filename, filenametrunk, 
                           selected_beginstr, selected_endstr, timeformat );
          break;

       case 'e':
          ciao = 1;
          break;

       case 'f':
          warnnr = select_files ( selected_tmpname,
                                  filename,
                                  filenametrunk,
                                  homepath,
                                  logpath,
                                  total_beginstr,
                                  total_endstr,
                                  selected_beginstr,
                                  selected_endstr,
                                  &gotdata );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          break;

       case 'h':
          fprintf ( stderr, "%s", MENUEHELP );
          break;

       case 'i':
          warnnr = test_settings ( gotdata, 
                                   selected_beginstr, selected_endstr,
                                   coorsys, rate );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          else
             info_vec ( selected_tmpname, logpath, filenametrunk, timeformat );
          break;

       case 'l':
          warnnr = test_settings ( gotdata, 
                                   selected_beginstr, selected_endstr,
                                   coorsys, rate );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          else
             long_bin ( selected_tmpname,
                        datpath, filenametrunk, 
                        coorsyscmd, ratecmd );
          break;

       case 'm':
          select_timeformat ( timeformat );
          break;

       case 'n':
          warnnr = select_next_intervall ( selected_tmpname,
                                           filename,
                                           total_endstr,
                                           selected_beginstr,
                                           selected_endstr,
                                           &gotdata );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          break;

       case 'r':
          select_rate ( rate, ratecmd );
          break;

       case 's':
          warnnr = test_settings ( gotdata, 
                                   selected_beginstr, selected_endstr,
                                   coorsys, rate );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          else
             short_bin ( selected_tmpname,
                         datpath, filenametrunk, 
                         coorsyscmd, ratecmd );
          break;

       case 't':
          warnnr = select_intervall ( selected_tmpname,
                                      filename,
                                      total_beginstr,
                                      total_endstr,
                                      selected_beginstr,
                                      selected_endstr,
                                      &gotdata );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          break;

       case 'v':
          warnnr = select_previous_intervall ( selected_tmpname,
                                               filename,
                                               total_beginstr,
                                               selected_beginstr, 
                                               selected_endstr,
                                               &gotdata );
          if ( warnnr )
             fprintf ( stderr, warning[warnnr], PROGRAM );
          break;

       default:
          fprintf ( stderr, "Invalid input. Use 'h' for help.\n" );
          break;
      }
   }
 while ( !ciao );

/*----------------------------------------------------------------------------- 
 * Write the actual settings to the '.fgmdprc'-file.                          *
 -----------------------------------------------------------------------------*/
 write_rcfile ( homepath, filename,
                selected_beginstr, selected_endstr, 
                coorsys, rate, timeformat );

 remove ( selected_tmpname );
 exit(0);
}
