/*
 * Copyright (C) 1995/96 IGM TU Braunschweig
 */

#define PROGRAM "put{satt|stof|ltof}"
#define VERSION "5.0 (1996-12-18)"
#define PURPOSE "put CLUSTER attitude or orbit files (with DDS headers) into a merged file (without DDS headers)"
#define USAGE "USAGE:

 putsatt <infile>... [-o [<outfile>]]
 putstof <infile>... [-o [<outfile>]]
 putltof <infile>... [-o [<outfile>]]

 The program `putsatt' reads CLUSTER spacecraft attitude (SATT) files like those
 distributed by the CLUSTER DDS or on CD-ROMs, i.e. with a DDS packet header
 attached to each record entry.  It creates or updates a merged SATT file in
 which these DDS headers are removed and which thus can be used e.g. with the 
 FGM data processing software.

 The program `putstof' does the same for CLUSTER short term orbit files;
 the program `putltof' does the same for CLUSTER long term orbit files.

 Each input file must be time-ordered, and all input files must belong to the
 same spacecraft.

 For more detailed information, see the appropriate man page."

#define OPTIONS "OPTIONS:

 -o   use <outfile> as output file.  If -o is used but <outfile> is omitted,
      or if <outfile> is `-', output is written to stdout.

      Default is to create or update the file 
        `$SATTPATH/satt.cl#'   for `putsatt',
        `$ORBITPATH/stof.cl#'  for `putstof',
        `$ORBITPATH/ltof.cl#'  for `putltof',
      where `#' is replaced by the satellite id number (1,2,3,4).  To specify
      the name of the directory where these default files are installed, the
      appropriate environment variable (SATTPATH or ORBITPATH) should be used.

 -V   print version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Stephan Buchert       (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke      (joerg@igpp.ucla.edu)"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static char *errmsg[] =
  {
/*  0 */  "Use `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Cannot allocate enough memory for input files.",
/*  3 */  "ERROR in %s: No input file specified.\n",
/*  4 */  "ERROR in %s: Could not open file %s.\n",
/*  5 */  "ERROR in %s: Could not evaluate command name (length: %d).\n",
/*  6 */  "ERROR in %s: Could not open temporary file %s.\n",
/*  7 */  "ERROR in %s: Could not read s/c id from temporary file %s.\n",
/*  8 */  "ERROR in %s: Could not update file existing file %s.\n",
/*  9 */  "ERROR in %s: Could not open pipe for reading stdout of `%s'.\n",
/* 10 */  "ERROR in %s while executing `%s'.\n",
  };

static char *warning[] =
  {
/*  0 */  "%s WARNING: Environment variable %s not found.\n",
  };

int main ( int argc, char *argv[], char *envp[] )
  {
   FILE          *infile;
   int           i, n, sc, *infile_i=0, infile_n, type;
   char          copt, outname[128], cmd[256], string[130], ddshrm[128],
                 mrgaux[128], tmpname_in[L_tmpnam], tmpname_out[L_tmpnam];

   /* set output file name to empty */
   outname[0] = '\0';

   /* allocate memory for the array infile_i */
   if (infile_i) 
      free ( infile_i );
   if ( (infile_i = (int *) calloc(argc,sizeof(int))) == NULL )
     {
      fprintf ( stderr, errmsg[2], PROGRAM );
      exit(2);
     }

   /* initialize command strings */
   strcpy ( ddshrm, " | " );
   strcpy ( mrgaux, " | " );
   i = strlen ( argv[0] );
   if ( i > 7 )
     {
      strncat ( ddshrm, argv[0], i-7 );
      strncat ( mrgaux, argv[0], i-7 );
     }
   strcat ( ddshrm, "ddshrm -i " );

   if (!strcmp(argv[0]+i-7,"putstof"))
     {
      /* short term orbit files */
      strcat( ddshrm, "3" );
      type = 3;
      strcat ( mrgaux, "mrgorb " );
     }
   else if (!strcmp(argv[0]+i-7,"putltof"))
     {
      /* long term orbit files */
      strcat ( ddshrm, "1" );
      type = 1;
      strcat ( mrgaux, "mrgorb " );
     }
   else if (!strcmp(argv[0]+i-7,"putsatt"))
     {
      /* spacecraft attitude files */
      strcat ( ddshrm, "5" );
      type = 5;
      strcat ( mrgaux, "mrgatt " );
     }
   else
     {
      fprintf ( stderr, errmsg[5], PROGRAM, i );
      exit(5);
     }

   /* get command line parameters */
   i = 1;
   infile_n = 0;
   while ( i < argc )
     {
      copt = *(argv[i]);
      if (copt == '-')
        {
         copt = *(argv[i]+1);
         switch ( copt )
           {
            case 'h':
               fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                         PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
               exit(0);

            case 'o':
               if ( *(argv[i]+2) == '\0' && i+1 < argc )
                 {
                  i++;
                  strcpy ( outname, argv[i] );
                 }
               else
                  strcpy ( outname, argv[i]+2 );
               if ( strlen(outname) == 0 )
                  strcpy ( outname, "-" );
               break;

            case 'V': 
               fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
               exit(0);

            default:
               fprintf ( stderr, errmsg[1], argv[0], argv[i] );
               fprintf ( stderr, errmsg[0], argv[0] );
               exit(1);

           }
        }
      else
        {
         *(infile_i + infile_n) = i;
         infile_n++;
        }
      i++;
     }

   if ( infile_n < 1 )
     {
      fprintf ( stderr, errmsg[3], PROGRAM );
      exit(3);
     } 

   tmpname_in[0] = '\0';

   for ( n = 0; n < infile_n; n++)
     {
      /* for each file: remove header, merge with previous file */
      infile = fopen ( argv[*(infile_i + n)], "r" );
      if (infile == NULL )
        {
         fprintf ( stderr, errmsg[4], PROGRAM, argv[*(infile_i + n)] );
         if ( strlen(tmpname_in) > 0 )
            remove ( tmpname_in );
         exit(4);
        }
      fclose ( infile );
      strcpy ( cmd, "cat " );
      strcat ( cmd, argv[*(infile_i + n)] );
      strcat ( cmd, ddshrm );
      if ( strlen(tmpname_in) > 0 )
        {
         strcat ( cmd, mrgaux );
         strcat ( cmd, tmpname_in );
        }
      strcat ( cmd, " > " );
      tmpnam ( tmpname_out );
      strcat ( cmd, tmpname_out );
      if ( system(cmd) != 0 )
        {
#ifndef SECURE
         fprintf ( stderr, errmsg[10], PROGRAM, cmd );
#endif
         remove ( tmpname_out );
         if ( strlen(tmpname_in) > 0 )
            remove ( tmpname_in );
         exit(10);
        }
      if ( strlen(tmpname_in) > 0 )
        remove ( tmpname_in );
      strcpy ( tmpname_in, tmpname_out );
     }

   if ( (strlen(outname) > 1) || (outname[0] != '-'))
     {
      if ( strlen(outname) == 0 )
        {
	 /* use default output file name */
         for ( i= 0; envp[i] != NULL; i++ )
           {
            if ((type == 1) || (type == 3))
              {
               if ( !strncmp(envp[i],"ORBITPATH",9) )
                 {
                  strcpy ( outname, getenv ( "ORBITPATH" ) );
                  strcat ( outname, "/" );
                  break;
                 }
              }
            else if ( type == 5 )
              {
               if ( !strncmp(envp[i],"SATTPATH",8) )
                 {
                  strcpy ( outname, getenv ( "SATTPATH" ) );
                  strcat ( outname, "/" );
                  break;
                 }
              }
           }
         infile = fopen ( tmpname_in, "r" );
         if ( infile == NULL )
           {
            fprintf ( stderr, errmsg[6], PROGRAM, tmpname_in );
            remove ( tmpname_in );
            exit(6);
           }
         i = fscanf ( infile, "%2d", &sc );
         fclose ( infile );
         if ( i != 1 )
           {
            fprintf ( stderr, errmsg[7], PROGRAM, tmpname_in );
            remove ( tmpname_in );
            exit(7);
           }
         i = strlen ( outname );
         if ( type == 3 )
	   {
	    if ( i == 0 )
	       fprintf ( stderr, warning[0], PROGRAM, "ORBITPATH" );
            sprintf ( &outname[i], "stof.cl%d", sc );
	   }
         else if ( type == 1 )
	   {
	    if ( i == 0 )
	       fprintf ( stderr, warning[0], PROGRAM, "ORBITPATH" );
            sprintf ( &outname[i], "ltof.cl%d", sc );
	   }
         else if ( type == 5 )
	   {
	    if ( i == 0 )
	       fprintf ( stderr, warning[0], PROGRAM, "SATTPATH" );
            sprintf ( &outname[i], "satt.cl%d", sc );
	   }
        }
      infile = fopen ( outname, "r" );
      if ( infile != NULL )
        {
         fclose ( infile );
	 /* merge output with old file */
	 strcpy ( cmd, "cat " );
         strcat ( cmd, outname );
         strcat ( cmd, mrgaux );
         strcat ( cmd, tmpname_in );
         strcat ( cmd, " > " );
         tmpnam ( tmpname_out );
         strcat ( cmd, tmpname_out );
         if ( system(cmd) != 0 )
           {
#ifndef SECURE
            fprintf ( stderr, errmsg[10], PROGRAM, cmd );
#endif
            remove ( tmpname_out );
            remove ( tmpname_in );
            exit(10);
           }
         remove ( tmpname_in );
	 /* check whether output file has changed */
	 strcpy ( cmd, "diff " );
	 strcat ( cmd, outname );
	 strcat ( cmd, " " );
	 strcat ( cmd, tmpname_out );
	 infile = popen ( cmd, "r" );
	 if ( infile != NULL )
	   {
	    if ( NULL == fgets(string,130,infile) )
	      {
	       /* no changes in output file */
	       remove ( tmpname_out );
	       exit(0);
	      }
	    pclose ( infile );
	   }
         if ( remove(outname) != 0 )
           {
            fprintf ( stderr, errmsg[8], PROGRAM, outname );
            remove ( tmpname_in );
            exit(8);
           }
         strcpy ( tmpname_in, tmpname_out );
        }
      strcpy ( cmd, "cat " );
      strcat ( cmd, tmpname_in );
      strcat ( cmd, " > " );
      strcat ( cmd, outname );
      if ( system(cmd) != 0 )
        {
#ifndef SECURE
         fprintf ( stderr, errmsg[10], PROGRAM, cmd );
#endif
         remove ( tmpname_in );
         exit(10);
        }
     } 
   else
     {
      strcpy ( cmd, "cat " );
      strcat ( cmd, tmpname_in );
      infile = popen ( cmd, "r" );
      if ( infile == NULL ) 
        {
         fprintf ( stderr, errmsg[9], PROGRAM, cmd );
         remove ( tmpname_in );
         exit(9);
        }
      while ( NULL != fgets(string,130,infile) )
         fprintf ( stdout, "%s\n", string );
      pclose ( infile );
     }
   
   remove ( tmpname_in );
   exit(0);
  }
