
      SUBROUTINE DJ2000(DAY,I,J,K,JHR,MI,SEC)
CP  COMPUTES CALENDER DATE FROM MODIFIED JULIAN DAY 2000
C   VALID FOR DATES BETWEEN 1950/JAN/1 AND 2099/DEC/31.
C   MJD(2000) = MJD(1950) - 18262.0 IS = 0 ON 2000/01/01 AT 00:00:00.
C
CI  (REAL*8) DAY = MOD. JULIAN DAY, REFERRED TO 2000 (MAY BE NEGATIVE).
CO  (INTEGERS): I=YEAR, J=MONTH, K=DAY, JHR=HOUR, MI=MINUTE
CO  (REAL*8): SEC=SECOND.
C
      IMPLICIT REAL*8(A-H,O-Z)
C  MAKE SURE TO ROUND-OFF ONLY DOWN, ALSO FOR NEGATIVE MJD:
      JDAY = DAY + 18262.D0
      L = (4000*(JDAY + 18204))/1461001
      N = JDAY - (1461*L)/4 + 18234
      M = (80*N)/2447
      K = N - (2447*M)/80
      JJ = M/11
      J = M + 2 - 12*JJ
      I = 1900 + L + JJ
      SEC = (DAY - DFLOAT(JDAY-18262))*24.D0
      JHR = SEC
      SEC = (SEC - DFLOAT(JHR))*6.D1
      MI = SEC
      SEC = (SEC - DFLOAT(MI))*6.D1
      RETURN
      END
C
      SUBROUTINE GEI_GEO ( MJD, A )
C
C   Given the time as a Modified Julian Date, the matrix for transformation
C   from GEI to GEO coordinates is computed.
C
C   Input parameter:
C     MJD     (REAL*8)  Modified Julian Days, referred to 2000-01-01, 0 UT
C
C   Output parameter:
C     A(3,3)  (REAL*8)  transformation matrix from GEI to GEO
C
C   References:
C     Landolt-Boernstein Vol. V/2a, 1984, 
C       chap. 1.2.2.1: formula for GMST
C       chap. 1.2.4:   rotation rate
C
C     M. A. Hapgood: Space Physics Coordinate Transformations: A User
C       Guide.  Planet. Space Sci. 40, 711-717, 1992.
C
C   Joerg Warnecke  IGM TU BS  1994-09-08
C
      IMPLICIT NONE
C
      REAL*8      PI, RAD, G0, G1, G2, G3, RR
      PARAMETER  (PI=3.14159265358979323846D0,RAD=PI/180.D0)
      PARAMETER  (G0=+1.004606183750000D+02)
      PARAMETER  (G1=+3.600077005360833D+04)
      PARAMETER  (G2=+3.879333333333333D-04)
      PARAMETER  (G3=-2.583333333333333D-08)
      PARAMETER  (RR=+1.5041067D+01)
C
      REAL*8      MJD, A(3,3), JD, JC, GMST, UT
      INTEGER*4   ID
C
C     JD: Julian Date, referred to 2000-01-01, 12 UT
C     ID: Number of full days from 2000-01-01, 0 UT until 0 UT of MJD
C     JC: Julian Centuries from 2000-01-01, 12 UT until 0 UT of MJD
C
      JD = MJD - 0.5D0
      ID = NINT(JD)
      JC = (DBLE(ID)-0.5D0) / 36525.D0
C
C     UT: Universal Time
C
      UT = 24.D0*(MJD-DBLE(ID))
C
C     GMST: mean sidereal time [rad] of Greenwich
C
      GMST = RAD * MOD ( ((G3*JC+G2)*JC+G1)*JC+G0 + RR*UT, 360.D0 )
C
C     rotation of angle GMST about Z-GEI
      A(1,1) =  COS(GMST)
      A(1,2) =  SIN(GMST)
      A(1,3) =  0.D0
      A(2,1) = -A(1,2)
      A(2,2) =  A(1,1)
      A(2,3) =  0.D0
      A(3,1) =  0.D0
      A(3,2) =  0.D0
      A(3,3) =  1.D0
C
      RETURN
      END
C*******************************************************************C
C
      SUBROUTINE GEI_GSE ( MJD, A )
C
C   Given the time as a Modified Julian Date, the matrix for transformation
C   from GEI to GSE coordinates is computed.
C
C   Input parameter:
C     MJD     (REAL*8)  Modified Julian Days, referred to 2000-01-01, 0 UT
C
C   Output parameter:
C     A(3,3)  (REAL*8)  transformation matrix from GEI to GSE
C
C   References:
C     Landolt-Boernstein Vol. V/2a, 1984,
C       chap. 1.2.4: formula for obliquity of the ecliptic
C
C     M. A. Hapgood: Space Physics Coordinate Transformations: A User
C       Guide.  Planet. Space Sci. 40, 711-717, 1992.
C
C   Joerg Warnecke  IGM TU BS  1994-09-08  (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      REAL*8     PI, RAD
      PARAMETER (PI=3.14159265358979323846D0,RAD=PI/180.D0)
C
      REAL*8     MJD, A(3,3), EPS, CE, SE, EL, CL, SL, MA, ML,
     &           JD, JC, UT
      INTEGER*4  ID
C
C   JD: Julian Date, referred to 2000-01-01, 12 UT
C
      JD = MJD - 0.5D0
C
C   JC: Julian Centuries from 2000-01-01, 12 UT
C
      JC = JD / 36525.D0
C
C   EPS: obliquity of the ecliptic
C
      EPS = (23.43929111111111D0 - 0.01300416666666667D0*JC) * RAD
      SE = SIN(EPS)
      CE = SQRT(1.D0-SE*SE)
C
C   ID: Number of full days from 2000-01-01, 0 UT to the previous midnight
C   JC: Julian Centuries from 2000-01-01, 12 UT to the previous midnight
C   UT: Universal Time
C
      ID = NINT(JD)
      JC = (DBLE(ID)-0.5D0) / 36525.D0
      UT = 24.D0*(MJD-DBLE(ID))
C
C   MA: mean anomaly of the sun
C   ML: mean longitude of the sun
C   EL: ecliptic longitude of the sun
C
      MA = ( 357.528D0+35999.050D0*JC+0.04107D0*UT ) * RAD
      ML = ( 280.460D0+36000.772D0*JC+0.04107D0*UT ) * RAD
      EL = ML + ( (1.915D0-0.0048D0*JC)*SIN(MA) + 0.020D0*SIN(MA+MA) )
     &          * RAD
      SL = SIN(EL)
      CL = COS(EL)
C
C   rotation of angle EPS about X-GEI and of angle EL about Z-GSE
C
      A(1,1) =  CL
      A(1,2) =  SL*CE
      A(1,3) =  SL*SE
      A(2,1) = -SL
      A(2,2) =  CL*CE
      A(2,3) =  CL*SE
      A(3,1) =  0.D0
      A(3,2) =    -SE
      A(3,3) =     CE
C
      RETURN
      END
C*********************************************************************C
C
      SUBROUTINE GSE_GSM ( DX, DY, DZ, A )
C
C   Given the unit vector to the northern geomagnetic pole in GSE coordinates,
C   the matrix for transformation from GSE to GSM coordinates is computed.
C
C   Input parameters:
C     DX, DY, DZ  (REAL*8)  unit vector to northern geomagnetic pole
C                           in GSE system
C 
C   Output parameter:
C     A(3,3)      (REAL*8)  transformation matrix from GSE to GSM
C
C   Reference:
C     M. A. Hapgood: Space Physics Coordinate Transformations: A User
C       Guide.  Planet. Space Sci. 40, 711-717, 1992.
C 
C   Author: Joerg Warnecke  IGM TU BS  1994-09-08
C
      IMPLICIT NONE
C
      REAL*8      DX, DY, DZ, A(3,3), PSI
C
      PSI = ATAN2(DY,DZ)
C     rotation of angle -PSI about X-GSE
      A(1,1) =  1.D0
      A(1,2) =  0.D0
      A(1,3) =  0.D0
      A(2,1) =  0.D0
      A(2,2) =  COS(PSI)
      A(2,3) = -SIN(PSI)
      A(3,1) =  0.D0
      A(3,2) = -A(2,3)
      A(3,3) =  A(2,2)
C
      RETURN
      END
C*******************************************************************C
C
      SUBROUTINE GSM_SM ( DX, DY, DZ, A )
C
C   Given the unit vector to the northern geomagnetic pole in GSE coordinates,
C   the matrix for transformation from GSM to SM coordinates is computed.
C
C   Input parameters:
C     DX, DY, DZ  (REAL*8)  unit vector to northern geomagnetic pole
C                           in GSE (!) system
C
C   Output parameter:
C     A(3,3)      (REAL*8)  transformation matrix from GSM to SM
C
C   Reference:
C     M. A. Hapgood: Space Physics Coordinate Transformations: A User
C       Guide.  Planet. Space Sci. 40, 711-717, 1992.
C 
C   Author: Joerg Warnecke  IGM TU BS  1994-09-08
C
      IMPLICIT NONE
C
      REAL*8      DX, DY, DZ, A(3,3), PSI
C
      PSI = ATAN2(DX,SQRT(DY*DY+DZ*DZ))
C     rotation of angle PSI about Y-GSM
      A(1,1) =  COS(PSI)
      A(1,2) =  0.D0
      A(1,3) = -SIN(PSI)
      A(2,1) =  0.D0
      A(2,2) =  1.D0
      A(2,3) =  0.D0
      A(3,1) = -A(1,3)
      A(3,2) =  0.D0
      A(3,3) =  A(1,1)
C
      RETURN
      END
C*******************************************************************C
      INTEGER*4 FUNCTION IFLU_ATT ( ISAT, NAME )
C
C  This subroutine opens the appropriate spacecraft attitude (SATT) file
C  for a CLUSTER satellite.  On output, the returned value IFLU_ATT
C  represents the following conditions:
C
C   IFLU_ATT > 0:  The file has been opened sucessfully, 
C                  its logical unit number is IFLU_ATT ( = 10 + ISAT )
C   IFLU_ATT = 0:  No file has been opened,
C                  because ISAT is an illegal satellite number
C   IFLU_ATT < 0:  The file could not be opened, 
C                  the I/O error number was IOSTAT = ABS(IFLU_ATT)
C
C  Input parameters:
C    ISAT     (INTEGER*4)   ID number of the CLUSTER satellite (1,2,3,4)
C    NAME (CHARACTER*(*))   file name of the SATT file to be opened
C
C  Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
C
      IMPLICIT NONE
C
      INTEGER*4     ISAT, L, N
      LOGICAL*4     OPENED
      CHARACTER*(*) NAME
C
      IF ((ISAT.GT.0).AND.(ISAT.LT.5)) THEN
C
        L = 10 + ISAT
C
C       if an attitude file for this satellite has been opened before, close it
        INQUIRE (L,OPENED=OPENED,IOSTAT=N)
        IF ((N.EQ.0).AND.(OPENED)) CLOSE (L,STATUS='KEEP',IOSTAT=N)
C
        OPEN (L,FILE=NAME,STATUS='OLD',FORM='FORMATTED',
     &        ACCESS='SEQUENTIAL',IOSTAT=N)
C
        IF (N.NE.0) L = -ABS(N)
C
      ELSE
C
        L = 0
C
      END IF
C
      IFLU_ATT = L
C
      RETURN
      END
C******************************************************************C

          

      INTEGER*4 FUNCTION IFLU_ORB ( ISAT, NAME )
C
C  This subroutine opens a CLUSTER (long-term or short-term) orbit file.
C  On output, the returned value IFLU_ORB represents the following conditions:
C
C   IFLU_ORB > 0:  The file has been opened sucessfully, 
C                  its logical unit number is IFLU_ORB ( = 20 + ISAT )
C   IFLU_ORB = 0:  No file has been opened,
C                  because ISAT is an illegal satellite number
C   IFLU_ORB < 0:  The file could not be opened, 
C                  the I/O error number was IOSTAT = ABS(IFLU_ORB)
C
C  Input parameters:
C    ISAT     (INTEGER*4)   ID number of the CLUSTER satellite (1,2,3,4)
C    NAME (CHARACTER*(*))   file name of the orbit file to be opened
C
C  Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      INTEGER*4     ISAT, L, N
      LOGICAL*4     OPENED
      CHARACTER*(*) NAME
C
      IF ((ISAT.GT.0).AND.(ISAT.LT.5)) THEN
C
        L = 20 + ISAT
C
C       if an orbit file for this satellite has been opened before, close it
        INQUIRE (L,OPENED=OPENED,IOSTAT=N)
        IF ((N.EQ.0).AND.(OPENED)) CLOSE (L,STATUS='KEEP',IOSTAT=N)
C
        OPEN (L,FILE=NAME,STATUS='OLD',FORM='FORMATTED',
     &        ACCESS='SEQUENTIAL',IOSTAT=N)
C
        IF (N.NE.0) L = -ABS(N)
C
      ELSE
C
        L = 0
C
      END IF
C
      IFLU_ORB = L
C
      RETURN
      END
C******************************************************************C

          

      SUBROUTINE JD2000(DAY,JEAR,MONTH,KDAY,JHR,MI,SEC)
CP GIVES THE NEW MOD. JULIAN DAY (MJD=0.0 ON 2000/JAN/1 AT 0:00:00)
CP FOR INPUT CALENDAR DATES BETWEEN 1950/JAN/1 AND 2099/DEC/31.
C
C   MJD(2000) = MJD(1950) - 18262.0
C
CI  (INT*4) JEAR = YEAR WITH 2 OR 4 DIGITS; 2 DIGITS => 1950 TO 2049
CI  (INT*4) MONTH = MONTH
CI  (INT*4) KDAY = DAY
CI  (INT*4) JHR = HOUR
CI  (INT*4) MI = MINUTE
CI  (REAL*8) SEC = SECOND.
CO  (REAL*8) DAY = MOD. JUL. DAY, REFERRED TO 2000.
C
      IMPLICIT REAL*8(A-H,O-Z)
      JJ = (14 - MONTH)/12
      L = JEAR - JJ - 1900*(JEAR/1900) + 100*(2000/(JEAR+1951))
      DAY = KDAY-36496+(1461*L)/4+(367*(MONTH-2+JJ*12))/12
      DAY = DAY + (DFLOAT((JHR*60 + MI)*60) + SEC)/864.D2
      RETURN
      END
C
      REAL*8 FUNCTION MJD2000 ( ISEC, NSEC )
C
C   This subroutine returns the Modified Julian Day
C   (referred to 2000-01-01, 0 UT)
C   for a given UNIX system time ISEC, NSEC (INTEGER*4)
C   (seconds since 1970-01-01, 0 UT, and nanoseconds within the second)
C
C   Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      REAL*8       MJD0
C                  Modified Julian Date (2000) of 1970-01-01, 0 UT
      PARAMETER   (MJD0=2440587.5D0-2451544.5D0)
C
      INTEGER*4    ISEC, NSEC
C
      MJD2000 = MJD0 + (DBLE(ISEC)+DBLE(NSEC)*1.D-9)/86400.D0
C
      RETURN
      END
C********************************************************************C
C
      SUBROUTINE MJD_CDS ( MJD, DY, MS, MY, FLAG )
C
C   This subroutine converts between the following time formats:
C
C   (1)  MJD   (REAL*8)
C        Modified Julian Day, referred to 2000-01-01, 0 UT
C   (2)  DY (INTEGER*2),  MS (INTEGER*4),  MY (INTEGER*2)
C        CCSDS Day Segmented time code: number of days since 1958-01-01,
C        milliseconds within the day, microseconds within the millisecond
C
C   The value of FLAG (INTEGER*4) controls the direction of conversion:
C        FLAG.GT.0:  input (1), output (2)
C        FLAG.LE.0:  input (2), output (1)
C
C   Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      REAL*8       MJD0
      PARAMETER   (MJD0=-15340.D0)
C                  Modified Julian Date (2000) of 1958-01-01, 0 UT
C
      REAL*8       MJD, DDY, DMS, DMY
      INTEGER*4    MS, FLAG, IDY, IMS, IMY
      INTEGER*2    DY, MY
C
      IF (FLAG.GT.0) THEN
        DDY = MJD - MJD0
        IDY = INT(DDY)
 10     CONTINUE
        DMS = (DDY-DBLE(IDY))*8.64D7
        IMS = INT(DMS)
 20     CONTINUE
        IF (IMS.GE.86400000) THEN
          IDY = IDY + 1
          GO TO 10
        ELSE IF (IMS.LT.0) THEN
          IDY = IDY - 1
          GO TO 10
        END IF
        DMY = (DMS-DBLE(IMS))*1.D3
        IMY = NINT(DMY)
        IF (IMY.GE.1000) THEN
          IMS = IMS + 1
          GO TO 20
        ELSE IF (IMY.LT.0) THEN
          IMS = IMS - 1
          GO TO 20
        END IF
        DY = IDY
        MS = IMS
        MY = IMY
      ELSE
        IMY = MY
        IMS = MS
        IDY = DY
        MJD = (DBLE(IMY)*1.D-3 + DBLE(IMS))/8.64D7 + DBLE(IDY) + MJD0
      END IF
C
      RETURN
      END
C*******************************************************************C
C
      SUBROUTINE MJD_STR ( MJD, STR, FLAG )
C
C   This subroutine converts between the following time formats:
C
C   (1)  MJD   (REAL*8)
C        Modified Julian Day, referred to 2000-01-01, 0 UT
C   (2)  STR   (CHARACTER*20)
C        CCSDS ASCII time code A string (YYYY-MM-DDThh:mm:ssZ)
C
C   The value of FLAG (INTEGER*4) controls the direction of conversion:
C        FLAG.GT.0:  input (1), output (2)
C        FLAG.LE.0:  input (2), output (1)
C
C   Subroutines:
C     JD2000  for calculating the Modified Julian Day of a date
C     DJ2000  for calculating the date of a Modified Julian Day
C
C   Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      REAL*8        MJD, SEC
      CHARACTER*20  STR
      INTEGER*4     FLAG, IYR, IMO, IDY, IHR, IMI, ISE
C
      IF (FLAG.GT.0) THEN
        CALL DJ2000 ( MJD, IYR, IMO, IDY, IHR, IMI, SEC )
        ISE = NINT(SEC)
        IF (ISE.EQ.60) THEN
          ISE = 0
          IMI = IMI + 1
          IF (IMI.EQ.60) THEN
            IMI = 0
            IHR = IHR + 1
          END IF
        END IF
        WRITE (STR,1000) IYR, IMO, IDY, IHR, IMI, ISE
      ELSE
        READ (STR,1001) IYR, IMO, IDY, IHR, IMI, ISE
        CALL JD2000 ( MJD, IYR, IMO, IDY, IHR, IMI, DBLE(ISE) )
      END IF
C
      RETURN
C
 1000 FORMAT ( I4, 2('-',I2.2), 'T', I2.2, 2(':',I2.2), 'Z' )
 1001 FORMAT ( I4, 1X, 5(I2,1X) )
      END
C********************************************************************C
      SUBROUTINE ORBIT(DAY,KODE,LFILE,IERROR,NSAT,X,REVNUM)
C
C   Changes for the FGM software (by Joerg Warnecke, 1996-04-15):
C   ----------------------------
C    The original subroutine returned error condition 3 every time
C    it met a time gap in the orbit file, even if the requested time
C    was beyond that gap.  This has been changed: it will now return
C    error condition 3 only if the requested time falls into the gap.
C
CP  ORBIT: RETRIEVAL ROUTINE FOR COMPRESSED CLUSTER ORBIT
C
C   INPUT:
CI  DAY (R*8) = MODIFIED JULIAN DAY, FROM 2000, FOR THE STATE VECTOR
CI  KODE (I*4) = NUMBER OF COMPONENTS OF STATE VECTOR = DIM. OF ARRAY
C              X(); = 3 FOR S/C POSITION, = 6 FOR POSITION & VELOCITY
CI  LFILE (I*4) = LOGICAL NUMBER OF INPUT DATA FILE
C   OUTPUT:
CO  IERROR (I*4) = RETURN CODE: 0=NO ERROR, 1='DAY' TOO EARLY, 2=TOO
C            LATE, 3=TIME GAP IN DATA, 4=WRONG VALUE OF 'KODE',
C            5=FILE CONTENT INCONSISTENT, 6=READ ERROR FROM DATA FILE
CO  NSAT (I*4) = SATELLITE NUMBER; 1, 2, 3, 4
CO  X(KODE) (R*8) = SPACECRAFT POSITION, KM (AND VELOCITY, KM/S)
CO  REVNUM (R*8) = REVOLUTION NUMBER
C
CF  READS A SEQUENTIAL FORMATTED FILE WITH LOGICAL NUMBER 'LFILE'
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION Y(6),COEFF(10,6),X(KODE)
C   INITIALISE FILE NUMBER TO FORCE FILE READING AT FIRST CALL
      DATA MFILE/-9999/
      SAVE
C
C  INITIALISE ERROR CODES
      IERROR = 0
      IF(KODE .LE. 0) GOTO 504
      IF(KODE .GT. 6) GOTO 504
C
C  ALWAYS REWIND IF A NEW FILE NUMBER IS USED
      IF(LFILE .NE. MFILE) GOTO 10
C
C  CHECK IF 'DAY' IS INSIDE LAST READ RECORD BLOCK
      IF(DAY .GT. DAYEND + 1.D-4) GOTO 20
      IF(DAY .GE. DAYBEG - 1.D-4) GOTO 70
C
C  INITIALISE THE READING FROM THE FILE
10    DAYFIR = 99.D9
      DAYEND = 99.D9
      DAYLAS = -99.D9
      MFILE = LFILE
      REWIND LFILE
C
20    CONTINUE
C  REWIND WHEN 'DAY' IS EARLIER THAN END OF PREVIOUS BLOCK
      IF(DAY .LT. DAYLAS) GOTO 10
C  READ 1ST RECORD IN A BLOCK
      READ(LFILE,41,ERR=506,END=509) NSAT
41    FORMAT(I3)
C  IF: NSTA = A SATELLITE NUMBER; THEN THIS IS 1ST RECORD IN A BLOCK
      IF(NSAT .LE. 0) GOTO 20
      IF(NSAT .GT. 4) GOTO 20
C
C  READ 2ND RECORD IN THE BLOCK
CF  NREC = RECORD IDENTIFICATION, SHALL BE = 200 + NSAT
CF  DAYBEG = BEGIN TIME OF THE RECORD (MJD)
CF  DAYEND = END TIME OF THE RECORD (MJD)
CF  EPOCH = EPOCH OF REFERENCE STATE VECTOR (MJD)
CF  REVEPO = REVOLUTION NUMBER AT EPOCH
CF  SMAXIS = SEMIMAJOR AXIS FOR THE KEPLER ORBIT
CF  OMOTIN = INVERSE MEAN MOTION FOR THE KEPLER ORBIT
      READ(LFILE,42,ERR=506,END=509)
     &NREC,DAYBEG,DAYEND,EPOCH,REVEPO,SMAXIS,OMOTIN
42    FORMAT(I3,2F12.6,F15.9,F11.3,2F13.5)
C  CHECK CONSISTENCY OF FILE
      IF(NREC .NE. 200 + NSAT) GOTO 505
      IF(DAYBEG .GT. DAYEND) GOTO 505
C
C  DAYFIR = START TIME OF 1ST RECORD ON 1ST BLOCK ON THE FILE - MARGIN
      DAYFIR = DMIN1(DAYFIR,DAYBEG - 1.D-4)
C  DAYLAS = END TIME OF LAST READ RECORD BLOCK + MARGIN
      DAYLAS = DAYEND + 1.D-4
C  ERROR RETURN IF 'DAY' IS BEFORE START OF FILE (WITH MARGIN)
      IF(DAY .LT. DAYFIR) GOTO 501
C  CONTINUE READ IF 'DAY' IS AFTER END OF THIS RECORD BLOCK
      IF(DAY .GT. DAYEND + 1.D-4) GOTO 20
C  ERROR RETURN IF THERE IS A GAP FROM LAST BLOCK (WITH MARGIN)
      IF(DAY .LT. DAYBEG - 1.D-4) GOTO 503
C
C  READ 3RD RECORD IN THE BLOCK
CF  NREC = RECORD IDENTIFICATION, SHALL BE = 300 + NUMBER OF POL.COEFF.
CF  Y(6) = REFERENCE STATE VECTOR FOR KEPLER ORBIT (KM, KM/S)
CF  RDIST = S/C EARTH CENTRE DISTANCE AT EPOCH
      READ(LFILE,43,ERR=506,END=505) NREC,Y,RDIST
43    FORMAT(I3,3F11.3,3F11.7,F11.3)

C  CHECK CONSISTENCY OF FILE
      IF(NREC .GT. 310) GOTO 505
      IF(NREC .LT. 300) GOTO 505
C   KOEFF = NUMBER OF POLYNOMIAL COEFFICIENTS, BETWEEN 0 AND 10
      KOEFF = NREC - 300
C
C  IF THERE ARE NO COEFFICIENTS IN THIS BLOCK
      IF(KOEFF. LE. 0) GOTO 70
      DO 60 K = 1,KOEFF
CF  NREC = RECORD IDENT. = KOEFF + 11*K
CF  COEFF(10,6) = MATRIX WITH UP TO 10 COEFFICIENTS OF THE CHEBYSHEV
CF  POLYNOMIAL FOR EACH OF THE 6 COMPONENTS OF THE STATE VECTOR
      READ(LFILE,44,ERR=506,END=505) NREC,(COEFF(K,I),I=1,6)
44    FORMAT(I3,3F11.3,3F11.7)
C
C  CHECK CONSISTENCY OF FILE
      IF(11*K + KOEFF .NE. NREC) GOTO 505
60    CONTINUE
C  END OF BLOCK READING SEQUENCE
70    CONTINUE
C
C  TIME CONVERTED TO DIFFERENCE IN MEAN ANOMALY
      DMANOM = (DAY - EPOCH)*864.D2/OMOTIN
C  ORBIT NUMBER
      REVNUM = REVEPO + DMANOM/6.2831853072D0
C
C  START MODELLING KEPLER ORBIT
      ARIN = SMAXIS/RDIST
      ARM = (RDIST - SMAXIS)/SMAXIS
      RVWAM = (Y(1)*Y(4) + Y(2)*Y(5) + Y(3)*Y(6))*OMOTIN/SMAXIS**2
C  CALC. OF ECC. ANOMALY BY NEWTON'S ITERATION
      TAM = DMANOM - RVWAM
      COMP = 1.D-7 + 1.D-10*DABS(TAM)
      B = TAM
C  ITERATIONS TO SOLVE KEPLER'S EQUATION:
      DO 130 ITER = 1,15
      GO = DCOS(B)
      G1 = DSIN(B)
      BET = TAM - ARM*G1 + RVWAM*GO
      D = (BET - B)/(1.D0 + ARM*GO + RVWAM*G1)
      B = B + D
C  THIS GIVES THE ACCURACY  1.D-14 IN B & THE G'S
      IF(DABS(D) .LE. COMP) GOTO 140
130   CONTINUE
C  NO CONVERGENCE, ERROR RETURN
      GOTO 505
140   CONTINUE
      GO = GO - D*G1
      G1 = G1 + D*GO
      G2 = 1.D0 - GO
      G3 = B - G1
      FX = 1.D0  - G2*ARIN
      GX = (DMANOM - G3)*OMOTIN
C
      K = MIN0(KODE,3)
      DO 150 J = 1,K
150   X(J) = FX*Y(J) + GX*Y(J+3)
C
      IF(KODE .LE. 3) GOTO 170
      RX = DSQRT(X(1)**2 + X(2)**2 + X(3)**2)
      FT = -G1*SMAXIS*ARIN/(OMOTIN*RX)
      GT = 1.D0 - G2*SMAXIS/RX
      DO 160 J = 4,KODE
160   X(J) = FT*Y(J-3) + GT*Y(J)
C  END OF MODELLING KEPLER ORBIT
170   CONTINUE
C
C  CHECK IF POLYNOMIAL COEFFICIENTS ARE REQUIRED (1 IS NOT WORTH WHILE)
      IF(KOEFF .LE. 1) GOTO 600
C  MID-POINT & SCALE FACTOR FOR CHEBYSHEV POLYNOMIAL
      DAYMID = 0.5D0*(DAYBEG + DAYEND)
      SCALE = 4.D0/(DAYEND - DAYBEG)
C  ADD CHEBYSHEV POLYNOMIAL TO KEPLER STATE VECTOR
      S = SCALE*(DAY - DAYMID)
      PA = 1.D0
      P = S*0.5D0
C
C  'KODE' = NUMBER OF COMPONENTS OF THE STATE VECTOR
      DO 200 J = 1,KODE
200   X(J) = X(J) + COEFF(1,J) + COEFF(2,J)*P
C
      IF(KOEFF .LE. 2) GOTO 600
      DO 210 L = 3,KOEFF
      PB = PA
      PA = P
      P = S*PA - PB
      DO 210 J = 1,KODE
210   X(J) = X(J) + COEFF(L,J)*P
230   CONTINUE
C
      GOTO 600
C  ERROR RETURNS; IERROR = 5, 6, 4, 3, 2 OR 1
509   CONTINUE
C  END-OF-FILE ONLY IF AT LEAST ONE RECORD HAS BEEN READ
      IF(DAYLAS .GT. -1.D9) GOTO 502
505   IERROR = -1
506   IERROR = IERROR + 2
504   IERROR = IERROR + 1
503   IERROR = IERROR + 1
502   IERROR = IERROR + 1
501   IERROR = IERROR + 1
C  FORCE A RE-INITIALISATION OF READ AT NEXT CALL AFTER AN ERROR
      MFILE = -9999
600   RETURN
      END
      SUBROUTINE PR2000(DAY,P)
CP  COMPUTES THE PRECESSION MATRIX P(3,3) FOR CONVERTING A VECTOR
C IN MEAN GEOCENTRIC EQUATORIAL SYSTEM OF 2000.0 TO MEAN-OF-DATE.
C REF: THE ASTRONOMICAL ALMANAC 1985 PAGE B18.
C
CINPUT:  DAY = MJD2000 = MOD. JULIAN DAY FOR THE MEAN-OF-DATE SYSTEM
C            = MJD(1950) - 18262.0
C
COUTPUT: P(3,3) = PRECESSION MATRIX FOR THE TRANSFORMATION:
C     R(MEAN-OF-DATE) = P(,)*R(2000)
C
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION P(3,3)
C
C CONVERT TO STANDARD EPOCH J2000.0 = 2000 JAN 1 AT 12:00:00
      T = DAY - 0.5D0
C
C  GZ=GREEK Z(A), ZA=Z(A), TH=THETA, ACCORDING TO THE REFERENCE.
C ORIGINAL, WITH TJC = (DAY-0.5D0)/36525.D0  IN JULIAN CENTURIES:
C     GZ = RAD*TJC*(0.6406161D0 + TJC*(839.D-7 + TJC*5.D-6))
C     ZA = GZ + RAD*TJC*TJC*(2202.D-7 + TJC*1.D-7)
C     TH = RAD*TJC*(0.5567530D0 - TJC*(1185.D-7 + TJC*116.D-7))
C
      GZ = T*(0.3061153D-6 + T*(0.10976D-14 + T*0.179D-20))
      ZA = GZ + T*T*(0.2881D-14 + T*0.358D-22)
      TH = T*(0.2660417D-6 - T*(0.1550D-14 + T*0.41549D-20))
C
      CGZ=DCOS(GZ)
      SGZ=DSIN(GZ)
      CZA=DCOS(ZA)
      SZA=DSIN(ZA)
      CTH=DCOS(TH)
      STH=DSIN(TH)
      P(1,1) = CGZ*CZA*CTH - SGZ*SZA
      P(1,2) = -SGZ*CZA*CTH - CGZ*SZA
      P(1,3) = -CZA*STH
      P(2,1) = CGZ*SZA*CTH + SGZ*CZA
      P(2,2) = -SGZ*SZA*CTH + CGZ*CZA
      P(2,3) = -SZA*STH
      P(3,1) = CGZ*STH
      P(3,2) = -SGZ*STH
      P(3,3) = CTH
      RETURN
      END
C
      SUBROUTINE SATT ( T_MJD, IFLU, IERR, NSAT, OMEGA, PHI_SRP, A_BBSR,
     &                  COM_BB, S_2000, VST_MJD, VET_MJD )
C
C   Returns information from a CLUSTER SATT file.  
C   Times are given as Modified Julian Days, referred to 2000-01-01, 0 UT.
C
C   Input parameters:
C     T_MJD       (REAL*8) time [UTC] for which information is requested
C     IFLU     (INTEGER*4) FORTRAN logical unit of the SATT file
C
C   Output parameters:
C     IERR     (INTEGER*4) return code:
C                          0 = no error
C                          1 = MJD too early, 2 = MJD too late,
C                          3 = time gap in data,
C                          4 = read error from SATT file
C     NSAT     (INTEGER*4) satellite number (1,2,3,4)
C     OMEGA       (REAL*8) mean spin frequency [rad/s]
C     PHI_SRP     (REAL*8) spin phase of sun reference pulse [rad]
C     A_BBSR(3,3) (REAL*8) transformation matrix from Body-Build system
C                          to Spin Reference system
C     COM_BB(3)   (REAL*8) center of mass position [mm] in Body-Build system
C     S_2000(3)   (REAL*8) unit vector of spin axis in GEI-of-J2000 system
C     VST_MJD     (REAL*8) validity start time [UTC]
C     VET_MJD     (REAL*8) validity end time [UTC]
C
C   Subroutine:
C     MJD_STR  for calculating the Modified Julian Day of a CCSDS time string
C
C   Joerg Warnecke  IGM TU BS  1994-07-20    (joerg@geophys.nat.tu-bs.de)
C
      IMPLICIT NONE
C
      REAL*8        PI, RAD
      PARAMETER    (PI=3.14159265358979323846D0, RAD=PI/180.D0)
C
      INTEGER*4     IFLU, IERR, NSAT, LFLU, SCID
      REAL*8        T_MJD, OMEGA, PHI_SRP, A_BBSR(3,3), COM_BB(3), 
     &              S_2000(3), VST_MJD, VET_MJD, PSI_1, PSI_2,
     &              VST, VET, RA, DE, C1, C2, S1, S2
      REAL*4        SPRASC, SPDECL, SPRATE, SCPHAS, COMSHF(3),
     &              TPSI_2, TPSI_1
      CHARACTER*20  VSTTIM, VENTIM, GENTIM
      CHARACTER*1   PREREC
C
C     values read from SATT file have to be saved until the next call
      SAVE LFLU, SCID, PREREC, VST, VET, SPRASC, SPDECL, SPRATE,
     &     SCPHAS, COMSHF, TPSI_2, TPSI_1, GENTIM
C
      DATA LFLU / -9999 /, VST / 0.D0 /, VET / 0.D0 /
C
      IERR = 0
C
      IF ((IFLU.NE.LFLU).OR.(T_MJD.LT.VST)) THEN
        LFLU = IFLU
        REWIND (LFLU,ERR=904)
        READ (LFLU,1000,ERR=904) SCID, PREREC, VSTTIM, VENTIM,
     &    SPRASC, SPDECL, SPRATE, SCPHAS, COMSHF, TPSI_2, TPSI_1, GENTIM
        CALL MJD_STR ( VST, VSTTIM, 0 )
        CALL MJD_STR ( VET, VENTIM, 0 )
        IF (T_MJD.LT.VST) GO TO 901
      END IF
C
      DO WHILE (T_MJD.GT.VET)
        READ (LFLU,1000,ERR=904,END=902) SCID, PREREC, VSTTIM, VENTIM,
     &    SPRASC, SPDECL, SPRATE, SCPHAS, COMSHF, TPSI_2, TPSI_1, GENTIM
        CALL MJD_STR ( VST, VSTTIM, 0 )
        CALL MJD_STR ( VET, VENTIM, 0 )
      END DO
C
      IF (T_MJD.LT.VST) GO TO 903
C
C     number of cluster satellite
      NSAT = SCID
C
C     validation start and end times
      VST_MJD = VST
      VET_MJD = VET
C
C     mean spin frequency
      OMEGA = (PI+PI)*DBLE(SPRATE)/60.D0
C
C     spin phase of sun reference pulse
      PHI_SRP = DBLE(SCPHAS)*RAD
C
C     first and second Euler angles for transformation from
C     Attitude System to Spin Reference System 
      PSI_2 = DBLE(TPSI_2)*RAD
      PSI_1 = DBLE(TPSI_1)*RAD
      C2 = COS(PSI_2)
      S2 = SIN(PSI_2)
      C1 = COS(PSI_1)
      S1 = SIN(PSI_1)
C     A_BBSR = L_SR-A * L_A-B (see DDID, Appendix I.1.3.2)
      A_BBSR(1,1) =    -S2
      A_BBSR(1,2) =     C2
      A_BBSR(1,3) =   0.D0
      A_BBSR(2,1) =  S1*C2
      A_BBSR(2,2) =  S1*S2
      A_BBSR(2,3) =  C1
      A_BBSR(3,1) =  C1*C2
      A_BBSR(3,2) =  C1*S2
      A_BBSR(3,3) = -S1
C
C     center of mass position
      COM_BB(1) = DBLE(COMSHF(1))
      COM_BB(2) = DBLE(COMSHF(2))
      COM_BB(3) = DBLE(COMSHF(3))
C
C     Right Ascension and Declination of spin axis
      RA = DBLE(SPRASC)*RAD
      DE = DBLE(SPDECL)*RAD
C     RA and DE are given in the GEI system of epoch 2000.
C     In this system, the unit vector of the spin axis is:
      S_2000(3) = SIN(DE)
      S_2000(1) = SQRT(1.D0-S_2000(3)*S_2000(3))
      S_2000(2) = S_2000(1)*SIN(RA)
      S_2000(1) = S_2000(1)*COS(RA)
C
      GO TO 900
C
 904  IERR = IERR + 1
 903  IERR = IERR + 1
 902  IERR = IERR + 1
 901  IERR = IERR + 1
 900  RETURN
C
 1000 FORMAT (I2,1X,A1,1X,A20,1X,A20,1X,F6.2,1X,F6.2,1X,F9.6,1X,
     &        F7.3,1X,3(F5.1,1X),F5.2,1X,F5.2,1X,A20)
      END
C********************************************************************C
C
      SUBROUTINE SCS_GSE ( SX, SY, SZ, A )
C
C   Given the GSE unit vector of the spin axis, the matrix for transformation
C   from SCS (spacecraft sun) to GSE coordinates is computed.
C
C   Input parameters:
C     SX, SY, SZ   (REAL*8)  GSE unit vector of the spin axis
C
C   Output parameter:
C     A(3,3)       (REAL*8)  transformation matrix from SCS to GSE
C
C   Authors: Joerg Warnecke & Reinhold Kempen
C            Institute of Geophysics and Meteorology
C            Technical University of Braunschweig
C            1994-11-07
C
      IMPLICIT NONE
C
      REAL*8        SX, SY, SZ, A(3,3)
C
C     The columns of this matrix contain the three unit vectors of SCS
C     given in GSE coordinates; the rows contain the three unit vectors
C     of GSE given in SCS coordinates.
C
C     Z_SCS is the spin axis.
      A(1,3) = SX
      A(2,3) = SY
      A(3,3) = SZ
C
C     The direction from the satellite to the sun lies in the X_SCS/Z_SCS
C     plane, having a positive X_SCS component.  It is approximated by
C     X_GSE, i.e. the unit vector from the center of the earth to the sun. 
C     The maximum error of this assumption is about
C       atan (20 R_E/150.E6 km) = 8.E-4 rad = 0.05 deg
C     With this approximation, we have
      A(1,2) = 0.D0
C     and thus (from A(1,1)^2 + A(1,3)^2 = 1.0, and A(1,1) > 0.0)
      A(1,1) = SQRT(1.D0-SX*SX)
C
C     Y_GSE is perpendicular to X_GSE,
C     so (from A(1,1)*A(2,1) + A(1,2)*A(2,2) + A(1,3)*A(2,3) = 0.0)
      A(2,1) = -SX*SY/A(1,1)
C     and X_SCS is perpendicular to Z_SCS,
C     so (from A(1,1)*A(1,3) + A(2,1)*A(2,3) + A(3,1)*A(3,3) = 0.0)
      A(3,1) = -SX*SZ/A(1,1)
C
C     Y_SCS is the vector product of Z_SCS and X_SCS,
C     so  (from A(2,2) = A(3,3)*A(1,1) - A(1,3)*A(3,1))
      A(2,2) =  SZ/A(1,1)
C     and (from A(3,2) = A(1,3)*A(2,1) - A(2,3)*A(1,1))
      A(3,2) = -SY/A(1,1)
C
      RETURN
      END
C********************************************************************C
        SUBROUTINE CALC_DIPOLE(DATE,SUB_STATUS,MLAT,MLON)
* Calculate position of north magnetic pole using dipole (n=1)
* terms of the spherical harmonic model
*
* Author: Mike Hapgood
*
* Revision
* 22 Jul 1997  original version
*
        IMPLICIT NONE
        INTEGER MJD,J,SUB_STATUS,DATE,MODEL_DATE,OLD_MODEL_DATE
        INTEGER im0,id0
        REAL*8 MLAT,MLON
        REAL*8 DELTA,ONEMINUS,G01,G11,H11

*
* Specify the dipole terms of the IGRF models. We use the 1995 version -
* namely IGRFs for 1900 to 1940, DGRFs for 1945 to 1990, IGRF for 1995
* and extrapolated values for 2000 and 2005 based on the predicted 
* secular variations supplied with IRGF 1995. Strictly these should
* apply only upto 2000 but I have extended to 2005 to support planning
* for Cluster-II ops in 2000-2003.
*
* The first subscript of COEFF indicates the tyep of data:
* 1 = epoch of model, 2 = g^0_1, 3 = g^1_1 and 4 = h^1_1
* The second subscript is the model number 
* Note that models are stacked two per line because of limits
* on the number of continuation lines in f77
        INTEGER N_MODELS 
        REAL COEFF(4,4) 
        DATA N_MODELS /4/
        DATA COEFF /
     *   1990,-29775,-1848,5406 ,   1995,-29682,-1789,5318,
     *   2000,-29594,-1724,5227 ,   2005,-29506,-1659,5136/
*
* Calculate MJD of first model. If requested MJD is before this,
* exit with error
        id0=1
        im0=1
        MODEL_DATE=MJD(INT(COEFF(1,1)),im0,id0)
        IF (DATE.LT.MODEL_DATE) THEN
          SUB_STATUS=5000
          GOTO 9999
        ENDIF
*
* Main loop, Scan through the models to find a pair of models whose
* epochs bracket the date of interest
*
* Initialise OLD_MODEL_DATE to MJD of first model, and status code
* to a failure value 
        OLD_MODEL_DATE=MODEL_DATE
        SUB_STATUS=5100
* Start of main loop
        DO J=2,N_MODELS
* Calculate MJD of next model
          MODEL_DATE=MJD(INT(COEFF(1,J)),im0,id0)
* Check if the requested MJD lies between the MJDs of the previous model
* and the current model
          IF (DATE.LT.MODEL_DATE) THEN
* If so, we can now interpolate the dipole terms to the requested MJD
* 1. calculate interpolation factors DELTA and ONEMINUS
            DELTA=FLOAT(DATE-OLD_MODEL_DATE)/
     *        FLOAT(MODEL_DATE-OLD_MODEL_DATE)
            ONEMINUS=1.0-DELTA
* 2. Interpolate terms g^0_1, g^1_1 and h^1_1
            G01=COEFF(2,J-1)*ONEMINUS+COEFF(2,J)*DELTA
            G11=COEFF(3,J-1)*ONEMINUS+COEFF(3,J)*DELTA
            H11=COEFF(4,J-1)*ONEMINUS+COEFF(4,J)*DELTA
* 3. Set status code to zero (success)
            SUB_STATUS=0
* 4. Exit Do loop. In standard Fortran one would use an EXIT command
* but in grotty old f77 one must use a GOTO. 
            GOTO 9000
          ELSE
* If requested MJD is greater than that for current model, save the
* latter in OLD_MODEL_DATE and go round the loop again
            OLD_MODEL_DATE=MODEL_DATE
          ENDIF
* End of main loop
        ENDDO
 9000   CONTINUE
* 
* Exit if error code set on exit from DO loop
        IF (SUB_STATUS.NE.0) GOTO 9999 
*
* Calculate colatitude and longitude of pole
* MLON is ambiguous, but we know it's in the 4th quadrant, so:
        MLON=ATAN(H11/G11)
        MLON=8.*ATAN(1.)+MLON
* Colatitude based on 1997 corrected formula
        MLAT=2.*ATAN(1.)-ATAN((G11*COS(MLON)+H11*SIN(MLON))/G01)
c        MLAT=ASIN((G11*COS(MLON)+H11*SIN(MLON))/G01)
*
* Come here to exit if error detected
 9999   CONTINUE
*
        END

c        
       INTEGER FUNCTION MJD(YEAR,MONTH,DAY)
C Calculate the Modified Julian Date         
       INTEGER YEAR,MONTH,DAY
       REAL WY,WM
        
       IF (MONTH.GT.2) THEN
         WY=FLOAT(YEAR)
         WM=FLOAT(MONTH+1)
       ELSE
         WY=FLOAT(YEAR-1)
         WM=FLOAT(MONTH+13)
       ENDIF
c        
       MJD=INT(365.25*WY)+INT(30.6001*WM)+DAY-679019
c        
       RETURN
       END
c        
