/* cotrans.c -- coordinate transformations for the CLUSTER FGM data processing
 *modified V4.3  (1996-05-15)
 * Version 6.0  (2000-05-25)
 * uses time dependent dipole position 
 * NOTE: Though this is a C source code, it should be linked into the `Fortran'
 *       library to enable a more convenient access to those subroutines.
 *
 * Edita Georgescu (eg@mpe.mpg.de) & Joerg Warnecke  (joerg@igpp.ucla.edu)
 -----------------------------------------------------------------------------*/

#include "cotrans.h"

int j2ktrans ( int code, double mjd, double *x, double *y, double *z )
/*
 * Transforms a vector (components x, y, z) from the GEI-of-J2000 system
 * into one of the following coordinate systems, according to the value of
 * code:
 *       4    GSE  (geocentric solar ecliptic)
 *       5    GSM  (geocentric solar magnetospheric)
 *       6    SM   (solar magnetic) 
 *       7    J2K  (GEI-of-J2000: no transformation)
 * The value mjd specifies the time [UT] of the vector, in Modified Julian
 * Days, referred to 2000-01-01, 0 UT.
 *
 * Returns 0 on success, or 1 on error.
 */
{
 static double  mjd0=-.123456E7, t0[9], t1[9], t2[9], t3[9], t4[9];
 double         lat, lon, x1, y1, z1, x2, y2, z2, delta=1.0/86400.0;
 int            res,istat,jd;

 if ( code == 7 )
    /* nothing to do */
    return(0);

 if ( fabs(mjd-mjd0) > delta )
   {
    mjd0 = mjd;
/* re-calculate transformation matrices */
    pr2000_  ( &mjd0, &t0[0] );
    gei_geo_ ( &mjd0, &t1[0] );
    gei_gse_ ( &mjd0, &t2[0] );

/* location of northern geomagnetic pole in GEO coordinates */
    jd=(int) (mjd0+ 51544.0);
    calc_dipole_ ( &jd, &istat, &lat, &lon);
//    lat = MAGLAT * M_PI / 180.0;
//    lon = MAGLON * M_PI / 180.0;
    x1 = cos(lat)*cos(lon);
    y1 = cos(lat)*sin(lon);
    z1 = sin(lat);
/* we need the dipole axis in GSE coordinates */
    x2 = t1[0]*x1 + t1[1]*y1 + t1[2]*z1;
    y2 = t1[3]*x1 + t1[4]*y1 + t1[5]*z1;
    z2 = t1[6]*x1 + t1[7]*y1 + t1[8]*z1;
    x1 = t2[0]*x2 + t2[3]*y2 + t2[6]*z2;
    y1 = t2[1]*x2 + t2[4]*y2 + t2[7]*z2;
    z1 = t2[2]*x2 + t2[5]*y2 + t2[8]*z2;
    gse_gsm_ ( &x1, &y1, &z1, &t3[0] );
    gsm_sm_  ( &x1, &y1, &z1, &t4[0] );
   }

 x1 = *x;
 y1 = *y;
 z1 = *z;
 x2 = t0[0]*x1 + t0[3]*y1 + t0[6]*z1;
 y2 = t0[1]*x1 + t0[4]*y1 + t0[7]*z1;
 z2 = t0[2]*x1 + t0[5]*y1 + t0[8]*z1;
 res = 0;

 if ( code == 4 )
   {
    *x = t2[0]*x2 + t2[3]*y2 + t2[6]*z2;
    *y = t2[1]*x2 + t2[4]*y2 + t2[7]*z2;
    *z = t2[2]*x2 + t2[5]*y2 + t2[8]*z2;
   }
 else
   {
    x1 = t2[0]*x2 + t2[3]*y2 + t2[6]*z2;
    y1 = t2[1]*x2 + t2[4]*y2 + t2[7]*z2;
    z1 = t2[2]*x2 + t2[5]*y2 + t2[8]*z2;
    if ( code == 5 )
      {
       *x = t3[0]*x1 + t3[3]*y1 + t3[6]*z1;
       *y = t3[1]*x1 + t3[4]*y1 + t3[7]*z1;
       *z = t3[2]*x1 + t3[5]*y1 + t3[8]*z1;
      }
    else
      {
       x2 = t3[0]*x1 + t3[3]*y1 + t3[6]*z1;
       y2 = t3[1]*x1 + t3[4]*y1 + t3[7]*z1;
       z2 = t3[2]*x1 + t3[5]*y1 + t3[8]*z1;
       if ( code == 6 )
	 {
	  *x = t4[0]*x2 + t4[3]*y2 + t4[6]*z2;
	  *y = t4[1]*x2 + t4[4]*y2 + t4[7]*z2;
	  *z = t4[2]*x2 + t4[5]*y2 + t4[8]*z2;
	 }
       else
	  res = 1;
      }
   }
 return(res);
}
