/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig
 * last version: 5.0 (1996-12-18)
 * timespec_t -> timesp !! for Alpha-DEC
 * libutil.h
 * reads header as array of bytes and uses new function read_ddsheadc,
 * write_ddshead out, to assure portability Alpha-DEC, SUN, etc. 
 * last change: 17/5/00
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "ddscut"
#define VERSION "6.0 (2000-05-15)"
#define PURPOSE "cut out a time interval of telemetry data"
#define USAGE "USAGE:

 ddscut [-b <beginning>] [-e <end>] [-x] [list] | ...

 The program reads telemetry raw data from files given in list (default stdin),
 compares each packet's time stamp with the specified time interval, and 
 writes only those packets to stdout whose time stamps fall into this interval.

 If the modifier '-x' is set, the mode of operation is inverted: only packets
 whose time stamps do not fall into the specified time period are written to 
 stdout. Thus, the given time interval is excluded from the data.

 In both cases, the time interval is inclusive, i.e., the time <beginning>
 and <end> belong to the time interval that is cut out or excluded.

 Note that a given end time will be increased by one block period of
 5.152222 seconds to ensure that the complete data of the selected time
 interval will be written."

#define OPTIONS "OPTIONS:

  -b    let the string <beginning> specify the beginning of the time interval.

  -e    let the string <end> specify the end of the time interval.

      The following formats may be used for these strings:

      If the string contains the letter 'T', it will be interpreted as 
        an ISO time string like '1994-09-01T12:00:04.012Z'.
      If the date is omitted, i.e. the string starts with a 'T', the date
        is being adopted from the time stamp of the first data package.
      If the string does not contain any 'T', it will be interpreted as
        seconds of the UNIX epoch, like '778420804.012443596'.

  -x     switch to `excluding' mode.

  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@stelab.nagoya-u.ac.jp)
 Reinhold Kempen  (reinhold@tu-bs.de)
 Joerg Warnecke   (joerg@igpp.ucla.edu)"

static char *errmsg[] =                           /* error messages */
  {
/*  0 */  "Use '%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Cannot allocate enough memory for the file list.\n",
/*  3 */  "ERROR in %s: Illegal usage.\n",
/*  4 */  "ERROR in %s: Could not open input file %s.\n",
/*  5 */  "ERROR in %s: Cannot allocate enough memory for input data.\n",
/*  6 */  "ERROR in %s while reading of science data.\n",
/*  7 */  "ERROR in %s: Could not evaluate time specification %s.\n",
/*  8 */  "ERROR in %s while writing %s to stdout.\n",
/*  9 */  "ERROR in %s: Invalid time interval specification.\n",
  };


#include "libutil.h"


static void read_telstream ( FILE *fp, char **telstream, int length )
/*----------------------------------------------------------------------------- 
 * First the function 'read_telstream' allocates new memory, if the actual    *
 * memory size is different from the data length to be read. Then the data    *
 * are read. If the memory cannot be allocated or an error occurs during      *
 * reading of the data, the program will be exited with the corresponding     *
 * exit code.                                                                 *
 -----------------------------------------------------------------------------*/
{
 static int telsize;

 if ( telsize != length )
   {
    if ( *telstream ) 
       free ( *telstream );
    telsize = length;
    if ( ( *telstream = malloc ( telsize ) ) == NULL )
      {
       fprintf ( stderr, errmsg[5], PROGRAM );
       exit(5);
      }
   }
 if ( fread ( *telstream, 1, telsize, fp ) != telsize )
   {
    fprintf ( stderr, errmsg[6], PROGRAM );
    exit(6);
   }
}


int main ( int argc, char *argv[] )
{
 FILE               *in_file = stdin;
 ddshead_t          head;
 timesp             t[2], ts;
 static int         i=1, j, k, *file_index, exclude=0, dowrite, dogettimes=1,
                    length;
 char               copt, *telstream = 0, string[2][32];
 headbytes          inbytes;

 string[0][0] = '\0';
 string[1][0] = '\0';

/*----------------------------------------------------------------------------- 
 * Allocate memory for the input_file_index array.                            *
 -----------------------------------------------------------------------------*/
 if ( file_index )
    free ( file_index );
 if ( ( file_index = (int *) calloc ( argc, sizeof(int) ) ) == NULL )
   {
    fprintf ( stderr, errmsg[2], PROGRAM );
    exit(2);
   }
 
/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 while ( i < argc )
   {
    if ( *argv[i] == '-' )                         /* Is it a program option? */
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 'b': 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                strcpy ( string[0], argv[i] );
               }
             else
                strcpy ( string[0], argv[i]+2 );
             break;

          case 'e':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                strcpy ( string[1], argv[i] );
               }
             else
                strcpy ( string[1], argv[i]+2 );
             break;

          case 'x':
             exclude = 1;
             break;

          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default: 
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else                                        /* Is it an input file name? */
      {
       *(file_index + j) = i;
       j++;
      }
    i++;
   }
  
/*----------------------------------------------------------------------------- 
 * open next input file 
 -----------------------------------------------------------------------------*/
 do 
   {
    if ( j != 0 )
      {
       in_file = fopen ( argv[*(file_index + k)], "r" );
       if ( in_file == NULL )
         {
          fprintf ( stderr, errmsg[4], PROGRAM, argv[*(file_index + k)] );
          exit(4);
         }
      }

/*----------------------------------------------------------------------------- 
 * read header bytes until end-of-file
 -----------------------------------------------------------------------------*/
    while (  fread ( inbytes, 1, header_length, in_file ) == header_length )
      { 
/* converts header bytes into dds_head_t structure */
      read_ddsheadc( inbytes, &head);
/* get time stamp of packet */
       ts = cds2unix ( head.t );
       if ( dogettimes )
         {
/* if not yet done, evaluate time strings */
          for ( i = 0; i < 2; i++ )
            {
             if (strlen(string[i]) > 0)
               {
                if ( timestr2unix(string[i],ts,&t[i]) == 0 )
                  {
                   fprintf ( stderr, errmsg[7], PROGRAM, string[i] );
                   exit(7);
                  }
                if ( i == 1 )
                  {   
/* increase block end time to get all vectors of given period */
                   t[i].tv_sec += block_sec;
                   inctime ( &t[i], block_nsec );
                  }
               }
             else if ( i == 0 )
               {
                t[i].tv_sec = 0;
                t[i].tv_nsec = 0;
               }
             else
               {
                t[i].tv_sec = 2147483647;
                t[i].tv_nsec = 999999999;
               }
            }
          if (cmptime(&t[0],&t[1]) > 0)
            {
             fprintf ( stderr, errmsg[9], PROGRAM );
             exit(9);
            }
          dogettimes = 0;
         }

       length = pktlen ( &head );
       read_telstream ( in_file, &telstream, length );

       dowrite = 0;
       if (exclude)
         {
          if ( (cmptime(&ts,&t[0]) < 0) || (cmptime(&ts,&t[1]) > 0) )
             dowrite = 1;
         }
       else
         {
          if ( (cmptime(&ts,&t[0]) >= 0) && (cmptime(&ts,&t[1]) <= 0) )
             dowrite = 1;
         }

       if ( dowrite )
         {
          if ( fwrite ( inbytes, 1, header_length, stdout ) != header_length )
            {
             fprintf ( stderr, errmsg[8], PROGRAM, "DDS header" );
             exit(8);
            }
          if ( fwrite ( telstream, 1, length, stdout ) != length )
            {
             fprintf ( stderr, errmsg[8], PROGRAM, "data packet" );
             exit(8);
            }
         }
      }

    if ( j != 0 )
       fclose ( in_file );
    k++;
   }
 while ( j > k );

 exit(0);
}
