/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching 
 * fgmfreq = nominal value (201.7462), needed in fgmrate 
 * libutil.h
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmav"  
#define VERSION "6.0 (2000-04-15)"
#define PURPOSE "compute averages of high-resolution CLUSTER FGM data"
#define USAGE "USAGE:

 ... | fgmav { {-f|-s|-m} [<int>] | -p [<flt>] | -t <tlfile> } | ... 

 The program reads high-resolution CLUSTER FGM data (in fgmtvec_t format)
 from stdin, computes averages and variance, and writes them to stdout.

 In the FGM status and id word, the flag for averaged data is set, and
 the quality of averaging is indicated by a bit field representing the
 number of missing data points."

#define OPTIONS "OPTIONS:

  -f <int>    compute averages over 1/<int> seconds.  Every <int>th 
              averaging interval starts at full UT second.
              <int> must be smaller then the sampling frequency.

  -s <int>    compute averages over <int> seconds.  Averaging intervals
              begin at a UT of <int>, 2*<int>, 3*<int>, ... seconds.

  -m <int>    compute averages over <int> minutes.  Averaging intervals
              begin at a UT of <int>, 2*<int>, 3*<int>, ... minutes.
 
     <int>    integer number specifying the length of an averaging 
              interval in minutes, seconds, or as fraction of a second.
              Default is 1.

  -p <flt>    compute averages over one spin rotation.  Averaging inter-
              vals begin at a spin phase of <flt> degrees.
      
     <flt>    float number specifying the offset spin angle (in degrees)
              at which a spin averaging interval starts.  Default is 0.

  -t <tlfile> read start, end, and tag of each averaging interval from the
              timelist file <tlfile>.  This is a binary file in the timelist
              format specified by UK CDHF.

  -V          print version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@igpp.ucla.edu)"

#include "libutil.h"        /* #include "utility.h" */

static char *errmsg[] =
   {
/*  0 */  "Type `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Illegal usage.\n",
/*  3 */  "ERROR in %s: Missing parameter {-f|-s|-m|-p|-t}.\n",
/*  4 */  "ERROR in %s while reading from stdin.\n",
/*  5 */  "ERROR in %s: No high-resolution data.\n",
/*  6 */  "ERROR in %s: Timelist file is empty.\n",
/*  7 */  "ERROR in %s: Missing timelist file name.\n",
/*  8 */  "ERROR in %s: Could not open timelist file %s.\n",
/*  9 */  "ERROR in %s while writing to stdout.\n",
/* 10 */  "ERROR in %s while reading timelist file.\n"
/* 11 */  "ERROR in %s: Illegal integration period on timelist file.\n"
   };

int check_phase ( timesp t1, float p1, timesp t2, float p2, float p0,
                 timesp *t0 )
/* ----------------------------------------------------------------------------
 * checks whether a specified spin phase has been run through.
 * If `no', it returns 0.
 * If `yes', it returns 1, and computes the estimated time of that spin phase.
 * If the specified spin period seems to have been run through more than once,
 * it returns -1.
 *
 * Input:  t1  -- time of previous vector,
 *         p1  -- spin phase [rad] of previous vector,
 *         t2  -- time of current vector,
 *         p2  -- spin phase [rad] of current vector,
 *         p0  -- spin phase [rad] to check,
 * Output: t0  -- if set: estimated time of occurence of spin phase p0
 ----------------------------------------------------------------------------*/
{
 timesp      t;
 int             res, isec, nsec;
 double          p20, p10, p21=0.0, t21, pp1, tt1;
 static double   sp_min=3.0, sp_max=5.0;  /* minimum and maximum spin period */

 p10 = (double) p1 - p0;
 if ( p10 < 0.0 ) p10 = p10 + M_PI + M_PI;

 p20 = (double) p2 - p0;
 if ( p20 < 0.0 ) p20 = p20 + M_PI + M_PI;

 p21 = p20 - p10;

 t21 = ((double) (t2.tv_sec - t1.tv_sec))
     + ((double) (t2.tv_nsec - t1.tv_nsec))*1.E-9;

 res = 0;
 if (p21 < -M_PI)             
   {
    if (t21 > 0.0)
      {
       if (t21 < 0.5*sp_max)
          res = 1;
       else
          res = -1;
      }
   }
 else if (p21 < 0.0)          
   {
    if (t21 > 0.5*sp_min)
      {
       if (t21 < sp_max)
          res = 1;
       else
          res = -1;
      }
   }
 else if (p21 < M_PI)       
   {
    if (t21 > sp_min)
      {
       if (t21 < 1.5*sp_max)
          res = 1;
       else
          res = -1;
      }
   }
 else 
   {
    if (t21 > 1.5*sp_min)
      {
       if (t21 < 2.0*sp_max)
          res = 1;
       else
          res = -1;
      }
   }

 if (res == 1)
   {
    /* phase interval from previous vector to `end of spin' */
    pp1 = M_PI + M_PI - p10;
    /* time interval from previous vector to `end of spin' */
    tt1 = pp1 * t21 / (M_PI + M_PI + p21);
    isec = ((int) tt1);
    nsec = ((int) floor ( (tt1 - ((double) isec))*1.E9 + 0.5 ));
    isec += t1.tv_sec;
    nsec += t1.tv_nsec;
    if (nsec > 999999999)
      {
       nsec = nsec - 1000000000;
       isec++;
      }
    else if (nsec < 0)
      {
       nsec = nsec + 1000000000;
       isec--;
      }
   }
 else
   {
    isec = 0;
    nsec = 0;
   }

 t.tv_sec  = isec;
 t.tv_nsec = nsec;
 *t0 = t;
 return(res);
}
    
int tread ( FILE *tfile, timesp *start, timesp *end, timesp *tag )
/* ---------------------------------------------------------------------------
 * reads one record from the timelist file `tfile', and returns
 * beginning, end, and tag of the next averaging interval.
 *
 * tread returns the number of bytes read (25 on success, 0 on EOF).
 * On error, the program quits with an error message.
 ----------------------------------------------------------------------------*/
{
 cdstime_t      cdst;
 int            i;
 unsigned char  status[1];
 double         test;

 i = read_cdstime ( tfile, &cdst );
 if ( i == 8 ) 
   {
    *start = cds2unix ( cdst );
    i += read_cdstime ( tfile, &cdst );
    if ( i == 16 )
      {
       *end = cds2unix ( cdst );
       /* SECURITY CHECK: prevent users from getting high time-resolution */
       test = ((double) (end->tv_sec  - start->tv_sec)) +
              ((double) (end->tv_nsec - start->tv_nsec))*1.E-9;
       if ( test < 1.0 )
         {            
          fprintf  ( stderr, errmsg[11], PROGRAM );
          exit(11);
         }
       i += read_cdstime ( tfile, &cdst );
       if ( i == 24 )
         {
          *tag = cds2unix ( cdst );
          i += fread ( status, 1, 1, tfile );
         }
      }
   }

 if ( i != 25 )
   {
    if ( ferror(tfile) )
      {
       fprintf  ( stderr, errmsg[10], PROGRAM );
       exit(10);
      }
    fclose ( tfile );
    i = 0;
   }
 return(i);
}
     
int main ( int argc, char *argv[] )
{
 FILE          *tfile=NULL;
 timesp    t1, t2, t0, tl;
 fgmtvec_t     fvi, fvo;                   
 int           i, secs=0, frac=0, num, nex=0, in, out, isen, itel, idec=0;
 int           ckph, fvsz, tr=0, loop=0, iqoa;
 unsigned int  isec;
 double        freq=0.0, nsecs=0.0, tav=0.0, fgmfreq=201.7462, sbx, sby, sbz, 
               sb2, sbm, b2, av_of_bx, av_of_by, av_of_bz, av_of_b2, 
               av_of_bm, m2_of_bav, missing;
 float         offset=0.0, offrad, pl;
 char          copt, tname[128];

 /* set timelist file name to empty */
 tname[0] = '\0';

 /* get command line parameters */
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch(copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else 
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);
             
          case 'f': 
             if ( secs == 0 && frac == 0 )
               {
                if ( *(argv[i]+2) == '\0' && i+1 < argc )
                  {
                   i++;
                   frac = atoi ( argv[i] );
                   if ( frac <= 0 )
                      i--;
                  }
                else
                   frac = atoi ( argv[i]+2 );
                if ( frac <= 0 )
                   frac = 1;
                nsecs = 1.E9 / ((double) frac);
               }
             else 
                fprintf ( stderr, "%s: option -f ignored.\n", PROGRAM );
             break;

          case 's': 
             if ( secs == 0 && frac == 0 )
               {
                if ( *(argv[i]+2) == '\0' && i+1 < argc )
                  {
                   i++;
                   secs = atoi ( argv[i] );
                   if ( secs <= 0 )
                      i--;
                  }
                else
                   secs = atoi ( argv[i]+2 );
                if ( secs <= 0 )
                   secs = 1;
               }
             else 
                fprintf ( stderr, "%s: option -s ignored.\n", PROGRAM );
             break;

          case 'm': 
             if ( secs == 0 && frac == 0 )
               {
                if ( *(argv[i]+2) == '\0' && i+1 < argc )
                  {
                   i++;
                   secs = 60 * atoi ( argv[i] );
                   if ( secs <= 0 )
                      i--;
                  }
                else
                   secs = 60 * atoi ( argv[i]+2 );
                if ( secs <= 0)
                   secs = 60;
               }
             else 
                fprintf ( stderr, "%s: option -m ignored.\n", PROGRAM );
             break;

          case 'p': 
             if ( secs == 0 && frac == 0 )
               {
                secs = -1;
                if ( *(argv[i]+2) == '\0' && i+1 < argc )
                  {
                   i++;
                   offset = atof ( argv[i] );
                   if ( offset < 0.0 )
                      i--;
                  }
                else
                   offset = atof ( argv[i]+2 );
                if ( offset < 0.0 )
                   offset = 0.0;
               }
             else 
                fprintf ( stderr, "%s: option -p ignored.\n", PROGRAM );
             break;

          case 't': 
             if (secs == 0 && frac == 0)
               {
                if ( *(argv[i]+2) == '\0' && i+1 < argc )
                  {
                   if ( *(argv[i+1]) != '-' )
                     {
                      i++;
                      strcpy ( tname, argv[i] );
                     }
                  }
                else
                   strcpy ( tname, argv[i]+2 );
                if ( strlen(tname) <= 0 )
                  {
                   fprintf ( stderr, errmsg[7], PROGRAM );
                   fprintf ( stderr, errmsg[0], argv[0] );
                   exit(7);
                  }
                tfile = fopen ( tname, "r" );
                if ( tfile == NULL )
                  {
                   fprintf ( stderr, errmsg[8], PROGRAM, tname );
                   exit(8);
                  }
                secs = -2;
               }
             else 
                fprintf ( stderr, "%s: option -t ignored.\n", PROGRAM );
             break;

          case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[2], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(2);
      }
    i++;
   }

 if ( secs == 0 && frac == 0 )
   {
    fprintf ( stderr, errmsg[3], PROGRAM );
    fprintf ( stderr, errmsg[0], argv[0] );
    exit(3);
   }

 offrad = offset * M_PI / 180.0;

 /* read first input structure */
 fvsz = sizeof ( fgmtvec_t );
 in = read_fgmtvec ( stdin, &fvi );

 if ( in == fvsz )
   {
    if ( subbits ( fvi.stat, 23, 24 ) )   /* if data are averaged */
      {
       fprintf ( stderr, errmsg[5], PROGRAM );
       exit(5);
      }
    if ( secs >= 0 )
      {
       /* beginning of first averaging interval: t2 */
       if (secs > 0)
         {
          isec = ((unsigned int) fvi.tv.tv_sec%secs);
          if ( ((float) isec) > 0.5*((float) secs) )
             t2.tv_sec = fvi.tv.tv_sec - isec + secs;
          else 
             t2.tv_sec = fvi.tv.tv_sec - isec;
          t2.tv_nsec = 0;
         }
       else
         {
          t2.tv_sec = fvi.tv.tv_sec;
          loop = (int) floor ( 0.5 + (double) fvi.tv.tv_nsec / nsecs );
          if ( loop < frac )
             t2.tv_nsec = (int) floor ( 0.5 + (double) loop * nsecs );
          else
            {
             t2.tv_sec++;
             t2.tv_nsec = 0;
             loop = 0;
            }
         }
       while ( (cmptime(&fvi.tv,&t2) < 0) && (in == fvsz) )
          /* skip forward to beginning of first averaging interval */
          in = read_fgmtvec ( stdin, &fvi );
      }
    else if ( secs == -1 )
      {
       /* skip forward to beginning of next spin */
       ckph = 0;
       while ( (ckph == 0) && (in == fvsz) )
         {
          tl = fvi.tv;
          pl = fvi.phavar;
          in = read_fgmtvec ( stdin, &fvi );
          if (in == fvsz)
             ckph = check_phase (tl,pl,fvi.tv,fvi.phavar,offrad,&t2);
         }
      }
    else
      {
       /* first averaging interval */
       tr = tread ( tfile, &t1, &t2, &t0 );
       if ( tr <= 0 )
         {
          fprintf ( stderr, errmsg[6], PROGRAM );
          exit(6);
         }
       while ( (cmptime(&fvi.tv,&t2) > 0) && (tr > 0) )
          tr = tread ( tfile, &t1, &t2, &t0 );
       /* skip forward to beginning of first averaging interval */
       while ( (cmptime(&fvi.tv,&t1) < 0) && (in = fvsz) )
          in = read_fgmtvec ( stdin, &fvi );
      }
   }
 else if ( in > 0 )
   {
    fprintf ( stderr, errmsg[4], PROGRAM );
    exit(4);
   }

 while ( in == fvsz )
   {
    if ( subbits ( fvi.stat, 23, 24 ) )            /* if data are averaged */
       fprintf ( stderr, "%s WARNING: no high-resolution data !\n", PROGRAM );
    isen = subbits ( fvi.stat, 28, 30 );           /* sensor number */
    itel = subbits ( fvi.stat,  0,  4 );           /* telemetry option */
    fgmrate ( isen, itel, &freq, &idec, fgmfreq ); /* sampling frequency of FGM */

    /* Initialization of sums */
    sbx = 0.0;
    sby = 0.0;
    sbz = 0.0;
    sb2 = 0.0;
    sbm = 0.0;
    num = 0;

    /* take status bits of first vector */
    fvo.stat  = fvi.stat;

    if (secs >= 0)
      {
       t1 = t2;
       if (secs > 0)
         {
          t2.tv_sec = t1.tv_sec + secs;
          /* averaging time period */
          tav = ((double) secs);
          /* tag time: center of averaging interval */
          t0.tv_sec = t1.tv_sec + secs/2;
          t0.tv_nsec = (secs%2)*500000000;
         }
       else
         {
          loop++;
          if ( loop < frac )
             t2.tv_nsec = (int) floor ( 0.5 + (double) loop * nsecs );
          else 
            {
             t2.tv_sec++;
             t2.tv_nsec = 0;
             loop = 0;
            }
          /* averaging time period */
          tav = nsecs/1.E9;
          /* tag time: center of averaging interval */
          t0.tv_sec = t1.tv_sec;
          t0.tv_nsec = t1.tv_nsec + (int) floor ( 0.5 + 0.5*nsecs );
         }
       while ( (cmptime(&fvi.tv,&t2) < 0) && (in == fvsz) )
         {
          sbx += fvi.b[0];
          sby += fvi.b[1];
          sbz += fvi.b[2];
          b2 = (fvi.b[0]*fvi.b[0]) + (fvi.b[1]*fvi.b[1]) + (fvi.b[2]*fvi.b[2]);
          sb2 += b2;
          sbm += sqrt(b2);
          num++;
          /* read next vector */
          in = read_fgmtvec ( stdin, &fvi );
         }
      }
    else if (secs == -1)
      {
       t1 = t2;
       ckph = 0; 
       while ( (ckph == 0) && (in == fvsz) )
         {
          sbx += fvi.b[0];
          sby += fvi.b[1];
          sbz += fvi.b[2];
          b2 = (fvi.b[0]*fvi.b[0]) + (fvi.b[1]*fvi.b[1]) + (fvi.b[2]*fvi.b[2]);
          sb2 += b2;
          sbm += sqrt(b2);
          num++;
          /* time and phase of previous vector */
          tl = fvi.tv;
          pl = fvi.phavar;
          in = read_fgmtvec ( stdin, &fvi );
          if (in == fvsz)
             ckph = check_phase (tl,pl,fvi.tv,fvi.phavar,offrad,&t2);
         }
       if (ckph > 0)
         {
          /* averaging time = spin period */
          tav = ((double) (t2.tv_sec - t1.tv_sec)) +
                ((double) (t2.tv_nsec - t1.tv_nsec))*1.E-9;
          /* tag time: center of averaging interval */
          isec = ((unsigned int) floor ( tav*0.5E9 + 0.5 ) );
          t0.tv_sec  = t1.tv_sec + (isec/1000000000);
          t0.tv_nsec = t1.tv_nsec + isec%1000000000;
          if ( t0.tv_nsec > 999999999 )
            {
             t0.tv_nsec = t0.tv_nsec - 1000000000;
             t0.tv_sec  = t0.tv_sec  + 1;
            }
         }
       else if (ckph < 0)
         {
          /* skip forward to beginning of next spin */
          ckph = 0;
          while ( (ckph == 0) && (in == fvsz) )
            {
             tl = fvi.tv;
             pl = fvi.phavar;
             in = read_fgmtvec ( stdin, &fvi );
             if (in == fvsz)
                ckph = check_phase (tl,pl,fvi.tv,fvi.phavar,offrad,&t2);
            }
          num = 0;
         }
       else
         {
          /* end of file, throw away incomplete spin */
          num = 0;
         }
      }
    else if ( tr > 0 )
      {
       while ( (cmptime(&fvi.tv,&t2) < 0) && (in == fvsz) )
         {
          sbx += fvi.b[0];
          sby += fvi.b[1];
          sbz += fvi.b[2];
          b2 = (fvi.b[0]*fvi.b[0]) + (fvi.b[1]*fvi.b[1]) + (fvi.b[2]*fvi.b[2]);
          sb2 += b2;
          sbm += sqrt(b2);
          num++;
          /* read next vector */
          in = read_fgmtvec ( stdin, &fvi );
         }
       /* averaging time period */
       tav = ((double) (t2.tv_sec - t1.tv_sec)) +
             ((double) (t2.tv_nsec - t1.tv_nsec))*1.E-9;
      }

    if ( (num > 0) && (idec > 0) )
      {
       /* tag time: center of averaging interval */
       fvo.tv = t0;
       /* compute averages and variance */
       av_of_bx = sbx / num;
       av_of_by = sby / num;
       av_of_bz = sbz / num;
       av_of_b2 = sb2 / num;
       av_of_bm = sbm / num;
       m2_of_bav = ( av_of_bx * av_of_bx ) + 
                   ( av_of_by * av_of_by ) +
                   ( av_of_bz * av_of_bz );
       fvo.b[0] = av_of_bx;
       fvo.b[1] = av_of_by;
       fvo.b[2] = av_of_bz;
       fvo.phavar = 1 - ( m2_of_bav / av_of_b2 );
       fvo.magvar = 1 - ( ( av_of_bm * av_of_bm ) / av_of_b2 );
 
       /* number of expected data points */
       nex = ((int) floor ( tav * freq ));
       /* number of missing data points */
       missing = 100.0 * ( nex - num ) / nex;
       if ( missing <= 0.0 )
          iqoa = 0;
       else if ( missing < 1.0 )
          iqoa = 1;
       else if ( missing < 3.0 )
          iqoa = 2;
       else if ( missing < 6.0 )
          iqoa = 3;
       else if ( missing < 10.0 )
          iqoa = 4;
       else if ( missing < 30.0 )
          iqoa = (int) floor ( missing / 5.0 ) + 3;
       else
          iqoa = (int) floor ( missing / 10.0 ) + 6;

       /* set status bits */
       fvo.stat |= iqoa << 19;
       fvo.stat |= 1 << 23;

       /* write output structure */
       out = write_fgmtvec ( stdout, &fvo );
       if (out < fvsz) 
         {
          fprintf  ( stderr, errmsg[9], PROGRAM );
          exit(9);
         }
      }

    if ((secs == -2) && (tr > 0))
      {
       /* read next averaging interval from timelist file */
       tr = tread ( tfile, &t1, &t2, &t0 );
       if ( tr > 0 )
         {
          /* skip forward to beginning of averaging interval */
          while ( (cmptime(&fvi.tv,&t1) < 0) && (in == fvsz) )
             in = read_fgmtvec ( stdin, &fvi );
         }
       else
         {
          /* EOF on timelist file => skip remaining input data */
          while (in == fvsz)
             in = read_fgmtvec ( stdin, &fvi );
         }
      }
   }

 /* quit */
 exit(0);
}
