/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig 2000 MPE-Garching
 * modified version 5.0 (1996-12-18)"
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmcut"  
#define VERSION "6.0 (2000-05-25)"
#define PURPOSE "cut out vectors of CLUSTER FGM data"
#define USAGE "USAGE:

 ... | fgmcut [-b <beginning>] [-e <end>] [-x] 
              [-s <sc>] [-a <adc>] [-r <range>] [-u <unit>] | ...

 The program reads CLUSTER FGM data (in fgmtvec_t format) from stdin, compares
 each vector's time with the specified time interval, and writes only those
 vectors to stdout which fall into this time period. 

 If the modifier `-x' is set, the mode of operation is inverted: only vectors
 which do not fall into the specified time period are written to stdout.
 Thus, the given time interval is excluded from the data stream.

 In both cases, the time interval is inclusive, i.e., the times <beginning>
 and <end> belong to the time interval that is cut out or excluded.

 When specifying `sc', `adc', `range' or `sensor' only those vectors will be
 written to stdout that have the specified setting for this parameter."


#define OPTIONS "OPTIONS:

  -b    let the string <beginning> specify the beginning of the time interval.

  -e    let the string <end> specify the end of the time interval.

      The following formats may be used for these strings:

      If the string contains the letter `T', it will be interpreted as 
        an ISO time string like `1994-09-01T12:00:04.012Z'.
      If the date is omitted, i.e. the string starts with a `T', the date
        is being adopted from the time stamp of the first input vector.
      If the string does not contain any `T', it will be interpreted as
        seconds of the UNIX epoch, like `778420804.012443596'.

  -x    switch to `excluding' mode.

  -s    write only vectors from spacecraft <sc> (1-4).

  -a    write only vectors measured with ADC number <adc>.
        0 = default ADC unit, 1 = spare ADC unit

  -r    write only vectors with range <range> (1-7).

  -u    write only vectors from the sensor <unit>.
        0 = outboard sensor unit, 1 = inboard sensor unit

  -V    print version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@geophys.nat.tu-bs.de)"


#include "libutil.h"

static char *errmsg[] =
  {
/*  0 */  "Type `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Illegal usage.\n",
/*  3 */  "ERROR in %s: Could not evaluate time specification %s.\n",
/*  4 */  "ERROR in %s: Invalid time interval specification.\n",
/*  5 */  "ERROR in %s: Invalid s/c specification.\n",
/*  6 */  "ERROR in %s: Invalid adc specification.\n",
/*  7 */  "ERROR in %s: Invalid range specification.\n",
/*  8 */  "ERROR in %s: Invalid sensor specification.\n",
  };

int main ( int argc, char *argv[] )
{
 fgmtvec_t    fv;
 timesp       t[2];
 int          i, fvsz, exclude = 0, dogettimes = 1,
              dosc = 0, doadc = 0, dorange = 0, dosensor = 0,
              sc = 1, adc = 0, range = 1, sensor = 0;
 char         copt, string[2][32];

 fvsz = sizeof ( fgmtvec_t );
 string[0][0] = '\0';
 string[1][0] = '\0';

 /* evaluate command line parameters */
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch (copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else 
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 'x':
             exclude = 1;
             break;

          case 'b': 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                strcpy ( string[0], argv[i] );
               }
             else
                strcpy ( string[0], argv[i]+2 );
             break;

          case 'e':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                strcpy ( string[1], argv[i] );
               }
             else
                strcpy ( string[1], argv[i]+2 );
             break;

          case 's':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                sc = atoi ( argv[i] );
                dosc = 1;
               }
             else
               {
                sc = atoi ( argv[i]+2 );
                dosc = 1;
               }
             break;

          case 'a':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                adc = atoi ( argv[i] );
                doadc = 1;
               }
             else
               {
                adc = atoi ( argv[i]+2 );
                doadc = 1;
               }
             break;

          case 'r':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                range = atoi ( argv[i] );
                dorange = 1;
               }
             else
               {
                range = atoi ( argv[i]+2 );
                dorange = 1;
               }
             break;

          case 'u':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               { 
                i++;
                sensor = atoi ( argv[i] );
                dosensor = 1;
               }
             else
               {
                sensor = atoi ( argv[i]+2 );
                dosensor = 1;
               }
             break;

          case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[2], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(2);
      }
    i++;
   }

 if ( sc < 1 || sc > 4 )
   {
    fprintf ( stderr, errmsg[5], PROGRAM );
    exit(5);
   }

 if ( adc < 0 || adc > 1 )
   {
    fprintf ( stderr, errmsg[6], PROGRAM );
    exit(6);
   }

 if ( range < 1 || range > 7 )
   {
    fprintf ( stderr, errmsg[7], PROGRAM );
    exit(7);
   }

 if ( sensor < 0 || sensor > 1 )
   {
    fprintf ( stderr, errmsg[8], PROGRAM );
    exit(8);
   }

 while ( read_fgmtvec(stdin,&fv) == fvsz )
   {
    if ( dogettimes )
      {
       /* if not yet done, evaluate time strings */
       for ( i = 0; i < 2; i++ )
         {
          if (strlen(string[i]) > 0)
            {
             if ( timestr2unix(string[i],fv.tv,&t[i]) == 0 )
               {
                fprintf ( stderr, errmsg[3], PROGRAM, string[i] );
                exit(3);
               }
            }
          else if ( i == 0 )
            {
             t[i].tv_sec = 0;
             t[i].tv_nsec = 0;
            }
          else
            {
             t[i].tv_sec = 2147483647;
             t[i].tv_nsec = 999999999;
            }
         }
       if (cmptime(&t[0],&t[1]) > 0)
         {
          fprintf ( stderr, errmsg[4], PROGRAM );
          exit(4);
         }
       dogettimes = 0;
      }

    if ( dosc && subbits ( fv.stat, 30, 32 ) + 1 != sc )
       continue;

    if ( doadc && subbits ( fv.stat, 8, 9 ) != adc )
       continue;

    if ( dorange && subbits ( fv.stat, 4, 7 ) != range )
       continue;

    if ( dosensor && 
         ( subbits(fv.stat, 7, 8) ^ subbits(fv.stat, 28, 30) ) != sensor )
       continue;

    if ( exclude )
      {
       if ( (cmptime(&fv.tv,&t[0]) < 0) || (cmptime(&fv.tv,&t[1]) > 0) )
          write_fgmtvec ( stdout, &fv );
      }
    else
      {
       if ( (cmptime(&fv.tv,&t[0]) >= 0) && (cmptime(&fv.tv,&t[1]) <= 0) )
          write_fgmtvec ( stdout, &fv );
      }
   }
 exit(0);
}
