/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching 
 * 6.0 (2000-04-15) : get_fgmconstants (alpha)
 * check_bad & discard
 * while loop changed; "-a" bug fixed
 * V6.1 (2000-05-25) : time dependent dipole position
 * instead of fixed MAGLAT, MAGLON in cotrans.h
 * check_auxf added, now '-a' accepts attfile WITH or WITHOUT DDS-headers
 * V6.2 (2000-09-30) : attitude time gap "bug" fixed, new 'const.fgm' format
 */
 

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmhrt"
#define VERSION "6.2 (2000-09-30)"
#define PURPOSE "transform CLUSTER FGM high-resolution data"
#define USAGE "USAGE:

  ... | fgmhrt [-a <attfile>] [-d <delta>] [-s <sys>] | ...

  The program performs a coordinate transformation on high-resolution
  (non-averaged) CLUSTER FGM vectors (in fgmtvec_t format). 

  On input, the vectors may be in one of the following coordinate systems:
  `fsr' (FGM spin reference), or `sr' (spin reference).

  If in the FGM status word, the bit for `averaged data' is set, fgmhrt
  proceeds from the non-spinning version of the input coordinate system.
  In this case, if fgmhrt has to despin the data, it will use a fixed 
  spin phase value of 0.0, and it will clear the bit for `averaged data',
  again.  Therefore, this program should not be used with data that has
  really been averaged!"

#define OPTIONS "OPTIONS:

  -a     use <attfile> as attitude file (*ga* on the CDROM). The file may, 
         or may not contain DDS-headers. Default is to use the filename: 
         `$SATTPATH/satt.cl#'(#=1,2,3,4), which contains preprocessed 
         attitude file, DDS-headers beeing removed.

  -d     use <delta> as the number of seconds after which new transformation
         matrices to GSE, GSM, and SM are computed.  Default is 1 sec.

  -s     select the output coordinate system.  <sys> may be
         `sr'   for spin-reference system,
         `scs'  for spacecraft-sun system,
         `gse'  for geocentric solar ecliptic system (default), 
         `gsm'  for geocentric solar magnetospheric system,
         `sm'   for solar magnetic system, or
         `j2k'  for geocentric equatorial inertial system of epoch J2000.0.

  -V     print version number on stdout, then exit."

#define AUTHOR "AUTHORS:
 
 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@igpp.ucla.edu)"


#include "libutil.h"
#include "cotrans.h"

static char *errmsg[] =                           /* error messages */
  {
/*  0 */  "Type `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Unknown coordinate system specified.\n",
/*  2 */  "ERROR in %s: Illegal option %s.\n",
/*  3 */  "ERROR in %s: Illegal usage.\n",
/*  4 */  "ERROR in %s while reading from stdin.\n",
/*  5 */  "ERROR in %s: Could not open attitude file %s.\n",
/*  6 */  "ERROR in %s while reading attitude file %s\n",
/*  7 */  "ERROR in %s while writing to stdout.\n",
/*  8 */  "ERROR in %s: Illegal input coordinate system (code %d).\n",
/*  9 */  "ERROR in %s: Illegal output coordinate system %s.\n",
/* 10 */  "ERROR in %s: Input data from multiple s/c.\n",
/* 11 */  "ERROR in %s: Input data from mixed reference system\n",
 };

static char *warning[] =                          /* warning messages */
{
/*  0 */ "%s WARNING: Environment variable %s is not set.\n",
/*  1 */ "%s WARNING: Gap in attitude file %s.\n",
};

int check_auxf(char *inp, int isat)
/**********************************************************************
 * check aux file format                                              *
 * if it contains DDS-headers then remove and output to file: tmp_att */

{
 FILE *inf, *outf=NULL;
 unsigned char c1[2],line[600],c[16]="",outp[8]="tmp_att";
 int len=15,first=0,i,llen;
 
 sprintf(c1,"%1d",isat);
 if ( (inf=fopen(inp,"r")) == NULL) {
    fprintf(stderr,"file %s not found\n",inp);
    return (-1);
    }  
 while ( (i=fread(c,1,len,inf)) == len ) {
   if ((c[8] < 6) && (c[8] > 0))
     {
     llen=(int)c[11]+(int)c[10]*256;
     fread(line,1,llen,inf);
     if (!first) {
        first=1;
        if ( (outf=fopen(outp,"w")) == NULL) {
           fprintf(stderr,"could not open file %s \n",outp);
           return -1;
           } 
         }   
     if ( (line[2] == *c1) || (line[1] == *c1) )
        fwrite(line,llen,1,outf);
     }
     else{
     if ( (c[1] == *c1) || (c[2] == *c1) )
        return 0;
     }  
  } //end while 
if (first) 
   fclose(outf);
if (inf != NULL)
   fclose(inf);
return 1;
} 

int main ( int argc, char *argv[], char *envp[] )
{
 FILE          *infile=stdin;
 fgmtvec_t     fv;
 int           fvsz=sizeof(fgmtvec_t), unit=0, delta=1, out=4, lsec=0,
               isat, idat, ierr, in, i, first=1, isat0=-1, in0=-1;
 double        a[9], c[9], p[9], t1[9], t2[9], t3[9], t4[9], 
               s_gei[3], s_gse[3], s[3], com[3], m_geo[3], m_gei[3], m_gse[3],
               d_bb[3], d_sr[3], d_scs[3], d_gse[3], d_tmp[3], 
               alpha=6.5, sa=0., ca=1., phi, sp, cp, lat, lon, omega, phisrp, tmjd,
               vstmjd=0.0, vetmjd=0.0;
 double        hfcfrq, fgmfreq, spinph0;
 int           obdelay[2], ibdelay[2], istat, jd, msg=0;
 char          copt, aname[128];

/* known coordinate systems */
 const char   *sys[8] = {"fs", "fsr", "sr", "scs", "gse", "gsm", "sm", "j2k"};

/* only to debug !!
 char  inp[]="./fgm";  
 if ( (infile = fopen(inp, "rb"))==NULL){
    fprintf(stderr,"file %s not found\n",inp);
    return(-1);
    }*/

/* set attitude file name to empty */
 aname[0] = '\0';

/* get command line parameters */
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 'a': 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               {
                if ( *argv[i+1] != '-' )
                  { 
                   i++;
                   strcpy ( aname, argv[i] );
                  }
               }
             else
                strcpy ( aname, argv[i]+2 );
             break;

          case 's': 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               {
                if ( *argv[i+1] != '-' )
                  { 
                   i++;
                   for ( out = 7; out >= 0; out-- )
                      if ( !strcmp ( argv[i], sys[out] ) ) 
                         break;
                  }
               }
             else
               {
                for ( out = 7; out >= 0; out-- )
                   if ( !strcmp ( argv[i]+2, sys[out] ) ) 
                      break;
               }
             if ( out < 0 )
               {
                fprintf ( stderr, errmsg[1], PROGRAM );
                fprintf ( stderr, errmsg[0], argv[0] );
                exit(1);
               }
             break;

          case 'd': 
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
               {
                i++;
                delta = atoi ( argv[i] );
                if ( delta < 0 )
                   i--;
               }
             else
                delta = atoi ( argv[i]+2 );
             if ( delta < 0 )
                delta = 1;
             break;
             
          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[2], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(2);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[3], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(3);
      }
    i++;
   }

/* read input structures */
 while ( read_fgmtvec ( infile, &fv ) == fvsz )
 {
/* discard vectors marked 'BAD' */
 if ( check_bad(fv.stat) )
     continue;

/* get the satellite number for opening the appropriate attitude file */
 isat = 1 + subbits ( fv.stat, 30, 32 );
 
/*-------------------------------------------------------------------------- 
 * Get the FGM sampling frequency, HFC frequency & other "constants".         *
 * Reference direction of FGM experiment: alpha = angle between Y-FGM 
 * (sensor system) and Y-BB (Body-Build system) 
 *--------------------------------------------------------------------------*/
 if (first) 
    {
/* coordinate system of input field vectors */
    in0 = subbits(fv.stat,16,19);
    if ((get_fgmconstants((isat-1), &hfcfrq, &fgmfreq, &spinph0, &alpha,
                          &obdelay[0], &ibdelay[0])) < 0)
       {
       fprintf(stderr,"file read error in 'const.fgm'\n");
       exit(4);
       }
    alpha = alpha * M_PI / 180.0;
    ca = cos(alpha);
    sa = sin(alpha);
    isat0 = isat;
    first = 0;
    }

 if ( isat != isat0 ) 
    {
    fprintf( stderr, errmsg[10], PROGRAM );
    continue; //exit(10);
    }

 in = subbits(fv.stat,16,19);
 if (in != in0)
    {
    fprintf( stderr, errmsg[11], PROGRAM );
    exit(11);
    }
    
/* coordinate system of input field vectors is FS */
 if (in < 1) 
    {
    fprintf( stderr,errmsg[8], PROGRAM );
    exit(8);
    }
    
 if (in != out)
    {
/* calculate Modified Julian Date, referred to 2000-01-01, 0 UT */
    tmjd = mjd2000 ( fv.tv );
    if (out > 2) 
/* need attitude information: if (out > 2) at least:  FSR->SCS */
       {
       if ( unit == 0 )
          {
/* location of northern geomagnetic pole */
          jd = (int) (tmjd + 51544.0);
          calc_dipole_ ( &jd, &istat, &lat, &lon);
          m_geo[0] = cos(lat)*cos(lon);
          m_geo[1] = cos(lat)*sin(lon);
          m_geo[2] = sin(lat);
/* check if attitude file name given */
          i = strlen ( aname );
          if ( i > 0 ) {
/* check if dds_header & remove it */
             if (check_auxf(aname, isat) > 0){
                 strcpy(aname,"");
                 strcat(aname,"tmp_att");
                 i=strlen(aname);
                 }
             }
             else
             {
/* use default file name */
             for ( i= 0; envp[i] != NULL; i++ )
                 if ( !strncmp(envp[i],"SATTPATH",8) )
                    {
                    strcpy ( aname, getenv ( "SATTPATH" ) );
                    strcat ( aname, "/" );
                    break;
                    }
             i = strlen ( aname );
             if ( i == 0 )
                fprintf ( stderr, warning[0], PROGRAM, "SATTPATH" );
             sprintf ( &aname[i], "satt.cl%1d", isat );
             i = strlen ( aname );
             }
/* open attitude file */
          unit = iflu_att_ ( &isat, aname, i );
          if ( unit <= 0 )
             {
              fprintf ( stderr, errmsg[5], PROGRAM, aname );
              exit(5);
             }
             else if ( subbits(fv.stat,30,32) != isat-1 )
             {
              fprintf ( stderr, errmsg[10], PROGRAM );
              exit(10);
             }
          }                        //end if (unit == 0)

/* get new attitude information */
          if ( (tmjd < vstmjd) || (tmjd > vetmjd) )
             {
             satt_ ( &tmjd, &unit, &ierr, &isat, &omega, &phisrp, &a[0],
                     &com[0], &s[0], &vstmjd, &vetmjd );
             if (ierr != 0)
                {
                if ( ierr < 3 )
                  {
                   fprintf ( stderr, errmsg[6], PROGRAM, aname );
	           fprintf ( stderr, "-- Attitude data not found " );
                   switch ( ierr )
                     {
                      case 1:
                         fprintf ( stderr, "(time too early) --\n" );
                         break;
                      case 2:
                         fprintf ( stderr, "(time too late) --\n" );
                         break;
                     }
		   exit(6);
                  }
                  else
                  {
		  if ( !msg){	
		     fprintf ( stderr, warning[1], PROGRAM, aname );	
		     msg=1;
		     }
                  continue;
		  }
                }
/* convert spin axis to GEI system of MJD */
            pr2000_ ( &tmjd, &p[0] );
            s_gei[0] = p[0]*s[0] + p[3]*s[1] + p[6]*s[2];
            s_gei[1] = p[1]*s[0] + p[4]*s[1] + p[7]*s[2];
            s_gei[2] = p[2]*s[0] + p[5]*s[1] + p[8]*s[2];
            }                     //end if (out > 2)
         }                        //end if (in != out)
       msg=0;
/* first we go from `in' to SR system */

       if ( in == 1 )
          {
/* transform data from FSR to SR system:
          (rotate about X-FSR axis with angle alpha, then rename axes) */
          d_sr[2] = ((double) fv.b[0]);
          d_sr[0] = ((double) fv.b[1])*ca + ((double) fv.b[2])*sa;
          d_sr[1] = ((double) fv.b[2])*ca - ((double) fv.b[1])*sa;
          }
          else if ( in == 2 ) 
          {
/* input data are in SR system */
          d_sr[0] = ((double) fv.b[0]);
          d_sr[1] = ((double) fv.b[1]);
          d_sr[2] = ((double) fv.b[2]);
          }
          else if ( in == 0 )
          {
/* transform data from FS to BB system:
          (rotate about X-FGM axis with angle alpha) */
          d_bb[0] = ((double) fv.b[0]);
          d_bb[1] = ((double) fv.b[1])*ca + ((double) fv.b[2])*sa;
          d_bb[2] = ((double) fv.b[2])*ca - ((double) fv.b[1])*sa;
/* transform data from BB to SR system */
          d_sr[0] = a[0]*d_bb[0] + a[3]*d_bb[1] + a[6]*d_bb[2];
          d_sr[1] = a[1]*d_bb[0] + a[4]*d_bb[1] + a[7]*d_bb[2];
          d_sr[2] = a[2]*d_bb[0] + a[5]*d_bb[1] + a[8]*d_bb[2];
          }
          else
          {
          fprintf ( stderr, errmsg[8], PROGRAM, in );
          exit(8);
          }

/* now we go from SR to `out' system */

       if ( out == 2 )
          {
/* output data in SR coordinates */
          fv.b[0] = d_sr[0];
          fv.b[1] = d_sr[1];
          fv.b[2] = d_sr[2];
          }
          else if ( (out > 2) && (out < 8) )
          {
          if ( subbits(fv.stat,23,24) == 0 )
             phi = phisrp + ((double) fv.phavar);
             else
             {
/* bit for `averaged data' is set, so assume spin phase = 0.0 */
             phi = phisrp;
/* clear `averaged data' bit */
             fv.stat = bitcpy2(0,fv.stat,23,24);
             }
/* despin data, i.e., transform data from SR to SCS system
                         (rotate about Z-SR axis with angle -phi) */
          cp = cos(phi);
          sp = sin(phi);
          d_scs[0] = d_sr[0]*cp - d_sr[1]*sp;
          d_scs[1] = d_sr[1]*cp + d_sr[0]*sp;
          d_scs[2] = d_sr[2];
 
          if (out == 3)
             {
/* output data in SCS coordinates */
             fv.b[0] = d_scs[0];
             fv.b[1] = d_scs[1];
             fv.b[2] = d_scs[2];
             }
             else 
             {
             if ( abs(fv.tv.tv_sec - lsec) >= delta )
                {
/* recalculate transformation matrices */
                lsec = fv.tv.tv_sec;
                gei_gse_ ( &tmjd, &t2[0] );
/* for SCS -> GSE, we need the spin axis in GSE coordinates */
                s_gse[0] = t2[0]*s_gei[0] + t2[3]*s_gei[1] + t2[6]*s_gei[2];
                s_gse[1] = t2[1]*s_gei[0] + t2[4]*s_gei[1] + t2[7]*s_gei[2];
                s_gse[2] = t2[2]*s_gei[0] + t2[5]*s_gei[1] + t2[8]*s_gei[2];
                scs_gse_ ( &s_gse[0], &s_gse[1], &s_gse[2], &c[0] );
                if ((out > 4) && (out < 7))
                   {
/* for GSE -> GSM or SM, we need the dipole axis in GSE */
                   gei_geo_ ( &tmjd, &t1[0] );
                   m_gei[0] = t1[0]*m_geo[0] + t1[1]*m_geo[1] + t1[2]*m_geo[2];
                   m_gei[1] = t1[3]*m_geo[0] + t1[4]*m_geo[1] + t1[5]*m_geo[2];
                   m_gei[2] = t1[6]*m_geo[0] + t1[7]*m_geo[1] + t1[8]*m_geo[2];
                   m_gse[0] = t2[0]*m_gei[0] + t2[3]*m_gei[1] + t2[6]*m_gei[2];
                   m_gse[1] = t2[1]*m_gei[0] + t2[4]*m_gei[1] + t2[7]*m_gei[2];
                   m_gse[2] = t2[2]*m_gei[0] + t2[5]*m_gei[1] + t2[8]*m_gei[2];
                   gse_gsm_ ( &m_gse[0], &m_gse[1], &m_gse[2], &t3[0] );
                   if (out == 6)
                      gsm_sm_ ( &m_gse[0], &m_gse[1], &m_gse[2], &t4[0] );
                   }
/* note that the precession matrix p is only calculated
                   when a new entry is read from the attitude file */
                }
             d_gse[0] = c[0]*d_scs[0] + c[3]*d_scs[1] + c[6]*d_scs[2];
             d_gse[1] = c[1]*d_scs[0] + c[4]*d_scs[1] + c[7]*d_scs[2];
             d_gse[2] = c[2]*d_scs[0] + c[5]*d_scs[1] + c[8]*d_scs[2];
             
             if (out == 4)
                {
/* output data in GSE coordinates */
                fv.b[0] = d_gse[0];
                fv.b[1] = d_gse[1];
                fv.b[2] = d_gse[2];
                }
	       else if (out == 5)
	       {
/* output data in GSM coordinates */
                fv.b[0] = t3[0]*d_gse[0] + t3[3]*d_gse[1] + t3[6]*d_gse[2];
                fv.b[1] = t3[1]*d_gse[0] + t3[4]*d_gse[1] + t3[7]*d_gse[2];
                fv.b[2] = t3[2]*d_gse[0] + t3[5]*d_gse[1] + t3[8]*d_gse[2];
	       }
                else if (out == 6)
                {
                d_tmp[0] = t3[0]*d_gse[0] + t3[3]*d_gse[1] + t3[6]*d_gse[2];
                d_tmp[1] = t3[1]*d_gse[0] + t3[4]*d_gse[1] + t3[7]*d_gse[2];
                d_tmp[2] = t3[2]*d_gse[0] + t3[5]*d_gse[1] + t3[8]*d_gse[2];
/* output data in SM coordinates */
	       fv.b[0] = t4[0]*d_tmp[0] + t4[3]*d_tmp[1] + t4[6]*d_tmp[2];
	       fv.b[1] = t4[1]*d_tmp[0] + t4[4]*d_tmp[1] + t4[7]*d_tmp[2];
	       fv.b[2] = t4[2]*d_tmp[0] + t4[5]*d_tmp[1] + t4[8]*d_tmp[2];
	       }
	       else
	       {
	       d_tmp[0] = t2[0]*d_gse[0] + t2[1]*d_gse[1] + t2[2]*d_gse[2];
	       d_tmp[1] = t2[3]*d_gse[0] + t2[4]*d_gse[1] + t2[5]*d_gse[2];
	       d_tmp[2] = t2[6]*d_gse[0] + t2[7]*d_gse[1] + t2[8]*d_gse[2];
/* output data in GEI-of-J2000 coordinates */
	       fv.b[0] = p[0]*d_tmp[0] + p[1]*d_tmp[1] + p[2]*d_tmp[2];
	       fv.b[1] = p[3]*d_tmp[0] + p[4]*d_tmp[1] + p[5]*d_tmp[2];
	       fv.b[2] = p[6]*d_tmp[0] + p[7]*d_tmp[1] + p[8]*d_tmp[2];
	       }
	    }                             //end else if (out ==3 )
         }                                 //end else if ((out>2) && (out<8))
         else
         {
         fprintf ( stderr, errmsg[9], PROGRAM, sys[out] );
         exit(9);
         }

/* set status bits */    
      fv.stat = bitcpy2(out, fv.stat, 16, 19);
      }

/* write output structure */
    if ((idat = write_fgmtvec ( stdout, &fv )) < fvsz) 
       {
       fprintf ( stderr, errmsg[7], PROGRAM );
       exit(7);
       }
 }                                         //end while read fgmvec

/* quit */
 exit(0);
}
