/*
 * Copyright (C) 1994/95/96 IGM TU Braunschweig, 2000 MPE Garching
 * modified version 5.0 (1996-12-18),
 * V6.0 (2000-04-15): timesp <-> timespec_t, libutil.h & new status word
 * V6.1 (2000-12-15): bits 13-15 used now for rng_chng/cal_mode/eclipse
 *                    new error messages
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmls"  
#define VERSION "6.1 (2000-12-15)"
#define PURPOSE "extract informations from a series of FGM vectors"
#define USAGE "USAGE:

 ... | fgmls [-t <int>] | ...

 The program reads CLUSTER FGM vectors of the `fgmtvec_t' data structure
 from stdin and writes some fundamental informations in ASCII code to stdout."

#define OPTIONS "OPTIONS:

  -t     print the time information in the following form 
         <int> = 0 : ISO standard time string like `1996-12-02T02:05:15.398Z'
         <int> = 1 : (int) seconds of the UNIX epoch, and (int) nanoseconds
         <int> = 2 : (float) seconds of the hour
         <int> = 3 : (float) hours of the day
         <int> = 4 : character string like `Mon Dec 02 02:05:15 1996'

  -V        print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@geophys.nat.tu-bs.de)"

static char *errmsg[] =                           /* error messages */
  {
/*  0 */  "Type `%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Incorrect choice of time representation.\n",
  };


#include "libutil.h"


static void hyphenline()
/*-----------------------------------------------------------------------------
 * The function `hyphenline' prints a line of hyphens for clarity.            *
 * The first character is a `%' to indicate that this is a comment line.      *
 -----------------------------------------------------------------------------*/
{
 printf("%%------------------------------------------------------------------------------\n");
}


static void write_err(int err0, int err1)
/*-----------------------------------------------------------------------------
 * The function 'write_err' writes the error message, if the corresponding    *
 * bit is set in err1 and not set in err0. This avoids multiple writing of    *
 * the same error message. All vectors within a reset period should have the  *
 * same error bits.                                                           *
 -----------------------------------------------------------------------------*/
{
 int i;
 static char  *errtext[3] =
   {
/*  0     "POSSIBLY CORRUPT SCIENCE DATA",*/
/*  1     "INCORRECT NUMBERS OF VECTORS SAMPLED DURING LAST RESET PERIOD",*/
/*  2    "SUM CHECK OF CODE FAILURE" */
/*  0 */  "RANGE CHANGE ERROR",
/*  1 */  "CALIBRATION MODE"
/*  2 */  "(HALF/)ECLIPSE" 
   };

 for (i = 0; i < 3; i++)
    if (subbits(err1, i, i + 1) && !subbits(err0, i, i + 1))
       printf("X %2d>Error: %s\n", i, errtext[i]);
}


static void write_t(char id, int timeform, timesp t)
/*----------------------------------------------------------------------------- 
 * The function 'write_t' writes the time according to the choosen format.    *
 -----------------------------------------------------------------------------*/
{
 char  *time_ptr, time_str[25];

 switch(timeform)
   {
    case 0: 
       get_time_str(&t, time_str);
       printf("%c %s\n", id, time_str);
       break;
    case 1: 
       printf("%c %9d %9d\n", id, (int)t.tv_sec, t.tv_nsec);
       break;
    case 2: 
       printf("%c % f\n", id, (t.tv_sec % 3600) + 1.e-9 * t.tv_nsec);
       break;
    case 3: 
       printf("%c % f\n", id, ((t.tv_sec % 86400) + 1.e-9 * t.tv_nsec) / 3600.);
       break;
    case 4: 
       time_ptr = asctime(gmtime(&t.tv_sec));
       printf("%c %s", id, time_ptr);
       break;
   }
}


static void do_error_event(int *flag, 
                           int *write_time_flag, 
                           int timeform, 
                           timesp t)
/*----------------------------------------------------------------------------- 
 * The function 'do_error_event' is called, if an error bit is set or a       *
 * parameter has changed. At first it sets the indicating flag. Then it       *
 * writes the time and sets the write_time_flag, if it has not been set       *
 * before. The write_time_flag is used to avoid multiple time writing.        *
 * Thus, this time indicates the end of a data block with no errors and the   *
 * same parameter settings.                                                   *
 -----------------------------------------------------------------------------*/
{
 *flag = 1;
 if (!(*write_time_flag))
   {
    *write_time_flag = 1;
    write_t('E', timeform, t);
    hyphenline();
   }
}


int main (int argc, char *argv[])
{
 fgmtvec_t          vec;
 static timesp  t, time_diff; 
 static int         i = 1, timeform,
                    sc[2], adc[2], sensor[2], scnd[2],
                    sd_filt[2], MSA_filt[2], 
                    OB_cal[2], IB_cal[2],
                    range[2], option[2],
                    error[2], 
                    declevel, missing, 
                    error_flag, event_flag, write_time_flag = 1;
 static double      vector_period, sample_period, sampfreq, fgmfrq=201.7462;
 FILE               *inf=stdin;
 char               copt;
// char               inp[]="./fgm.dat";    only to debug

//
/* only to debug !!
 if ( (inf = fopen(inp, "rb"))==NULL){
    fprintf(stderr,"file %s not found\n",inp);
    return(-1);
    }*/
  
/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 while ( i < argc )
   {
    if (*argv[i] == '-')                           /* Is it a program option? */
      {
       copt = *(argv[i]+1);
       switch(copt)
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n",
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
                      exit(0);

          case 't': 
             if (*(argv[i]+2) == '\0')
               {
                i++;
                timeform = atoi(argv[i]);
               }
             else
                timeform = atoi(argv[i]+2);
             if (timeform < 0 || timeform > 4)
               {
                fprintf ( stderr, errmsg[2], PROGRAM );
                exit(2);
               }
             break;
             
          case 'V': 
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);
             
          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    i++;
   }

/*----------------------------------------------------------------------------- 
 * Reading the input vectors until end of vector stream.                      *
 -----------------------------------------------------------------------------*/
 while (read_fgmtvec(inf, &vec) == sizeof(fgmtvec_t))
   {

/*----------------------------------------------------------------------------- 
 * Determine the error code.                                                  * 
 -----------------------------------------------------------------------------*/
    error[1] = subbits(vec.stat, 13, 16);
/*----------------------------------------------------------------------------- 
 * Continue to read input vectors at the beginning until vector has no error. * 
 -----------------------------------------------------------------------------*/
    if (t.tv_sec == 0 && error[1])
       continue;
/*----------------------------------------------------------------------------- 
 * Do the error handling.                                                     * 
 -----------------------------------------------------------------------------*/
    if (error[1])
      {
       do_error_event(&error_flag, &write_time_flag, timeform, t);
       write_err(error[0], error[1]);
      }
/*----------------------------------------------------------------------------- 
 * Set the event flag, if this is the first vector without error after a      *
 * faulty vector.                                                             *
 -----------------------------------------------------------------------------*/
    if (!error[1] && error[0])
       event_flag = 1;

/*----------------------------------------------------------------------------- 
 * Determine the S/C number.                                                  * 
 -----------------------------------------------------------------------------*/
    sc[1] = subbits(vec.stat, 30, 32) + 1;
    if (sc[1] != sc[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       printf ("S %2d>S/C number %d\n", sc[1], sc[1]);
      }

/*----------------------------------------------------------------------------- 
 * Determine the sensor.                                                      * 
 -----------------------------------------------------------------------------*/
    sensor[1] = subbits(vec.stat, 7, 8);
    if (sensor[1] != sensor[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!sensor[1])
          printf ("U  1>outboard magnetometer unit\n");
       else
          printf ("U  2>inboard magnetometer unit\n");
      }
      
/*----------------------------------------------------------------------------- 
 * Determine whether primary or secondary sensor.                             *
 -----------------------------------------------------------------------------*/
    scnd[1] = subbits(vec.stat, 28, 30);
    if (scnd[1] != scnd[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!scnd[1])
          printf ("D  1>primary sensor\n");
       else
          printf ("D  2>secondary sensor\n");
      }

/*----------------------------------------------------------------------------- 
 * Determine the used ADC.                                                    * 
 -----------------------------------------------------------------------------*/
    adc[1] = subbits(vec.stat, 8, 9) + 1;
    if (adc[1] != adc[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       printf ("A %2d>ADC number %d active\n", adc[1], adc[1]);
      }

/*----------------------------------------------------------------------------- 
 * Determine the telemetry option.                                            * 
 -----------------------------------------------------------------------------*/
    option[1] = subbits(vec.stat, 0, 4);
    if (option[1] != option[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       printf ("O %2d>FGM telemetry option %X\n", option[1], option[1]);
      }

/*----------------------------------------------------------------------------- 
 * Determine the range of the sensor.                                         * 
 -----------------------------------------------------------------------------*/
    range[1] = subbits(vec.stat, 4, 7);
    if (range[1] != range[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       printf ("R %2d>FGM sensor range %d\n", range[1], range[1]);
      }

/*----------------------------------------------------------------------------- 
 * Determine the science data filtering mode.                                 * 
 -----------------------------------------------------------------------------*/
    sd_filt[1] = subbits(vec.stat, 12, 13);
    if (sd_filt[1] != sd_filt[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!sd_filt[1])
          printf ("F  0>science filtering off\n");
       else
          printf ("F  1>science filtering on\n");
      }

/*----------------------------------------------------------------------------- 
 * Determine the MSA data filtering mode.                                     * 
 -----------------------------------------------------------------------------*/
    MSA_filt[1] = subbits(vec.stat, 11, 12);
    if (MSA_filt[1] != MSA_filt[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!MSA_filt[1])
          printf ("M  0>MSA filtering off\n");
       else
          printf ("M  1>MSA filtering on\n");
      }
      
/*----------------------------------------------------------------------------- 
 * Determine the outboard sensor calibration mode.                            * 
 -----------------------------------------------------------------------------*/
    OB_cal[1] = subbits(vec.stat, 10, 11);
    if (OB_cal[1] != OB_cal[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!OB_cal[1])
          printf ("C  0>outboard calibration off\n");
       else
          printf ("C  1>outboard calibration on\n");
      }
      
/*----------------------------------------------------------------------------- 
 * Determine the inboard sensor calibration mode.                             * 
 -----------------------------------------------------------------------------*/
    IB_cal[1] = subbits(vec.stat, 9, 10);
    if (IB_cal[1] != IB_cal[0] || t.tv_sec == 0)
      {
       do_error_event(&event_flag, &write_time_flag, timeform, t);
       if (!IB_cal[1])
          printf ("K  0>inboard calibration off\n");
       else
          printf ("K  1>inboard calibration on\n");
      }

/*----------------------------------------------------------------------------- 
 * Test if there is a data gap.                                               * 
 -----------------------------------------------------------------------------*/
    if (t.tv_sec != 0)
      {
       fgmrate(scnd[1], option[1], &sampfreq, &declevel, fgmfrq);
       sample_period = 1.0 / sampfreq;
       time_diff = subtime(&vec.tv, &t);
       vector_period = time_diff.tv_sec + 1.e-9 * time_diff.tv_nsec;
       if (vector_period >= 1.5 * sample_period)
         {
          missing = (int)(vector_period / sample_period + 0.5);
          do_error_event(&event_flag, &write_time_flag, timeform, t);
          printf ("G %2d>Data gap: %f sec, sampling period: %f sec\n", missing, vector_period, sample_period);
         }
      }

/*----------------------------------------------------------------------------- 
 * Write the vector time and reset the write_time_flag, if the event flag is  *
 * set and the vector has no error bits set, ie if one ore more parameters    *
 * have changed or a period of faulty vectors has finished.                   *
 * Thus, this time indicates the start of a data block with no errors and the *
 * same parameter settings.                                                   *
 -----------------------------------------------------------------------------*/
    if (event_flag && !error_flag)
      {
       hyphenline();
       write_t('B', timeform, vec.tv);
       write_time_flag = 0;
      }

/*----------------------------------------------------------------------------- 
 * Set the old parameters to the new ones. Reset the event and the error flag.*
 -----------------------------------------------------------------------------*/
    error[0] = error[1];
    sc[0] = sc[1];
    adc[0] = adc[1];
    scnd[0] = scnd[1];
    sensor[0] = sensor[1];
    option[0] = option[1];
    range[0] = range[1];
    sd_filt[0] = sd_filt[1];
    MSA_filt[0] = MSA_filt[1];
    OB_cal[0] = OB_cal[1];
    IB_cal[0] = IB_cal[1];
    t = vec.tv;
    event_flag = 0;
    error_flag = 0;
   }

/*----------------------------------------------------------------------------- 
 * Write the last time as the end time of the last data block.                *
 -----------------------------------------------------------------------------*/
 write_t('E', timeform, t);
 hyphenline();
 exit(0);
}
