/* Alpha-Dec & Linux
 * Copyright (C) 1994/95/96 TU Braunschweig, 2000 MPE Garching 
 * get_fgmconstants
 * check_bad
 * "-l <cal>" added to transmit calfilename info to output
 * modified V6.0 (2000-04-15) for consistency with igmvec
 * changed options: '-d' -> '-f', '-c' ->'-v', '-l' -> '-i' 
 * -o, but no name => C#_yyyymmdd_sys.mag !! datestr different AlphaDEC<->SUN
 * modified V6.1 (2000-05-25) to use new (libutil.c) function: mark_vector
 * modified V6.2 (2000-12-15) to write variances with enough decimal places
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "fgmvec"
#define VERSION "6.3 (2001-01-03)"
#define PURPOSE "writing ASCII output of FGM vectors to stdout"
#define USAGE "USAGE:

 ... | fgmvec [-t <int>] [-a] [-r ] [-m] [-p] [-v] [-f] 
                                              [-i [<cal>]] [-o [<out>]]

 The program reads CLUSTER FGM data (in fgmtvec_t format) from stdin and
 writes the magnetic field vectors as an ASCII listing to stdout.
 Each vector consists of a time string and the three vector components.

 If option -r is used, the number of decimal places of each component will
 depend on the range of the magnetometer, i.e. on the data resolution
 currently used."

#define OPTIONS "OPTIONS:

  -t     print the time information in the following form:
         <int> = 0 : ISO standard time string like '2000-12-02T02:05:15.798Z'
         <int> = 1 : seconds and nanoseconds of the UNIX epoch
         <int> = 2 : (float) seconds of the hour
         <int> = 3 : (float) hours of the day
         <int> = 4 : character string like 'Mon Dec  2 02:05:15 2000'
         Default is the ISO time string.

  -r     choose the number of decimal places according to the data resolution.
         Default is to use a fixed format with a resolution of 0.1 nT.

  -m     add the magnitude to each output vector.

  -p     add the phase (for high-resolution data) or
         the normalized variance of the total magnetic field
         (for averaged data) to each output vector.

  -v     add the raw data variance (for high-resolution data) or 
         the normalized variance of the magnetic field magnitude
         (for averaged data) to each output vector.

  -f     add the data acquisition frequency to each output vector.
 
  -a     output all vectors of the fgmtvec_t stream, even if marked 'BAD'. 
         Adds TM mode, range, sensor, and data flags to each output record.
         A non-zero value in the last column means 'BAD' vector, 0 else.
         Default is to output only non-marked data.
 
  -i     add calibration filename info to output.
         use <cal> as calfile-name logfile. If '-i', but no name given,
         default name is 'cal.log'. Column name info is appended too. 
         
  -o     use <out> as output file. If '-o', but no <out>,
         default name is : C#_yyyymmdd_sys.mag, where #=1,2,3,4 stands for
         satellite number and sys for the output coordinate system. 
         Default output is stdout.

  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu  (eg@mpe.mpg.de)
 Stephan Buchert  (scb@geophys.nat.tu-bs.de)
 Reinhold Kempen  (reinhold@geophys.nat.tu-bs.de)
 Joerg Warnecke   (joerg@geophys.nat.tu-bs.de)"

#include <math.h>
#include "libutil.h"


static char *errmsg[] =
  {
/*  0 */  "Use '%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Invalid choice of time representation (%d).\n",
/*  3 */  "ERROR in %s: Could not open output file %s.\n",
/*  4 */  "ERROR in %s: Illegal usage.\n"
/*  5 */  "ERROR in %s: Invalid telemetry option detected at %s.\n",
  };


int main (int argc, char **argv)
{
 FILE          *outfile=stdout, *callogf;
 fgmtvec_t      fv;
 int            i, ifv, timeform=0, dores=0, domagnitude=0, dophase=0,
                css=0, domagvar=0, dorate=0, all=0, range, option, doinfo=0,
                fvsz=0, isat=0, out=0, avg=0;
 static int     rate[16]={0, 0, 16, 18, 22, 0, 0, 0, 0, 0, 16, 18, 22, 67, 0, 0};
 char           copt, call[101]="", calnm[101]="", time_str[25], 
                outname[128]="", *time_ptr, *datest="",tailstr[120]="";
 static double  magnitude;
 static char    *csstr[8]={"FS","FSR","SR","SCS","GSE","GSM","SM","GEI"};

/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 't': 
             if (*(argv[i]+2) == '\0' && i+1 <= argc )
               {
                i++;
                timeform = atoi(argv[i]);
               }
             else
                timeform = atoi(argv[i]+2);
             if (timeform < 0 || timeform > 4)
               {
                fprintf ( stderr, errmsg[2], PROGRAM, timeform );
                fprintf ( stderr, errmsg[0], argv[0] );
                exit(2);
               }
             break;

          case 'r':
             dores = 1;
             break;

          case 'm':
             domagnitude = 1;
             break;

          case 'p':
             dophase = 1;
             break;

          case 'v':
             domagvar = 1;
             break;

          case 'f':
             dorate = 1;
             break;

          case 'a':
             all = 1;
             break;

          case 'i':
             doinfo = 1;
             if ( *(argv[i]+2) == '\0' && i+1 <= argc )
                {
                if ( i+1 < argc && *argv[i+1] != '-' )
                  {
                  i++;
                  strcpy (call, argv[i]);
                  }
                }
                else
                strcpy (call, (argv[i]+2));
             if (strlen(call) < 2)
                strcpy(call,"cal.log");
             break;

          case 'o':
             out=1;
             if ( *(argv[i]+2) == '\0' && i+1 <= argc )
               {
                if ( i+1 < argc && *argv[i+1] != '-' )
                  {
                  i++;
                  strcpy ( outname, argv[i] );
                  }
               }
               else
               strcpy ( outname, (argv[i]+2) );
             break;
             
           case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[4], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(4);
      }
    i++;
   }

/* read first record to define s/c & coordinate system & high-res or avg */
 fvsz = sizeof ( fgmtvec_t );
 if ( (ifv = read_fgmtvec(stdin,&fv) ) == fvsz ){
    css = subbits(fv.stat,16,19);
    avg = get_bit(fv.stat,23);
    isat = 1 + subbits ( fv.stat, 30, 32 );   
    datest=datestr(&fv.tv.tv_sec);
    }

/* define default outname : if '-o' and no <outf> given & open outfile*/
 if (out){
    if (strlen(outname) < 2){
       sprintf(outname,"C%1d_%8s_",isat,datest);
       strcat(outname,csstr[css]);
       strcat(outname,".mag");
       }
     outfile = fopen (outname, "w" );
     if ( outfile == NULL )
        {
        fprintf ( stderr, errmsg[3], PROGRAM, outname );
        exit(3);
        }
        else
        fprintf(stderr,"OUTPUT to: %s\n",outname);
     }     
  
/*---------------------------------------------------------------------------- 
 * Reading the data, and writing the ASCII output to the output file.         *
 -----------------------------------------------------------------------------*/
 while ( ifv == fvsz )
   {
   if ( all || !check_bad(fv.stat) )
    { 
    switch ( timeform )
      {
       case 0: 
          get_time_str ( &fv.tv, time_str );
          fprintf ( outfile, "%s ", time_str );
          sprintf (tailstr,"# Date          Time    ");
          break;
       case 1: 
          fprintf ( outfile, "%10d.%09d ", (int) fv.tv.tv_sec, fv.tv.tv_nsec );
           sprintf (tailstr,"# sec_since 1970   ");
         break;
       case 2:
          fprintf ( outfile, "% f ", 
                    (fv.tv.tv_sec%3600) + 1.e-9*fv.tv.tv_nsec );
          sprintf (tailstr,"#sec_of_hour");
          break;
       case 3: 
          fprintf ( outfile, "%12.8f ",((double)(fv.tv.tv_sec%86400) + 
                    1.e-9*(double)fv.tv.tv_nsec) / 3600.0 );
          sprintf (tailstr,"#hours_of_day");
          break;
       case 4: 
          time_ptr = asctime ( gmtime(&fv.tv.tv_sec) );
          *(time_ptr + 24) = ' ';
          fprintf ( outfile, "%s", time_ptr );
          sprintf (tailstr,"# Date          Time    ");
          break;
      }
    strcat (tailstr,"    Bx/nT    By/nT    Bz/nT");
    if ( domagnitude ){
       magnitude = sqrt ( pow ( fv.b[0], 2 ) + 
                          pow ( fv.b[1], 2 ) +
                          pow ( fv.b[2], 2 ) );
       strcat (tailstr,"     B/nT");
       }
       
    if ( dores )
      {
       range = subbits ( fv.stat, 4, 7 );
       switch ( range )
         {
          case 0:
          case 1:
          case 2:
             fprintf ( outfile, "% 8.4f % 8.4f % 8.4f ", 
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.4f ", magnitude );
             break;
          case 3:
             fprintf ( outfile, "% 8.3f % 8.3f % 8.3f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.3f ", magnitude );
             break;
          case 4:
             fprintf ( outfile, "% 8.2f % 8.2f % 8.2f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.2f ", magnitude );
             break;
          case 5: 
             fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.1f ", magnitude );
             break;
          default:
             fprintf ( outfile, "% 8.0f % 8.0f % 8.0f ",
                      fv.b[0], fv.b[1], fv.b[2] );
             if (domagnitude)
                fprintf ( outfile, "% 8.0f ", magnitude );
             break;
         }
      }
    else
      {
       fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ", fv.b[0], fv.b[1], fv.b[2] );
       if ( domagnitude )
          fprintf ( outfile, "% 8.1f ", magnitude );
      }

    if ( dophase ){
       if (avg){
         fprintf ( outfile, " %.3e", fv.phavar );
         strcat (tailstr,"   total_var");
         }
         else{ 
         fprintf ( outfile, "%8.4f", fv.phavar );
         strcat (tailstr,"    phase");
         }      
       }
       
    if ( domagvar ){
       if (avg){
         fprintf ( outfile, " %.3e", fv.magvar );
         strcat (tailstr,"  magn_var");
         }
         else{ 
       fprintf ( outfile, "%10.3f ", fv.magvar );
       strcat (tailstr,"    raw_var");
         }      
       }
    option = subbits(fv.stat, 0, 4);
    if ( dorate )
      {
        if ( rate[option] ){
          fprintf ( outfile, "%5d ", rate[option] );
          strcat (tailstr,"  f/Hz");
          }
       else
         {
          get_time_str ( &fv.tv, time_str );
          fprintf ( stderr, errmsg[5], PROGRAM, time_str );
          exit(5);
         }
      }

    if (all)
      {
      fprintf ( outfile, "%4X%4d%4d%4d",option,subbits(fv.stat,4,7),
                get_bit(fv.stat,7),mark_vector(0,&fv));
      strcat(tailstr,"   TM  rng OB  BAD");
      }
    fprintf ( outfile, "\n" );
    }
    ifv = read_fgmtvec(stdin,&fv);
   } //while
   strcat(tailstr,"  coord.-> ");
   strcat(tailstr,csstr[css]);
   strcat(tailstr,"\n");
   if (doinfo){
     fprintf ( outfile,"%s",tailstr );
     if(all)
       fprintf ( outfile," TM=telemetry mode, rng=range, OB=0,1(onboard/inboard), BAD: 1,2,4 (rng/cal/ecl)\n");
     if (strlen(call) >= 1){
        callogf = fopen ( call, "r" );
        if ( callogf == NULL )
           fprintf (stderr, errmsg[3], PROGRAM, call);
           else
           {
           fgets(calnm, 100, callogf) ;
         fclose (callogf);     
         if (strlen(calnm) > 2)
            fprintf ( outfile,"#used calfile: %s",calnm );
         fflush ( outfile );
         }
      }   
   }     
 exit(0);
}
