/*
 * Alpha-Dec & SUN
 */

#define PROJECT "CLUSTER FGM DATA PROCESSING"
#define PROGRAM "igmvec"
#define VERSION "1.0 (2000-05-25)"
#define PURPOSE "writing ASCII output of IGM structure to stdout"
#define USAGE "USAGE:

 ... | igmvec [-t <int>] [-e] [-s] [-d [<int>]] [-m] [-v] [-f]
              [-i <cal>] [-o <outfile>]
 
 The program reads CLUSTER FGM data extended with position information
 (in fgmtrec format) from stdin and writes the time stamped magnetic field 
 vectors and the satellite positon vectors in cartesian coordinates as an ASCII
 listing to stdout. With option '-s' the output coordinate system is spherical"
  
#define OPTIONS "OPTIONS:

  -t     print the time information in the following form:
         <int> = 0 : ISO standard time string like '2000-12-02T02:05:15.798Z'
         <int> = 1 : (double) seconds of the day
         <int> = 2 : (double) hours of the day
         Default is the ISO time string.

  -e     position vector in Earth radii [Re=6317.2 km]. Default is [km].
   
  -s     spherical coordinate system output. Angles are in [deg].
         The precision in position is 0.1 km.
 
  -d     precision for the magnetic field output. 
         <int> is the number of decimal places. Possible values: 1,2,3 
         If '-d' but no <int> given, default is 1, i.e. 0.1 nT precision.
         Default is to use a format according to the data resolution.
 
  -m     add the magnitude to each output vector.

  -v     add the raw data variance (for high-resolution data) or 
         the normalized variance of the magnetic field magnitude
         (for averaged data) to each output vector.

  -f     add the data acquisition frequency to each output vector.
 
  -i     append calibration filename info & output column names to output.
         
  -o     use <outfile> as output file. If '-o', but no <outfile> given
         default name is : C#_yyyymmdd_sys.txt, where #=1,2,3,4 stands for
         satellite number and sys for the output coordinate system.
         Default is to write to stdout.

  -V     print the version number on stdout, then exit."

#define AUTHOR "AUTHORS:

 Edita Georgescu      (eg@mpe.mpg.de)
 Karl-Heinz Fornacon  (k-h.fornacon@tu-bs.de)"

#include <math.h>
#include "libutil.h"

/*
 * FGM vector with id-word and position
 */
typedef struct fgmtrec 
  {
   int               id;       /* identification word */
   int               stat;     /* satellite status word */
   timesp            tv;       /* POSIX time */
   float             b[3];     /* magnetic field in nT */
   float             phavar;   /* spin phase, or total variance */
   float             magvar;   /* variance in magnitude */
   float             r[3];     /* position in km */
  } 
fgmtrec_t;


static char *errmsg[] =
  {
/*  0 */  "Use '%s -h' for help.\n",
/*  1 */  "ERROR in %s: Illegal option %s.\n",
/*  2 */  "ERROR in %s: Could not open output file %s.\n",
/*  3 */  "ERROR in %s: Illegal usage.\n"
/*  4 */  "ERROR in %s: Invalid telemetry option detected at %s.\n",
  };

void car2sph( float *r, float *theta, float *phi) 
/****************************************************************************
 * transforms from cartesian to spherical coordinate system                 *
 ****************************************************************************/
 {
 double x,y,z,rr,xy,the,ph,radeg=57.295779513;
 
 x=(double) *r;
 y=(double) *theta;
 z=(double) *phi;
 xy= pow ( x, 2 )+ pow ( y, 2 );
 rr= sqrt(xy + pow ( z, 2 ));
 if (xy > 0) {
    ph=atan2(y,x);
    the=atan2(sqrt(xy),z);
    if (ph < 0) ph += 6.283185307;
    }
    else
    {
    ph=0;
    if (z < 0)
       the=3.141592654;
       else
       the=0;
    }
 *r=(float) rr;   
 *theta=(float) (the*radeg);
 *phi=(float) (ph*radeg);
 }
 

int main (int argc, char **argv)
{
 FILE           *outfile=stdout, *inf=stdin, *callogf;
 fgmtrec_t      fv;
 int            timeform=0, domagnitude=0, dorearth=0, doinfo=0, css=0,
                isat=0, i, domagvar=0, dorate=0, prec=0, out=0,
                dosph=0, range, option,  fvsz;
 static int     ifv=0 ,rate[16]={0, 0, 16, 18, 22, 0, 0, 0, 0, 0, 
                16, 18, 22, 67, 0, 0};
 char           copt, call[101]="", calnm[101]="", time_str[25],*datest="", 
                outname[100]="",tailstr[120]="";
 static double  magnitude;
 float          pos[3], RE=6371.2;
 static char    *csstr[8]={"FS","FSR","SR","SCS","GSE","GSM","SM","GEI"};

/* only to debug !!, inp[10]="out.igm"
 if ( (inf = fopen(inp, "rb"))==NULL){
    fprintf(stderr,"file %s not found\n",inp);
    return(-1);
    }*/

/*----------------------------------------------------------------------------- 
 * Evaluation of the command line parameters.                                 *
 -----------------------------------------------------------------------------*/
 i = 1;
 while ( i < argc )
   {
    copt = *(argv[i]);
    if ( copt == '-' )
      {
       copt = *(argv[i]+1);
       switch ( copt )
         {
          case 'h':
#ifndef SECURE
             fprintf ( stdout, "\n%s  --  %s\n\n%s\n\n%s\n\n%s\n", \
                       PROGRAM, PURPOSE, USAGE, OPTIONS, AUTHOR );
#else
             fprintf ( stdout, "%s\n\n %s %s\n\n%s\n",\
                       PROJECT, PROGRAM, VERSION, AUTHOR );
#endif
             exit(0);

          case 't': 
             if (*(argv[i]+2) == '\0' && i+1 <= argc )
               {
                if (*argv[i+1] != '-' ){
                   i++;
                   timeform = atoi(argv[i]);
                   }
               }
             else
                timeform = atoi(argv[i]+2);
             if (timeform < 0 || timeform > 2)
                timeform=0; 
             break;

          case 'd':
             if ( *(argv[i]+2) == '\0' && i+1 < argc )
                {
                if (*argv[i+1] != '-' ){
                   i++;
                   prec = atoi(argv[i]);
                   }
                }   
                else
                   prec = atoi(argv[i]);
             if (prec < 1 || prec > 3)
                prec=1;
             break;

          case 'e':
             dorearth = 1;
             break;

          case 's':
             dosph = 1;
             break;

          case 'm':
             domagnitude = 1;
             break;

         case 'v':
             domagvar = 1;
             break;

          case 'f':
             dorate = 1;
             break;

          case 'i':
             doinfo=1;
             if ( *(argv[i]+2) == '\0' && i+1 <= argc )
                {
                if ( i+1 < argc && *argv[i+1] != '-' )
                  {
                  i++;
                  strcpy (call, argv[i]);
                  }
                }
                else
                strcpy (call, (argv[i]+2));
             if (strlen(call) < 2)
                strcpy(call,"cal.log");
             break;

          case 'o':
             out=1;
             if ( *(argv[i]+2) == '\0' && i+1 <= argc )
                {
                if ( i+1 < argc && *argv[i+1] != '-' )
                  {
                  i++;
                  strcpy ( outname, argv[i] );
                  }
                }
                else
                strcpy ( outname, (argv[i]+2) );
             break;
             
           case 'V':
             fprintf ( stdout, "%s %s\n", PROGRAM, VERSION );
             exit(0);

          default:
             fprintf ( stderr, errmsg[1], PROGRAM, argv[i] );
             fprintf ( stderr, errmsg[0], argv[0] );
             exit(1);
         }
      }
    else
      {
       fprintf ( stderr, errmsg[4], PROGRAM );
       fprintf ( stderr, errmsg[0], argv[0] );
       exit(3);
      }
    i++;
   }

/* read first record to define s/c & coordinate system */
 fvsz = sizeof ( fgmtrec_t );
 if ( (ifv=fread ( &fv, 1, fvsz, inf ))  == fvsz ){
    css = subbits(fv.stat,16,19);
    isat = 1 + subbits ( fv.stat, 30, 32 );   
    datest=datestr(&fv.tv.tv_sec);
    }

/* define default outname : if '-o' and no <outf> given & open outfile*/
 if (out){ 
    if (strlen(outname) < 2){
       sprintf(outname,"C%1d_%8s_",isat,datest);
       strcat(outname,csstr[css]);
       strcat(outname,".txt");
       }
     outfile = fopen (outname, "w" );
     if ( outfile == NULL )
        {
        fprintf ( stderr, errmsg[2], PROGRAM, outname );
        exit(2);
        }
        else
        fprintf(stderr,"OUTPUT to: %s\n",outname);  
     } 
/*----------------------------------------------------------------------------- 
 * Reading the data, and writing the ASCII listing to output                  *
 -----------------------------------------------------------------------------*/
 while ( ifv == fvsz ) {
    switch ( timeform )
      {
       case 0: 
          get_time_str ( &fv.tv, time_str );
          fprintf ( outfile, "%s ", time_str );
          sprintf (tailstr,"  Date          Time    ");
          break;
       case 1: 
          fprintf ( outfile, "%12.5f ",(double)(fv.tv.tv_sec%86400) + 
                     + 1.e-9*(double)fv.tv.tv_nsec );
          sprintf (tailstr,"  sec_of_day");
          break;
       case 2: 
          fprintf ( outfile, "%12.8f ",((double)(fv.tv.tv_sec%86400) + 
                    1.e-9*(double)fv.tv.tv_nsec) / 3600.0 );
          sprintf (tailstr,"hours_of_day");
          break;
      }
    strcat (tailstr,"    Bx/nT    By/nT    Bz/nT");

/* transform to spherical coordinates */
    if ( domagnitude ){
       magnitude = sqrt ( pow ( fv.b[0], 2 ) + 
                          pow ( fv.b[1], 2 ) +
                          pow ( fv.b[2], 2 ) );
       strcat (tailstr,"     B/nT");
       }
// to be changed according to help page !!!!
   if ( !prec )
      {
       range = subbits ( fv.stat, 4, 7 );
       switch ( range )
         {
          case 0:
          case 1:
          case 2:
             fprintf ( outfile, "% 8.4f % 8.4f % 8.4f ", 
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.4f ", magnitude );
             break;
          case 3:
             fprintf ( outfile, "% 8.3f % 8.3f % 8.3f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.3f ", magnitude );
             break;
          case 4:
             fprintf ( outfile, "% 8.2f % 8.2f % 8.2f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.2f ", magnitude );
             break;
          case 5: 
             fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.1f ", magnitude );
             break;
          default:
             fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ",
                      fv.b[0], fv.b[1], fv.b[2] );
             if (domagnitude)
               fprintf ( outfile, "% 8.0f ", magnitude );
             break;
         }
      }
    else
      { switch ( prec )
        {
        case 0:
        case 1:
             fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ", 
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
               fprintf ( outfile, "% 8.1f ", magnitude );
             break;
        case 2:
             fprintf ( outfile, "% 8.2f % 8.2f % 8.2f ", 
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
               fprintf ( outfile, "% 8.2f ", magnitude );
        break;
        case 3:
             fprintf ( outfile, "% 8.3f % 8.3f % 8.3f ",
                       fv.b[0], fv.b[1], fv.b[2] );
             if ( domagnitude )
                fprintf ( outfile, "% 8.3f ", magnitude );
             break;
          
          default:
             fprintf ( outfile, "% 8.1f % 8.1f % 8.1f ",
                      fv.b[0], fv.b[1], fv.b[2] );
             if (domagnitude)
               fprintf ( outfile, "% 8.0f ", magnitude );
             break;
         }
      }

    if ( domagvar ){
       fprintf ( outfile, "% 8.2f ", fv.magvar );
       strcat (tailstr,"    var ");
       }
    option = subbits(fv.stat, 0, 4);
    if ( dorate )
      {
        if ( rate[option] ){
          fprintf ( outfile, "%5d ", rate[option] );
          strcat (tailstr,"  f/Hz");
          }
       else
         {
          get_time_str ( &fv.tv, time_str );
          fprintf ( stderr, errmsg[5], PROGRAM, time_str );
          exit(4);
         }
      }

    if ( dorearth ){
       for (i=0;i<3;i++) pos[i] = fv.r[i]/RE;
         if (!dosph){
          fprintf ( outfile, "% 9.5f % 9.5f % 9.5f ",
                      pos[0], pos[1], pos[2] );
          strcat(tailstr,"     x/Re      y/Re      z/Re");
          }
       }    
       else{
       for (i=0;i<3;i++) pos[i] = fv.r[i];
       if (!dosph){
          fprintf ( outfile, "% 9.1f % 9.1f % 9.1f ",
                      pos[0], pos[1], pos[2] );
          strcat(tailstr,"     x/km      y/km      z/km");
          }            
       }              

     if ( dosph ){
       car2sph(  &pos[0], &pos[1], &pos[2] );
       if( dorearth){
          strcat(tailstr,"   R/Re     LAT/deg  LON/deg"); 
          fprintf ( outfile, "% 9.5f % 8.2f % 8.2f ",
                      pos[0], pos[1], pos[2] );
          }
          else{                   
          strcat(tailstr,"   R/km     LAT/deg  LON/deg"); 
          fprintf ( outfile, "% 9.1f % 8.2f % 8.2f ",
                      pos[0], pos[1], pos[2] );
          }            
       }
    fprintf ( outfile, "\n" );
    ifv = fread ( &fv, 1, fvsz, inf ); 
   } 
   strcat(tailstr," ->");
   strcat(tailstr,csstr[css]);
   strcat(tailstr,"\n");
   if (doinfo){
     fprintf ( outfile,"%s",tailstr );
     if (strlen(call) >= 1){
     callogf = fopen ( call, "r" );
     if ( callogf == NULL )
        fprintf (stderr, errmsg[3], PROGRAM, call);
        else
        {
        fgets(calnm, 100, callogf) ;
        fclose (callogf);
        if (strlen(calnm) > 2)
           fprintf ( outfile,"used calibration file: %s",calnm );
        fflush ( outfile );
        }
     }
   }
  
 exit(0);
}
