/* LIBRARY module includes constants, user defined types & function prototypes
 * struct + bitutil + utility , modified by eg
 * Version 1.0, 1999-12-17
 * Copyright (C) 1999 MPE Garching
 * Author : Edita Georgescu (eg@mpe.mpg.de)

MODIFICATIONS:
- rewritten inctime, timestr2unix introduced NANOSEC   17/12/99
- timesp instead timespec_t only for Alpha-DEC:
  defined as time_t + int (NOT long as bs pretends to be in time.h
- fgmfreq() -> out
- in fgmrate added arg: double fgmfrq
- get_fgmconstants(int scid, double *hfcrate,double *fgmfrq,double *alpha,
                                int *obdelay,int *ibdelay,double *spinph0)
- int get_bit(int status,n0)
- void set_bit(int *status,n0)
- int check_bad(int status)
- char *datestr(time_t)
- int read_ddsheadc(headbytes indata, dds_head_t *objp)   17/5/00 - V1.1
- new const.fgm file format,                    changed:  15/9/00 - V1.2
- added function: mark_vector,                  changed: 15/12/00 - V1.3
*/ 

/*============================================================================*/

/* struct.c -- functions to read/write structs and files for the FGM data processing 
 *
 * modified by Edita Georgescu (eg@mpe.mpg.de), Version 6.0, 2000-03-15 
 * Version 4.0, 1996-01-24
 * written by Stephan Buchert (scb@geophys.nat.tu-bs.de)
 *            Reinhold Kempen (reinhold@geophys.nat.tu-bs.de)
 *            Joerg Warnecke  (joerg@geophys.nat.tu-bs.de)
 -----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/param.h>

#include "libutil.h"
static int mask[]={0x00000000, 0x00000001, 0x00000003, 0x00000007,
                   0x0000000f, 0x0000001f, 0x0000003f, 0x0000007f,
                   0x000000ff, 0x000001ff, 0x000003ff, 0x000007ff,
                   0x00000fff, 0x00001fff, 0x00003fff, 0x00007fff,
                   0x0000ffff, 0x0001ffff, 0x0003ffff, 0x0007ffff,
                   0x000fffff, 0x001fffff, 0x003fffff, 0x007fffff,
                   0x00ffffff, 0x01ffffff, 0x03ffffff, 0x07ffffff,
                   0x0fffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff };

int read_cdstime (FILE *filep, cdstime_t *objp)
/*
 *  reads a CCSDS Day Segmented time code from the specified file;
 *  returns the number of bytes read (8 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{
  unsigned char in_data[time_length];
  int  ndat;

  ndat = fread(in_data, 1, time_length, filep);
  if (ndat == time_length)
    {
      objp->dd      = in_data[0] * 256 + in_data[1];
      objp->msec    = in_data[2] * 16777216 + in_data[3] * 65536 + 
                      in_data[4] * 256 + in_data[5];
      objp->musec   = in_data[6] * 256 + in_data[7];
    }
  return(ndat);
}

int get_fgmconstants(int scid, double *hfcfrq, double *fgmfreq,             
                               double *spinph0, double *alpha,              
                               int *obdelay, int *ibdelay)                  
/*
 * read values from $FGMPATH"/const.fgm"                                    *
 * returns 0 on succes, -1 if file not found or read-error                  *
 ---------------------------------------------------------------------------*/

{ 
 int     i=0,sc=0;
 char    com[80]="";
 double  dv0,dv1,dv2,dv3;
 double  line[8][4];
 FILE    *tmpf;
 char    *sttri,tmpp[96];
 
 sc=scid; 
 tmpp[0] = '\0';
 if ((sttri=getenv("FGMPATH")) != NULL)
    strcpy(tmpp, getenv("FGMPATH"));
    else
    strcat(tmpp,".");
 strcat(tmpp, "/const.fgm");

 if ((tmpf = fopen(tmpp, "r"))==NULL){
   fprintf(stderr,"can't open file %s \n",tmpp);
   return(-1);
   } 
 while(!feof(tmpf) && i<8){
    fgets(com,80,tmpf);     
    if (sscanf(com,"%lf%lf%lf%lf",&dv0,&dv1,&dv2,&dv3) < 1)
      return(-1);
    line[i][0]=dv0;
    line[i][1]=dv1;   
    line[i][2]=dv2;
    line[i][3]=dv3;   
    i++;
    }    
  *hfcfrq=line[0][sc];  
  *fgmfreq=line[1][sc];  
  *alpha=line[2][sc];
  obdelay[0]=(int)line[3][sc];
  obdelay[1]=(int)line[4][sc];
  ibdelay[0]=(int)line[5][sc];
  ibdelay[1]=(int)line[6][sc];
  *spinph0= line[7][sc];   
//  fprintf(stdout,"%lf %lf %lf %d %lf \n",*hfcfrq,*fgmfreq,*alpha,*obdelay,*spinph0);
  if (tmpf != NULL)
    fclose(tmpf);
return (0);
}

int read_ddshead (FILE *filep, ddshead_t *objp)
/*
 *  reads a Data Disposition System header from the specified file;
 *  returns the number of bytes read (15 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{
  unsigned char in_data[header_length];
  int  ndat, i;

  ndat = fread(in_data, 1, header_length, filep);
  if (ndat == header_length)
    {
      objp->t.dd      = in_data[0] * 256 + in_data[1];
      objp->t.msec    = in_data[2] * 16777216 + in_data[3] * 65536 + 
                        in_data[4] * 256 + in_data[5];
      objp->t.musec   = in_data[6] * 256 + in_data[7];
      objp->source    = in_data[8];
      for (i = 0; i < 3; i++)
        objp->length[i] = in_data[9 + i];
      objp->ID        = in_data[12];
      objp->VC        = in_data[13];
      objp->TC        = in_data[14];
    }
  return(ndat);
}

int read_ddsheadc (headbytes in_data, ddshead_t *objp)
/*
 *  reads a Data Disposition System header from an array of bytes  ;
 *  returns the number of bytes read (15 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{ 
  int  ndat=15, i;

  objp->t.dd      = in_data[0] * 256 + in_data[1];
  objp->t.msec    = in_data[2] * 16777216 + in_data[3] * 65536 + 
                    in_data[4] * 256 + in_data[5];
  objp->t.musec   = in_data[6] * 256 + in_data[7];
  objp->source    = in_data[8];
  for (i = 0; i < 3; i++)
     objp->length[i] = in_data[9 + i];
  objp->ID        = in_data[12];
  objp->VC        = in_data[13];
  objp->TC        = in_data[14];
  return(ndat);
}


int write_ddshead (FILE *filep, ddshead_t *objp)
/*
 *  writes a Data Disposition System header to the specified file;
 *  returns the number of bytes written (15 on success).
 -----------------------------------------------------------------------------*/
{
 int            ndat=0;
 
 if (fwrite(&objp->t.dd,1,2,filep) == 2)
    ndat = 2;
 if (fwrite(&objp->t.msec,1,4,filep) == 4)
    ndat += 4;
 if (fwrite(&objp->t.musec,1,2,filep) == 2)
    ndat += 2;
 if (ndat == 8)
    ndat += fwrite(&objp->source,1,header_length-8,filep);
 else
    ndat = 0;

 return(ndat);
}

 
int read_auxdata (FILE *filep, auxdata_t *objp)
/*
 *  reads an FGM science auxiliary data block from the specified file;
 *  returns the number of bytes read (34 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{
  unsigned char in_data[auxiliary_length];
  int  ndat, i;

  ndat = fread(in_data,1,auxiliary_length,filep);
  if (ndat == auxiliary_length)
    {
      objp->status_word      = in_data[ 0] * 256 + in_data[ 1];
      objp->HFC_reset        = in_data[ 2] * 256 + in_data[ 3];
      objp->HFC_previous_sun = in_data[ 4] * 256 + in_data[ 5];
      objp->HFC_recent_sun   = in_data[ 6] * 256 + in_data[ 7];
      objp->HFC_primary      = in_data[ 8] * 256 + in_data[ 9];
      objp->HFC_secondary    = in_data[10] * 256 + in_data[11];
      objp->counts           = in_data[12] * 256 + in_data[13];
      for (i = 0; i < 10; i++)
        objp->variance[i]    = in_data[14 + 2 * i] * 256 + in_data[15 + 2 * i];
    }
  return(ndat);
}


int read_fgmvec (FILE *filep, fgmvec_t *objp)
/*
 *  reads an FGM telemetry vector from the specified file;
 *  returns the number of bytes read (16 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{
  int ndat;

  ndat = fread(objp,1,sizeof(fgmvec_t),filep);
  return(ndat);
}


int read_fgmtvec (FILE *filep, fgmtvec_t *objp)
/*
 *  reads a time-stamped FGM vector from the specified file;
 *  returns the number of bytes read (32 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/
{
  int ndat;

  ndat = fread(objp,1,sizeof(fgmtvec_t),filep);
  return(ndat);
}


int write_fgmtvec (FILE *filep, fgmtvec_t *objp)
/*
 *  writes a time-stamped FGM vector to the specified file;
 *  returns the number of bytes written (32 on success)
 -----------------------------------------------------------------------------*/
{
  int ndat;

  ndat = fwrite(objp,1,sizeof(fgmtvec_t),filep);
  return(ndat);
}

/*============================================================================*/

/* bitutil.c -- bit manipulation functions for the FGM data processing
 *
 * Copyright (C) 1994/95 TU Braunschweig, completed by Edita Georgescu 03/2000
 * written by Stephan Buchert (scb@geophys.nat.tu-bs.de)
 *            Reinhold Kempen (reinhold@geophys.nat.tu-bs.de)
 *            Joerg Warnecke  (joerg@geophys.nat.tu-bs.de)
 -----------------------------------------------------------------------------*/

void set_bit(int *arg, int n0)
/*----------------------------------------------------------------------------- 
 * Return arg with bit n0 set to 1                                            *
 -----------------------------------------------------------------------------*/
{
   *arg |= (1 << n0);
}

int get_bit(int arg, int n0)
/*----------------------------------------------------------------------------- 
 * Return bit n0 of arg                                                       *
 -----------------------------------------------------------------------------*/
{
  return ( (arg & (1 << n0)) >> n0 );
}

int check_bad(int status)
/*----------------------------------------------------------------------------- 
 * Return 1 if one of the bad bits [9,10,26,27] set, else 0                   *                                                    *
 -----------------------------------------------------------------------------*/
{
  int bad=0;
  
  bad = get_bit(status,13);
  bad |= get_bit(status,14);
  bad |= get_bit(status,15);
  return ( bad );
}

int mark_vector(int unmark, fgmtvec_t *fv)
/*----------------------------------------------------------------------------- 
 * The function 'mark_vector' modifies the status word of an output vector    *
 * {un}marks "BAD" bits (13,14,15); return for rngchng/cal_mode/ecl : 1/2/4   *
 -----------------------------------------------------------------------------*/
{
 int rce=0;
 rce=get_bit(fv->stat,(10-get_bit(fv->stat,7)))&& !get_bit(fv->stat,28); 
 if (!get_bit(unmark,0) && get_bit(fv->stat,26)) 
    set_bit(&fv->stat,13);
 if (!get_bit(unmark,1) && rce) 
    set_bit(&fv->stat,14);
 if (!get_bit(unmark,2) && get_bit(fv->stat,27)) 
    set_bit(&fv->stat,15);
 rce=rce*2+4*get_bit(fv->stat,27)+get_bit(fv->stat,26);
 return rce;
} 

int bitcpy (int arg0, int n0, int arg1, int n1)
/*----------------------------------------------------------------------------- 
 * Copy bit n0 of arg0 to bit n1 of arg1.                                     *
 -----------------------------------------------------------------------------*/
{
  arg1 &= (-1) ^ (1<<n1);
  return(arg1 | ((1 && (arg0 & (1<<n0))) << n1));
}


int subbits (int arg, int n0, int n1)
/*----------------------------------------------------------------------------- 
 * Return bits n0 to (excl.) n1 of arg right-aligned.                         *
 -----------------------------------------------------------------------------*/
{
  return(((arg & (mask[n1 - n0] << n0)) >> n0) & mask[n1 - n0]);
}

int bitcpy2 (int arg0, int arg1, int n0, int n1)
/*----------------------------------------------------------------------------- 
 * Copy n1-n0 lowest bits of arg0 to bits n0 to (excl.) n1 in arg1.           *
 -----------------------------------------------------------------------------*/
{
  arg1 &= (-1) ^ (mask[n1 - n0] << n0);
  return(arg1 | ((arg0 & mask[n1 - n0]) << n0));
}

/*============================================================================*/

/* utility.c -- utility functions for the FGM data processing
 * Modified by Edita Georgescu, MPE-Garching, 2000, V6.0
 * Version 4.3, 1996-05-15
 * Copyright (C) 1994/95/96 TU Braunschweig
 * written by Stephan Buchert (scb@geophys.nat.tu-bs.de)
 *            Reinhold Kempen (reinhold@geophys.nat.tu-bs.de)
 *            Joerg Warnecke  (joerg@geophys.nat.tu-bs.de)
#include struct.h
 -----------------------------------------------------------------------------*/
timesp cds2unix (cdstime_t cdst)
/*----------------------------------------------------------------------------- 
 * The function 'cds2unix' converts a time stamp given in CCSDS day segmented *
 * time code to the UNIX (POSIX) timespec.                                    *
 -----------------------------------------------------------------------------*/
{
  timesp res;

  res.tv_sec  = (cdst.dd - 4383) * 86400 + cdst.msec / 1000;    
  res.tv_nsec = ((cdst.msec % 1000) * 1000 + cdst.musec) * 1000;

  return res;
}

int timestr2unix ( char *string, timesp t0, timesp *t )
/*-----------------------------------------------------------------------------
 * converts a time string to the UNIX (POSIX) time specification 't',
 * returns the used format identifier (1,2,3), or 0 on error.
 * 
 * The following formats are allowed for the time string:
 *  '2000-09-01T12:00:04.012'   - format id 1 (ISO time string)
 *  'T12:00:04.012'             - format id 2 (ISO time string without day)
 *  '978420804.012'             - format id 3 (seconds of the UNIX epoch)
 *
 * If format 2 is used, the input value of t0 is used to compute the day.
 * With other formats, t0 is ignored.
 -----------------------------------------------------------------------------*/
{
 int          year, month, day, hour, minute, res, isec=0, nsec=0;
 long double  second;
 struct tm    lt;

 res = 0;

 if ( strchr(string,'T') == NULL )
   {
    if ( 1 == sscanf ( string, "%Lf", & second ) )
      {
       isec = (int) second;
       nsec = (int) 1.E9*(second - (long double) isec);
       res = 3;
      }
   }
 else
   {
    if ( (string[0] == 'T') && 
         (sscanf(string,"T%2d:%2d:%Lf",&hour,&minute,&second) == 3) )
      {
       isec = (int) second;
       nsec = (int) 1.E9*(second - (long double) isec);
/*     take day from t0 */
       isec += (60*hour + minute)*60 + t0.tv_sec - t0.tv_sec%86400;
       res = 2;
      }
    else if ( sscanf(string,"%4d-%2d-%2dT%2d:%2d:%Lf",
                     &year,&month,&day,&hour,&minute,&second) == 6 )
      {
       isec = (int) second;
       nsec = (int) 1.E9*(second - (long double) isec);

/* set the timezone to UTC0, because the data are given in UTC0 */
       putenv ( "TZ=UTC0" );
       tzset();                   /* sets timezone, daylight, and altzone */
       lt.tm_year  = year - 1900;
       lt.tm_mon   = month - 1;
       lt.tm_mday  = day;
       lt.tm_hour  = hour;
       lt.tm_min   = minute;
       lt.tm_sec   = isec;

       isec = (int) mktime(&lt);
       res = 1;
      }
   }
 if (res)
   {
/*  check for rounding errors */
  if (nsec >=NANOSEC )
      {
       nsec -= NANOSEC;
       isec += 1;
      }
    else if (nsec < 0)
      {
       nsec += NANOSEC;
       isec -= 1;
      }
    t->tv_sec  = isec;
    t->tv_nsec = nsec;
   }
 return(res);
}
void get_time_str (timesp *t, char *time_str)
/*----------------------------------------------------------------------------- 
 * The function 'get_time_str' converts a timesp time to an ISO standard  *
 * time string.                                                               *
 -----------------------------------------------------------------------------*/
{
  struct tm  ut;
  int        year, month, day, hour, min;
  double     sec;

  ut = *gmtime(&t->tv_sec);

  year = 1900 + ut.tm_year;
  month = 1 + ut.tm_mon;
  day = ut.tm_mday;
  hour = ut.tm_hour;
  min = ut.tm_min;

  sec = ((double) ut.tm_sec)*1.0 + ((double) t->tv_nsec)*1.e-9;
  
  sprintf ( time_str, "%d-%02d-%02dT%02d:%02d:%06.3fZ", 
	    year, month, day, hour, min, sec );
}


char *datestr(const time_t *clock)
/* ---------------------------------------------------------------------------*
 * The function 'datestr' converts the first member of a timesp               *  
 * time to a date=yyyymmdd string                                             *
 -----------------------------------------------------------------------------*/
{
  struct tm   ut;
  int         year, month, day;
  static char timestr[9]="";
  time_t toffs=6;
  const time_t *newclock;
  
  toffs=toffs+*clock;
  newclock=&toffs;
  ut = *gmtime(newclock);
  year = 1900 + ut.tm_year;
  month = 1 + ut.tm_mon;
  day = ut.tm_mday;
  sprintf ( timestr, "%4d%02d%02d", year, month, day);
  return(timestr);
}

int cmptime (timesp *x0, timesp *x1)
/*----------------------------------------------------------------------------- 
 * The function 'cmptime' compares two timespecs. It returns the difference   *
 * *x0 - *x1 in seconds. If the seconds are the same, it will return the      *
 * difference in nanoseconds. If the times are exactly the same, it will      *
 * return zero.                                                               *
 -----------------------------------------------------------------------------*/
{
  if (x0->tv_sec != x1->tv_sec)
    return(x0->tv_sec - x1->tv_sec);
  else
    return(x0->tv_nsec - x1->tv_nsec);
}


void inctime (timesp *tp, int nsec)
/*----------------------------------------------------------------------------- 
 * adds to a timespec a time interval in  nanoseconds.                        *
 -----------------------------------------------------------------------------*/
 {
  nsec +=tp->tv_nsec;
  if (nsec >= 0){
  tp->tv_nsec = nsec % NANOSEC;
  tp->tv_sec += nsec / NANOSEC;
  }
  else{
    nsec += NANOSEC;
  tp->tv_nsec = nsec;
  tp->tv_sec -= NANOSEC;
  }
 }


timesp subtime (timesp *a, timesp *b)
/*----------------------------------------------------------------------------- 
 * The function 'subtime' returns the result of the subtraction *a - *b.      *
 -----------------------------------------------------------------------------*/
{
  timesp res;

  res.tv_sec = a->tv_sec - b->tv_sec;
  if ((res.tv_nsec = a->tv_nsec - b->tv_nsec) < 0)
    {
      res.tv_sec--;
      res.tv_nsec += NANOSEC;
    }
  return res;
}


int pktlen (ddshead_t *p)
/*----------------------------------------------------------------------------- 
 * The function 'pktlen' returns the length of the science data packet (incl. *
 * the science auxiliary data) from the ESOC header.                          *
 -----------------------------------------------------------------------------*/
{
  return (p->length[0] * 65536 + p->length[1] * 256 + p->length[2]);
}


int scid (unsigned char ID)
/*----------------------------------------------------------------------------- 
 * The function 'scid' returns the spacecraft ID from the ESOC header ID. The *
 * result is in the range 0-3, ie S/C 1 = 0, S/C 2 = 1, S/C 3 = 2, S/C 4 = 3. *
 -----------------------------------------------------------------------------*/
{
  int arg = ID;

  return (subbits(arg, 4, 8) - 1);
}


int gsid (unsigned char ID)
/*----------------------------------------------------------------------------- 
 * The function 'gsid' returns the ground station ID from the ESOC header ID. *
 * 1 = Odenwald, 2 = Redu, 3 = Kourou, 4 = Perth, 5 = Malindi, 6 = Canberra,  *
 * 7 = Goldstone, 15 = N/A                                                    *
 -----------------------------------------------------------------------------*/
{
  int arg = ID;

  return (subbits(arg, 0, 4));
}


int timecal (unsigned char TC)
/*----------------------------------------------------------------------------- 
 * The function 'timecal' returns the time calibration indicator from the     *
 * ESOC header TC byte.                                                       *
 * 0 = actual time, 1 = extrapolated time, 2 = contingency time               *
 -----------------------------------------------------------------------------*/
{
  int arg = TC;

  return (subbits(arg, 4, 8));
}


int TMacqmode (unsigned char TC)
/*----------------------------------------------------------------------------- 
 * The function 'TMacqmode' returns the TM acquisition mode from the ESOC     *
 * header TC byte.                                                            *
 * It represents the current TM acquisition mode as extracted from the CAB.   *
 -----------------------------------------------------------------------------*/
{
  int arg = TC;

  return (subbits(arg, 0, 4));
}


void fgmrate (int numsens, int telmode, double *sampfreq, 
                         int *declevel, double fgmfrq)
/*----------------------------------------------------------------------------- 
 * returns the sampling frequency "sampfreq" and the decimation level         *
 * "declevel" of the FGM for sensor "numsens" (0,1) in telemetry mode         *
 * "telmode" (0-F).                                                           *
 *                                                                            *
 * on error, sampfreq = 0.0 and declevel = 0 are returned.                    *
 -----------------------------------------------------------------------------*/
{
 int    ndr[2][16] = { {0,0, 13,11, 9,0,0,0,0,0, 13,11, 9, 3,0,0},
                       {0,0,185,29,67,0,0,0,0,0,185,29,67,26,0,0}  };
 
 *sampfreq = 0.0;
 *declevel = 0;

 if ( (numsens >= 0) && (numsens < 2) )
   {
    if ( (telmode >= 0) && (telmode < 16) )
      {
       *declevel = ndr[numsens][telmode];
       if ( *declevel > 0 ) 
          *sampfreq = fgmfrq / ((double) *declevel);
      }
    if ( *declevel <= 0) 
       fprintf ( stderr, "WARNING: illegal telemetry mode %x\n", telmode );
   }

 else
    fprintf ( stderr, "WARNING: illegal sensor number %d\n", numsens );
}


double mjd2000 ( timesp t )
/* ----------------------------------------------------------------------------
 * returns the Modified Julian Day (referred to 2000-01-01, 0 UT) for         *
 * a given UNIX (POSIX) timespec `t'.                                         *
 * ---------------------------------------------------------------------------*/
{
 double         mjd0=-10957.0;    /* MJD2000 of 1970-01-01, 0 UT */
 long double    second;

 second = (long double) t.tv_sec + 1.E-9*(long double) t.tv_nsec;

 return ( mjd0 + second/86400.0 );
}

/*============================================================================*/
