/* LIBRARY module includes constants, user defined types & function prototypes
 * struct + bitutil + utility , modified by eg
 * Version 1.0, 1999-12-17
 * Copyright (C) 1999 MPE Garching
 * Author : Edita Georgescu (eg@mpe.mpg.de)
 * V1.1 (17-05-2000): added & modified functions: get_bit, set_bit, check_bad, 
 * datestr,get_fgmconstants, inctime, timestr2unix, fgmrate, read_ddsheadc
 * V1.2 (15-09-2000): new const.fgm file format (changed: get_fgmconstants)      
 * V1.3 (15-12-2000): added new function: mark_vector
 =============================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/param.h>

#ifndef _FGM_STRUCT_H
#define	_FGM_STRUCT_H

#define time_length          8        /*  - length of CCSDS-CDF time format  */
#define header_length       15        /*  - length of ESOC frame header      */
#define auxiliary_length    34        /*  - length of science auxiliary data */
#define NANOSEC     1000000000
#define block_sec            5        /*  - period of one telemetry frame    */
#define block_nsec   152221680        /*    given in sec and nanosec         */

#ifdef __cplusplus
extern "C" {
#endif

typedef double matrix[48][5];
typedef unsigned char headbytes[15];
/*typedef timespec_t timesp;  SUN <-> AlphaDEC */
typedef struct timesp
{
    time_t  tv_sec;
    int     tv_nsec;
}timesp; 

typedef struct cdstime      /* CCSDS day segmented time code */
  {
   unsigned short dd;              /* days since 1.1.1958 */
   unsigned int   msec;            /* milliseconds of the day */
   unsigned short musec;           /* microseconds of the milliseconds */
  }
cdstime_t;

typedef struct ddshead      /* data disposition system header */
  {
   cdstime_t      t;               /* packet time stamp */
   unsigned char  source;          /* data source / type ID */
   unsigned char  length[3];       /* packet length in bytes */
   unsigned char  ID;              /* bits 0-3: spacecraft ID    
                                      bits 4-7: ground station ID */
   unsigned char  VC;              /* data stream (virtuell channel) */
   unsigned char  TC;              /* bits 0-3: time calibration
                                      bits 4-7: telemetry acquisition mode */
  }
ddshead_t;

typedef struct auxdata      /* science auxiliary data block */
  {
   unsigned short status_word;     /* status word */
   unsigned short HFC_reset;       /* time of reset pulse */
   unsigned short HFC_previous_sun;/* time of previous sun reference pulse */
   unsigned short HFC_recent_sun;  /* time of most recent sun reference pulse */
   unsigned short HFC_primary;     /* time of 1st primary vector */
   unsigned short HFC_secondary;   /* time of 1st secondary vector */
   unsigned short counts;          /* number of resets */
   unsigned short variance[10];    /* variance of magnetic field */
  }
auxdata_t;

typedef struct fgmvec       /* FGM vector in telemetry data */
  {
   int rng;                       /* range */
   int v[3];                      /* measured digits */
  }
fgmvec_t;

typedef struct fgmtvec      /* time-stamped FGM vector */
  {
   int         stat;              /* status and id word */
   timesp      tv;                /* time value */
   float       b[3];              /* magnetic field */
   float       phavar;            /* spin phase or normalized 
                                     magnetic variance: total*/
   float       magvar;            /* raw data variance or normalized
                                     magnetic variance: magnitude */
  }
fgmtvec_t;

int read_cdstime (FILE *filep, cdstime_t *objp);
/*
 *  reads a CCSDS Day Segmented time code from the specified file;
 *  returns the number of bytes read (8 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int read_ddshead (FILE *filep, ddshead_t *objp);
/*
 *  reads a Data Disposition System header from the specified file;
 *  returns the number of bytes read (15 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int read_ddsheadc (headbytes indata, ddshead_t *objp);
/*
 *  reads a Data Disposition System header from the specified file;
 *  returns the number of bytes read (15 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int write_ddshead (FILE *filep, ddshead_t *objp);
/*
 *  writes a Data Disposition System header to the specified file;
 *  returns the number of bytes written (15 on success).
 -----------------------------------------------------------------------------*/

int read_auxdata (FILE *filep, auxdata_t *objp);
/*
 *  reads an FGM science auxiliary data block from the specified file;
 *  returns the number of bytes read (34 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int read_fgmvec (FILE *filep, fgmvec_t *objp);
/*
 *  reads an FGM telemetry vector from the specified file;
 *  returns the number of bytes read (16 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int read_fgmtvec (FILE *filep, fgmtvec_t *objp);
/*
 *  reads a time-stamped FGM vector from the specified file;
 *  returns the number of bytes read (28 on success, 0 on EOF)
 -----------------------------------------------------------------------------*/

int write_fgmtvec (FILE *filep, fgmtvec_t *objp);
/*
 *  writes a time-stamped FGM vector to the specified file;
 *  returns the number of bytes written (28 on success)
 -----------------------------------------------------------------------------*/

//----------------------------------------------------------------------------
int get_fgmconstants(int,double *,double *,double*, double*,int*,int* );
 
// reads values from 'const.fgm' 
// returns 0 on succes, -1 if file not found or read-error
 

#ifdef __cplusplus
}
#endif

#endif /* !_FGM_STRUCT_H */

/*============================================================================*/

/* bitutil.h -- definitions for bit manipulations for the FGM data processing
 *
 * Copyright (C) 1994/95 TU Braunschweig
 * written by Stephan Buchert (scb@geophys.nat.tu-bs.de)
 *            Reinhold Kempen (reinhold@geophys.nat.tu-bs.de)
 *            Joerg Warnecke  (joerg@geophys.nat.tu-bs.de)
 -----------------------------------------------------------------------------*/

#ifndef _FGM_BITUTIL_H_
#define _FGM_BITUTIL_H_


#ifdef __cplusplus
extern "C" {
#endif

extern int get_bit(int arg, int n0);
/*----------------------------------------------------------------------------- 
 * Return bit n0 of arg                                                       *
 -----------------------------------------------------------------------------*/

extern void set_bit(int *, int); 
/*-----------------------------------------------------------------------------
 * Return arg with bit n0 set to 1                                            *
 -----------------------------------------------------------------------------*/

extern int mark_vector(int unmark, fgmtvec_t *fv);
/*----------------------------------------------------------------------------- 
 * The function 'mark_vector' modifies the status word of an output vector    *
 * {un}marks "BAD" bits (13,14,15); return for rngchng/cal_mode/ecl : 1/2/4   *
 -----------------------------------------------------------------------------*/

extern int check_bad(int status);
/*----------------------------------------------------------------------------- 
 * Return 1 if bad bits set, else 0                                           *
 -----------------------------------------------------------------------------*/

int bitcpy(int arg0, int n0, int arg1, int n1);
/*-----------------------------------------------------------------------------
 * Copy bit n0 of arg0 to bit n1 of arg1.                                     *
 -----------------------------------------------------------------------------*/

int subbits (int arg, int n0, int n1);
/*-----------------------------------------------------------------------------
 * Return bits n0 to (excl.) n1 of arg right-aligned.                         *
 -----------------------------------------------------------------------------*/

int bitcpy2 (int arg0, int arg1, int n0, int n1);
/*-----------------------------------------------------------------------------
 * Copy n1-n0 lowest bits of arg0 to bits n0 to (excl.) n1 of arg1.           *
 -----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif /* !_FGM_BITUTIL_H */


/*============================================================================*/

/* utility.h -- definitions for utilities for the FGM data processing
 *
 * Version 4.0, 1996-01-24
 *
 * Copyright (C) 1994/95/96 TU Braunschweig
 * written by Stephan Buchert (scb@geophys.nat.tu-bs.de)
 *            Reinhold Kempen (reinhold@geophys.nat.tu-bs.de)
 *            Joerg Warnecke  (joerg@geophys.nat.tu-bs.de)
 -----------------------------------------------------------------------------*/

#ifndef _FGM_UTILITY_H
#define _FGM_UTILITY_H

#ifdef __cplusplus
extern "C" {
#endif

timesp cds2unix (cdstime_t cdst);
/*----------------------------------------------------------------------------- 
 * The function 'cds2unix' converts a time stamp given in CCSDS day segmented *
 * time code to the UNIX (POSIX) timespec.                                    *
 -----------------------------------------------------------------------------*/

int timestr2unix ( char *string, timesp t0, timesp *t );
/*-----------------------------------------------------------------------------
 * converts a time string to the UNIX (POSIX) time specification 't',         *
 * returns the used format identifier (1,2,3), or 0 on error.                 *
 *                                                                            * 
 * The following formats are allowed for the time string:                     *
 *  '1994-09-01T12:00:04.012'   - format id 1 (ISO time string)               *
 *  'T12:00:04.012'             - format id 2 (ISO time string without day)   *
 *  '778420804.012'             - format id 3 (seconds of the UNIX epoch)     *
 *                                                                            *
 * If format 2 is used, the input value of t0 is used to compute the day.     *
 * With other formats, t0 is ignored.                                         *
 -----------------------------------------------------------------------------*/

void get_time_str (timesp *t, char *time_str);
/*----------------------------------------------------------------------------- 
 * The function 'get_time_str' converts a timesp time to an ISO standard      *
 * time string.                                                               *
 -----------------------------------------------------------------------------*/

 char *datestr(const time_t *clock);
/*----------------------------------------------------------------------------- 
 * Return the date as "yyyymmdd" for a time in seconds                        *
 * input: the first member of a timespec structure (time in seconds)          *
 -----------------------------------------------------------------------------*/

int cmptime (timesp *x0, timesp *x1);
/*----------------------------------------------------------------------------- 
 * The function 'cmptime' compares two timespecs. It returns the difference   *
 * *x0 - *x1 in seconds. If the seconds are the same, it will return the      *
 * difference in nanoseconds. If the times are exactly the same, it will      *
 * return zero.                                                               *
 -----------------------------------------------------------------------------*/

void inctime (timesp *tp, int nsec);
/*----------------------------------------------------------------------------- 
 * increments a timespec by a number of nanoseconds.
 -----------------------------------------------------------------------------*/

timesp subtime (timesp *a, timesp *b);
/*----------------------------------------------------------------------------- 
 * The function 'subtime' returns the result of the subtraction *a - *b.      *
 -----------------------------------------------------------------------------*/

int pktlen (ddshead_t *p);
/*----------------------------------------------------------------------------- 
 * The function 'pktlen' returns the length of the science data packet (incl. *
 * the science auxiliary data) from the ESOC header.                          *
 -----------------------------------------------------------------------------*/

int scid (unsigned char ID);
/*----------------------------------------------------------------------------- 
 * The function 'scid' returns the spacecraft ID from the ESOC header ID. The *
 * result is in the range 0-3, ie S/C 1 = 0, S/C 2 = 1, S/C 3 = 2, S/C 4 = 3. *
 -----------------------------------------------------------------------------*/

int gsid (unsigned char ID);
/*----------------------------------------------------------------------------- 
 * The function 'gsid' returns the ground station ID from the ESOC header ID. *
 * 1 = Odenwald, 2 = Redu, 3 = Kourou, 4 = Perth, 5 = Malindi, 6 = Canberra,  *
 * 7 = Goldstone, 15 = N/A                                                    *
 -----------------------------------------------------------------------------*/

int timecal (unsigned char TC);
/*----------------------------------------------------------------------------- 
 * The function 'timecal' returns the time calibration indicator from the     *
 * ESOC header TC byte.                                                       *
 * 0 = actual time, 1 = extrapolated time, 2 = contingency time               *
 -----------------------------------------------------------------------------*/

int TMacqmode (unsigned char TC);
/*----------------------------------------------------------------------------- 
 * The function 'TMacqmode' returns the TM acquisition mode from the ESOC     *
 * header TC byte.                                                            *
 * It represents the current TM acquisition mode as extracted from the CAB.   *
 -----------------------------------------------------------------------------*/


void fgmrate (int , int , double *, int *, double );
/*----------------------------------------------------------------------------- 
 * returns the sampling frequency "sampfreq" and the decimation level         *
 * "declevel" of the FGM for sensor "numsens" (0,1) in telemetry mode         *
 * "telmode" (0-F).                                                           *
 *                                                                            *
 * on error, sampfreq = 0.0 and declevel = 0 are returned.                    *
 -----------------------------------------------------------------------------*/

double mjd2000 ( timesp t );
/* ----------------------------------------------------------------------------
 * returns the Modified Julian Day (referred to 2000-01-01, 0 UT) for         *
 * a given UNIX (POSIX) timespec `t'.                                         *
 * ---------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif /* !_FGM_UTILITY_H */

/*============================================================================*/

